/* BEGIN JPSS HEADER */
/*****************************************************************************
*
* Peraton, Inc.  All rights reserved.
*
* GOVERNMENT RIGHTS:
* Per contract 1332KP23CNEEJ0001, CAR 1352.227-70 applies giving full
* copyrights to data, including reports and other copyrighted materials
* to the United States Government.
*
* THIRD PARTY RIGHTS:
* Note: Third party entities may own copyrights in this work.
*
* EXPORT CONTROL:
* Unless otherwise specified beneath this header, this item does not contain
* Technology or Technical Data controlled under either the U.S. International
* Traffic in Arms Regulations or the U.S. Export Administration Regulations.
*
* CLASSIFICATION: B
* Refer to Software Standards and Practices Manual (SSPM) for coding and
* documentation standards.
*
*****************************************************************************/
/* END JPSS HEADER */

/*****************************************************************************
*
*  NAME: Spectra.h
*
*  DESCRIPTION:   See description preceeding declaration below
*
*****************************************************************************/

/***********************************************************************
*  HISTORY:
*  
*  DATE       PR#     AUTHOR        Build  DESCRIPTION
*  ---------  ------  ------------  -----  ------------------------------------
*  25APR2005          B. Henderson  1.3    Initial Version
*                     D. Elliott
*                     T. Hahn
*  01MAR2006          D. Elliott    1.4    Fringe Count Error
*                     T. Hahn              Follow On Drop
*  02AUG2006  011858  D. Elliott    1.4    Modified ITT header with
*                                          permission from Steven G.
*                                          Minnie from ITT Space
*                                          Systems, LLC
*  14DEC2007  015906  K. Boswell    1.5x1  ECR A-103, EDRPR 1.8 CP3 Updates
*                                          changed updateCrisSdrGeolocation's
*                                          parameter to be a pointer to the
*                                          entire algorithm structure
*  07JAN2008  015748  B. Henderson  1.5x1  ISTN_CRIS_SDR_NGST_1.1
*                                          drop implementation;
*                                          ref. NP-EMD-2007.510.0027 RevB
*  12JUN2008  017910  K. Boswell    1.5x1  Made theScienceDataProcessor_
*                                          a smart pointer.
*  25OCT2008          L. Wang    ALG01342  Implem with ITT update 2.14
*  20JAN2009  019301  K. Boswell    1.5x1  ISTN_CRIS_SDR_UNIX_NGST_2.1
*                                          drop implementation
*                                          per TM NP-EMD.2008.510.0055
*  21AUG2009  020632  R. Ferrel     SenCh  Added setImaginary
*             020861  R. Ferrel     SenCh
*  14JUL2009          L. Wang     ALG1443  Update for nonlinearity correction
*  03FEB2010  021718  K. Boswell    SenCh  ISTN_CRIS_SDR_UNIX_NGST_2.2
*                                          drop implementation
*                                          per TM NP-EMD.2009.510.0046
*  19MAY2010  023135  K. Boswell    SenCh  ISTN_CRIS_SDR_UNIX_NGST_2.2.1
*                                          drop implementation per TM
*                                          NP-EMD.2010.510.0006 and
*                                          ISTN_CRIS_SDR_UNIX_NGST_2.2.2
*                                          per TM NP-EMD.2010.510.0006 Rev A
*                                          and ECR A-275
*  06JUN2013  034187  T. Gardner    Mx8.0  LaserWavelength is updated only
*             034188                       when CMO is updated and saved
*  05NOV2013  035224  S. Joo        Bk2.0  Fixed compiler warnings for
*                                          Spectra.cpp
*  20MAY2013  038279  R. Neely      Blk2.0 Clone of PCR035945.  Changed type
*                                          of forTimeStampBias to Int32 in 
*                                          declaration of the method 
*                                          updateCrisSdrGeolocation.
* 14OCT2014   042555  SEClark   Blk2.0.0.4 CrIS SSPM Compliance. 
* 25JUN2014           Y. Chen/Y.Han Mx8.5  Add variable calibrationOrder 
*                                          and overload functions
* 03JAN2015    41268  J. Eberle    Blk2.0  Header Changes
* 11MAY2015    48586  T. Gardner   Blk2.0  Integrate TR/FR updates into
*                                          block 2.0
* 15May2015           L. Wang   V2.0_beta  Fringe count error correction
*                                          implementation
* 01JUL2015           L. Wang   V2.0_beta  Updates for FCE handling
* 04APR2016           Y. Chen     Blk2.0   Integrate TR/FR updates into
*                                          block 2.0 
* 24JUL2017           H. Xu/Y. Chen Blk2.0 Added parameter coefficients to
*                                          Spectra 
* 31AUG2017    63167  J. Gibbs    Blk2.1   fixed compile warnings
* 26APR2018    65628  G. Pachl    Blk2.1   Merged into Baseline CCR 3542.
***********************************************************************/

/******************************************************************************* 
* This document contains information proprietary and confidential to ITT Space 
* Systems, LLC ("ITT"), or a third party for whom ITT may have a legal 
* obligation to protect such information from unauthorized disclosure, use or
* duplication. Any disclosure, use, duplication or diversion contrary
* to U.S. Law of this document or of any of the information contained herein 
* for other than the
* specific purpose for which it was disclosed is expressly prohibited, 
* except as ITT has otherwise
* agreed in writing or as the U.S. Government has authorized. All copies of 
* this document are the
* sole property of ITT and will be returned promptly upon request
******************************************************************************/


#ifndef _INC_SPECTRA_3F2F9E6E028D_INCLUDED
#define _INC_SPECTRA_3F2F9E6E028D_INCLUDED

#include <ProSdrCrisGbl.h>
#include <ProSdrCrisStruct.h>
#include <CcsdsSecondaryHeader.h>
#include <TelemetryProcessor.h>
#include <Interferogram.h>
#include <SweepDirection.h>
#include <RawDataRecordStatusRegister.h>
#include <ScienceDataRecordStatus.h>
#include <Band.h>
#include <CrIS_SDR.h>
#include <string>
#include <InfUtil_TimApi.h>
#include <LibraryReplacements.h>
#include<SpectralResDefs.h>
#include <cstdlib>

class ScienceDataProcessor;

#define ROUND(num)  ((num >= 0) ? floor(num+0.5) : ceil(num-0.5))

class Spectra
{
 public:
   enum FCE_LIMIT { REF_THRESHOLD, CAL_THRESHOLD, NUM_THRESHOLD_TYPES};

   static const Int32 DAYS_TO_MICROSECS_SHIFT_CONV;
   static const Int32 MILISECS_TO_MICROSECS_SHIFT_CONV;
   static const Float32 MICRO_TO_MILI_UNIT_CONV;

   typedef struct
   {
      Float64        ampThreshRejectLimit[NUM_THRESHOLD_TYPES];
      Int32          defaultDetectorBand;
      Int32          defaultDetectorFOV;
      Float64        dimensionThresholdLimit;
      Float64        fractionalFceThresholdLimit;
      Float64        goodLinearFittingThreshLimit;
      Float64        maxFceThreshLimit;
      Int32          maxIndex;
      Int32          minIndex;

      Float64        esImagThreshold;
      UInt32         esFceNumErrorThreshold;
      UInt32         startWavenumber;
      UInt32         endWavenumber;

   }FCEParam;
   static Float64 LASER_WAVELENGTH[Band::TOTAL_BANDS];
   static bool usingDefaultLW;
   static Float64 metrologyWavelengthShift[Band::TOTAL_BANDS];
   static Float64 RESAMPLING_WAVELENGTH[Band::TOTAL_BANDS];
   static bool INVALID_NEON_CAL;
   static Float64 DESIRED_BAND_CENTER[Band::TOTAL_BANDS];
   static UInt32 IMPULSE_NOISE_COUNT_THRESHOLD;
   static UInt32 NUMBER_OPD_OVERSCAN_SAMPLES;
   inline Spectra();
   Spectra(SceneElement fovInit, Band::Wavelength bandInit, 
           FieldOfRegard forInit, SweepDirection psdInit);
   // let spectra recieve PCT coefficients
   Spectra(const Interferogram& interferogram, UInt32 theFoldIdx, 
           Float64 theFirstW, Float64 deltaSigma, 
	   CalibrationOrder theOrder, CrisSdrCoefficients* cfgParmsPtr = NULL, bool removeAlias = true);
   Spectra(const Spectra& source);
   enum Method 
      {
         RAW,
         QUITE
      };
   const BOOST::vector<Float64>& getMagnitude() const;
   inline  const BOOST::vector<Float64>& getReal() const
      {
         return realWavenumberBin;
      }
   inline  BOOST::vector<Float64>& bindReal() {return realWavenumberBin;}
   inline  BOOST::vector <Float64>& bindImaginary() 
      {return imaginaryWavenumberBin;}
   inline  BOOST::vector <Float64>& bindNEdN() {return NEdN_Estimate;}
   inline const BOOST::vector<Float64>& getImaginary() const
      {
         return imaginaryWavenumberBin;
      }
   inline BOOST::vector<Float64>& get_interferogram_DS_reals()
      {
         return interferogram_DS_reals;
      }
   inline BOOST::vector<Float64>& get_interferogram_DS_imags()
      {
         return interferogram_DS_imags;
      }
   inline const BOOST::vector<Float64>& getNEdN() const {return NEdN_Estimate;}
   inline void setComplex(UInt32 bin, Float64 real, Float64 imaginary) 
      {realWavenumberBin[bin] = real; imaginaryWavenumberBin[bin] = imaginary;}
   inline void setImaginary(UInt32 bin, Float64 imaginaryValue) 
      {imaginaryWavenumberBin[bin] = imaginaryValue;}
   inline void setLosElevationAngle(Float64 theLosElevationAngle) 
      {SDR_Status.losElevationAngle = theLosElevationAngle;}
   inline void setLosRollAngle(Float64 theLosRollAngle) 
      {SDR_Status.losRollAngle = theLosRollAngle;}
   inline void setLosPitchAngle(Float64 theLosPitchAngle) 
      {SDR_Status.losPitchAngle = theLosPitchAngle;}
   inline void setLosYawAngle(Float64 theLosYawAngle) 
      {SDR_Status.losYawAngle = theLosYawAngle;}
   inline void setSateliteZenithAngle(Float64 theSateliteZenithAngle) 
      {SDR_Status.sateliteZenithAngle = theSateliteZenithAngle;}
   inline void setSateliteAzimuthAngle(Float64 theSateliteAzimuthAngle) 
      {SDR_Status.sateliteAzimuthAngle = theSateliteAzimuthAngle;}
   inline void setSunZenithAngle(Float64 theSunZenithAngle) 
      {SDR_Status.sunZenithAngle = theSunZenithAngle;}
   inline void setSunAzimuthAngle(Float64 theSunAzimuthAngle) 
      {SDR_Status.sunAzimuthAngle = theSunAzimuthAngle;}
   inline void setSateliteAltitude(Float64 theSateliteAltitude) 
      {SDR_Status.sateliteAltitude = theSateliteAltitude;}
   inline void setRange(Float64 theRange) {SDR_Status.range = theRange;}
   inline void setLongitude(Float64 theLongitude) 
      {SDR_Status.longitude = theLongitude;}
   inline void setLatitude(Float64 theLatitude) 
      {SDR_Status.latitude = theLatitude;}
   inline void setMajorXAxisFootprint(Float32 theMajorXAxisFootprint) 
      {SDR_Status.majorXAxisFootprint = theMajorXAxisFootprint;}
   inline void setMinorXAxisFootprint(Float32 theMinorXAxisFootprint) 
      {SDR_Status.minorXAxisFootprint = theMinorXAxisFootprint;}
   inline void setSpectralResamplingLaserWavelength(
               Float64 theSpectralResamplingLaserWavelength) 
      {SDR_Status.spectralResamplingLaserWavelength = 
         theSpectralResamplingLaserWavelength;}
   inline void setMonitoredLaserWavelength(Float64 theMonitoredLaserWavelength) 
      {SDR_Status.monitoredLaserWavelength = theMonitoredLaserWavelength;}
   inline void setSdrFringeCount(Int32 theSdrFringeCount) 
      {SDR_Status.sdrFringeCount = theSdrFringeCount;}
   void generateUSN(Spectra* rawSum, Spectra* squaredSum, Int32 windowSize);

   inline void setIctTempStab(Float32 ictTempStab)
      {SDR_Status.ictTemperatureStability = ictTempStab;} 

   void setSize(UInt32 newSize);
   void clear();
   virtual void operator+=(const Spectra& rightOperand);
   virtual void operator-=(const Spectra& rightOperand);
   virtual void operator/=(const Spectra& divisor);
   virtual void operator/=(const Float64 factor);
   virtual Spectra& operator=(const Spectra& source);
   inline virtual Float64* operator[](UInt32 bin) 
      { return &realWavenumberBin[bin];}
   void addSquare(const Spectra& source, bool complex = true);
   void removeSquare(const Spectra& source, bool complex = true);
   virtual void operator*=(const Float64 factor);
   virtual void operator*=(const Spectra& factor);
   virtual bool calibrate(const Spectra* hotReference, 
                          const Spectra* coldReference, 
                          const Spectra* hotTargetTemp, 
                          const Spectra* coldTargetTemp, 
                          const BOOST::vector<Float64>& theFirGainRealBin, 
                          const BOOST::matrix<Float64>* calibrationMatrix) = 0;
   virtual bool calibrate(const Spectra* hotReference, 
                          const Spectra* coldReference, 
                          const Spectra* hotTargetTemp, 
                          const Spectra* coldTargetTemp, 
                          const BOOST::matrix<Float64>* calibrationMatrix) = 0; 
   virtual bool correct(const Spectra* correctionSpectra) = 0;
   std::string save() const;
   inline Int32 getChannel() const { return channel;}
   virtual BOOST::vector<std::complex<Float64> >& correctFringeCountError(
                        BOOST::vector<std::complex<Float64> >& linearPhaseShift,
                        Float64 fringeCount) = 0;
   virtual void adjustPhase(
             const BOOST::vector<std::complex<Float64> >& linearPhaseShift) = 0;
   inline bool isValid() const { return(!SDR_Status.invalidData); }
   inline Float64 getFirstWavenumber() const { return firstWavenumber;}
   inline Float64 getWavenumberSpacing() const { return wavenumberBinSize;}
   inline void setFirstWavenumber(Float64 theFirstWavenumber) 
      {firstWavenumber = theFirstWavenumber;}
   inline void setWavenumberSpacing(Float64 theWavenumberBinSize) 
      {wavenumberBinSize = theWavenumberBinSize;}
   Float64 getResponse(Int32 wavenumber) const;

// FCE update
   virtual Float64 detectReferenceSpectraFringeCountError(
                      const Spectra* hotReference, 
                      const Spectra* coldReference) = 0;
   virtual Int32 detectESFringeCountError(
                      const Spectra* hotReference, 
                      const Spectra* coldReference, 
                      const Spectra* rawEarthSpectra,
                      const Spectra* hotTargetTemp, 
                      const Spectra* coldTargetTemp) = 0;

   virtual bool estimateNEdN(const Spectra* theSlope, 
                             const Spectra* theIntercept, 
                             UInt32 userNumberOfPoints, 
                             Float64 userMinWavenumber, 
                             Float64 userDeltaSigma) = 0;
   inline const SweepDirection& getSweepDirection() const 
      { return sweepDirection;}
   inline const FieldOfRegard& getFieldOfRegard() const { return fieldOfRegard;}
   inline SceneElement getFieldOfView() const { return fieldOfView;}
   inline void setFieldOfView(const SceneElement fovID) { fieldOfView = fovID;}
   inline Band::Wavelength getBand() const { return spectralBand; }
   inline void setBand(Band::Wavelength newBand) { spectralBand = newBand;}
   inline void attach() { userLocks++;}
   void detach();
   inline UInt32 getFoldIndex() const { return foldIndex;}
   inline void setFoldIndex(UInt32 theFoldIndex) {foldIndex = theFoldIndex;}
   inline const std::complex<Float64> getComplex(Int32 bin) const;
   const BOOST::vector <std::complex <Float64> >  & getComplex();
   virtual bool resample(Float64 startWavenumber, Float64 stopWavenumber, 
                         Float64 binSize);
   bool clipGuardBands(Float64 startWavenumber, Float64 stopWavenumber);
   inline const SYSTEMTIME& getFrameTime() const { return RDR_Status.frameTime;}
   inline void setFrameTime(const SYSTEMTIME& newTime) 
      { RDR_Status.frameTime = newTime;}
   inline UInt16 getRDR_FrameType() const { return RDR_Status.frameType; }
   inline UInt16 getRDR_FlightVersion() const 
      { return RDR_Status.flightSoftwareVersion;}
   inline UInt8 getRDR_SensorID() const { return RDR_Status.sensorID;}
   inline char getRDR_ImpulseNoiseCount() const 
      { return RDR_Status.impulseNoiseCount;}
   inline bool getRDR_BitTrimFail() const { return RDR_Status.bitTrimFailure; }
   inline bool getRDR_InvalidData() const { return RDR_Status.invalidData; }
   inline bool getRDR_FringeCountError() const 
      { return RDR_Status.fringeCountError; }
   inline Int32 getRDR_ZPDAmplitude() const 
      { return RDR_Status.zpdAmplitude;}
   inline UInt32 getRDR_ZPDLocation() const
      { return RDR_Status.zpdLocation;}
   inline UInt16 getSDR_FrameType() const 
      { return RDR_Status.frameType ^ 0x400; }

   /**
    * Returns the state of the flag which indicates the RDR packet is missing.
    *
    * @return true if the interferogram is "missing", false if not.
    */
   bool getRDR_MissingData() const;

   inline bool getSDR_InvalidData() const { return SDR_Status.invalidData; }
   inline bool getSDR_FCEDetect() const 
      { return SDR_Status.fringeCountErrorDetected; }
   inline bool getSDR_FCECorrect() const 
      { return SDR_Status.fringeCountErrorCorrected; }
   inline bool getSDR_FCECorrectFail() const 
      { return SDR_Status.fringeCountErrorCorrectionFailure; }
   inline bool getSDR_FCEExcess() const 
      { return SDR_Status.fringeCountErrorExcessive; }
   inline bool getSDR_InvalidSpectralCal() const 
      { return SDR_Status.invalidSpectralCalibration; }
   inline bool getSDR_InvalidNeonCal() const 
      { return SDR_Status.invalidNeonCalibration; }
   inline bool getSDR_ThermalDriftExcess() const 
      { return SDR_Status.thermalDriftExcessive; }
   inline bool getSDR_ExcessiveNEdN() const { return SDR_Status.excessiveNEdN; }
   inline bool getSDR_InvalidResampling() const 
      { return SDR_Status.invalidSpectralResampling; }
   inline bool getSDR_InvalidGeolocation() const 
      { return SDR_Status.invalidGeolocation; }
   inline bool getSDR_useConfigLaserWavelength() const
      { return SDR_Status.useConfigLaserWavelength; }
   inline UInt8 getSDR_ApodType() const { return SDR_Status.apodizationType; }
   inline UInt8 getSDR_CalOrder() const { return SDR_Status.calibrationOrder; }
   inline Int32  getSDR_NumHotTargets() const 
      { return SDR_Status.hotCalTargetSignificance; }
   inline Int32  getSDR_NumColdTargets() const 
      { return SDR_Status.coldCalTargetSignificance; }
   inline Float64  getSDR_LaserWavelength() const 
      { return SDR_Status.laserWavelength; }
   inline SpectralCalibrationMethod getSDR_CalibrationType() const 
      { return SDR_Status.calibrationType; }
   static inline void configureLaserWavelength(const Float64 newWavelength)
   {
       LASER_WAVELENGTH[Band::LONGWAVE] = newWavelength * 
          metrologyWavelengthShift[Band::LONGWAVE];
       LASER_WAVELENGTH[Band::MIDWAVE] = newWavelength * 
          metrologyWavelengthShift[Band::MIDWAVE];
       LASER_WAVELENGTH[Band::SHORTWAVE] = newWavelength * 
          metrologyWavelengthShift[Band::SHORTWAVE];
   }
   static inline void configureLaserWavelengthOffSet(Band::Wavelength band, 
                                                     Float64 newOffset) 
      { metrologyWavelengthShift[(int)band] = newOffset;}
   static inline void configureImpulseNoiseCountThreshold(UInt32 newLimit) 
      { IMPULSE_NOISE_COUNT_THRESHOLD = newLimit;}
   static inline void configureNumberOPDOverscanSamples(UInt32 newOPDOverscan) 
      { NUMBER_OPD_OVERSCAN_SAMPLES = newOPDOverscan;}
   static inline void enableScaling(bool newState) 
      { scalingEnabled = newState;}
   inline void setTargetSignificance(Int16 numHotSamples, Int16 numColdSample) 
   { 
        SDR_Status.hotCalTargetSignificance = numHotSamples;
        SDR_Status.coldCalTargetSignificance = numColdSample;
   }
   inline void setSDR_CalibrationType(SpectralCalibrationMethod newType) 
      { SDR_Status.calibrationType = newType; }
   inline void setSDR_InvalidData(bool newStatus) 
      { SDR_Status.invalidData = newStatus; }
   inline void setSDR_FCEDetect(bool newStatus) 
      { SDR_Status.fringeCountErrorDetected = newStatus; }
   inline void setSDR_FCECorrectFail(bool newStatus) 
      { SDR_Status.fringeCountErrorCorrectionFailure = newStatus; }
   inline void setSDR_FCEExcess(bool newStatus) 
      { SDR_Status.fringeCountErrorExcessive = newStatus; }
   inline void setSDR_InvalidSpectralCal(bool newStatus) 
      { SDR_Status.invalidSpectralCalibration = newStatus; }
   inline void setSDR_InvalidNeonCal(bool newStatus) 
      { SDR_Status.invalidNeonCalibration = newStatus; }
   inline void setSDR_ThermalDriftExcess(bool newStatus) 
      { SDR_Status.thermalDriftExcessive = newStatus; }
   inline void setSDR_ExcessiveNEdN(bool newStatus) 
      { SDR_Status.excessiveNEdN = newStatus; }
   inline void setSDR_InvalidResampling(bool newStatus) 
      { SDR_Status.invalidSpectralResampling = newStatus; }
   inline void setSDR_InvalidGeolocation(bool newStatus) 
      { SDR_Status.invalidGeolocation = newStatus; }
  inline void setSDR_useConfigLaserWavelength(bool newStatus)
      { SDR_Status.useConfigLaserWavelength = newStatus; }
   inline void setSDR_ApodType(UInt8 newType) 
      { SDR_Status.apodizationType = newType; }
   inline void setSDR_CalOrder(UInt8 newType) 
      { SDR_Status.calibrationOrder = newType; }
   inline void setSDR_LinearityCorrMode(bool newMode)
      {SDR_Status.linearityCorrMode = newMode; }
   static inline  const Float64& getMetrologyWavelengthShift(
                                 Band::Wavelength band) 
      { return metrologyWavelengthShift[band];}
   static inline void buildResamplingWavelength(Float64 scale) 
   { 
       RESAMPLING_WAVELENGTH[Band::LONGWAVE] = 
          LASER_WAVELENGTH[Band::LONGWAVE] / scale;
       RESAMPLING_WAVELENGTH[Band::MIDWAVE] = 
          LASER_WAVELENGTH[Band::MIDWAVE] / scale;
       RESAMPLING_WAVELENGTH[Band::SHORTWAVE] = 
          LASER_WAVELENGTH[Band::SHORTWAVE] / scale;
   }
   inline void setSDR_LinearityCorrVdc(Float64 detectorVoltage) 
      {SDR_Status.Vdc = detectorVoltage; }
   inline void setSDR_LinearityCorrFactor(Float64 newScale) 
      {SDR_Status.LinErrCorrectionFactor = newScale; }
   static inline void configureInvalidNeonCal(bool neonCal) 
      { INVALID_NEON_CAL = neonCal;}
   static inline void configureDesiredBandCenter(Float64 lwDesiredBandCenter,
                                                 Float64 mwDesiredBandCenter, 
                                                 Float64 swDesiredBandCenter);
   inline Float64 getSDR_SpectralResamplingWavelength() 
      {return SDR_Status.spectralResamplingLaserWavelength;}
   inline Float64 getSDR_MonitoredWavelength() 
      {return SDR_Status.monitoredLaserWavelength;}
   inline Float64 getLosRollAngle() const 
      {return SDR_Status.losRollAngle;}
   inline Float64 getLosPitchAngle() const 
      {return SDR_Status.losPitchAngle;}
   inline Int32 getSDR_FringeCount() const 
      {return SDR_Status.sdrFringeCount;}
   inline bool getSDR_LinearityCorrectionMode() const 
      {return SDR_Status.linearityCorrMode;}
   inline Float64 getSDR_LinearityCorrVdc() const { return SDR_Status.Vdc; }
   inline Float64 getSDR_LinearityCorrFactor() const 
      { return SDR_Status.LinErrCorrectionFactor; }
   void applyLinearityErrorCorrection();
   static inline void configureFCEParamAmpThreshRejectLimit(
                                      FCE_LIMIT type, 
                                      Float64& theFCEParamAmpLimit, 
                                      Band::Wavelength theBand);
   static inline void configureFCEParamDimensionThresholdLimit(
                                      Float64& theFCEParamDimLimit, 
                                      Band::Wavelength theBand);
   static inline void configureFCEParamFractionalFceThresholdLimit(
                                      Float64& theFCEFractionalLimit, 
                                      Band::Wavelength theBand);
   static inline void configureFCEParamGoodLinearFittingThreshLimit(
                                      Float64& theGoodnessLimit, 
                                      Band::Wavelength theBand);
   static inline void configureFCEParamMaxFceThreshLimit(
                                      Float64& theMaxFCELimit, 
                                      Band::Wavelength theBand);
   static inline void configureFCEParamMaxIndex(Int32& theMaxIndex, 
                                                Band::Wavelength theBand);
   static inline void configureFCEParamMinIndex(Int32& theMinIndex, 
                                                Band::Wavelength theBand);
   static inline void configureFCEParamDefaultDetectorBand(Int32& theBand);
   static inline void configureFCEParamDefaultDetectorFOV(Int32& theFov);
   static inline void configureFCEParamICT_DSRefScene(
                      Float64& theFCEParamICT_DSRefScene);
   static inline void configureFCEParamICT_DS_ESScene(
                      Float64& theFCEParamICT_DS_ESScene);

// FCE update
   static inline void configureFCEParamEsImagThreshold(
                          Float64& theEsImagThreshold);
   static inline void configureFCEParamEsFceNumErrorThreshold(
                          UInt32& theEsFceNumErrorThreshold);
   static inline void configureFCEParamStartWavenumber(
                          UInt32& theStartWavenumber);
   static inline void configureFCEParamEndWavenumber(UInt32& theEndWavenumber);

   inline bool getSDR_LunarIntrusionDetected() const  
      {return SDR_Status.lunarIntrusionDetected;}
   inline void setSDR_LunarIntrusionDetected(bool newState) 
      {SDR_Status.lunarIntrusionDetected = newState;}
   inline Float64 calculateRealRadiance()
      {
         if(realWavenumberBin.size() > 0)
         {
            Int32 status = TrapezoidalIntegration(&realWavenumberBin,
                                                  &(this->response));

            if(status !=0)
            {
               EventLogEngine::append("Monitor Lunar Intrusion Failed.");
               SDR_Status.invalidData = true;  
            }
         }
         return response;
      }
   void updateCrisSdrGeolocation(CrisSdrAlgDataType *algDataPtr,
                                 UInt32 scanIdx,
                                 Int32 forTimeStampBias);

   void updateCrisSdrData(CrisSdrData *sdrPtr, CrisSdrHdrDataType *hdrPtr, 
                          UInt32 scanIdx, 
                  std::auto_ptr<ScienceDataProcessor>& ScienceDataProcessorPtr);

   inline virtual ~Spectra() {};

   ScienceDataRecordStatus  SDR_Status;  

   void smooth(Int32 windowSize);
   inline Int32 getSize() const { return realWavenumberBin.size();}

   static Int32 timestampBiasDifference;
   static inline void configureTimeStampBias(Int32 parameterValue) 
      {timestampBiasDifference = parameterValue;}

   // FCE update
   inline Int32 getMinIndex(Band::Wavelength band){
                    return fceParameters[band].minIndex;}
   inline Int32 getMaxIndex(Band::Wavelength band){
                    return fceParameters[band].maxIndex;}
   inline Float64 getAmpThreshRejectLimit(Band::Wavelength band, FCE_LIMIT type)
                     {return fceParameters[band].ampThreshRejectLimit[type]; }
   void reGenerateInterferogram();

 protected:
   Float64 response;
   BOOST::vector<Float64> realWavenumberBin;
   BOOST::vector<Float64> imaginaryWavenumberBin;
   BOOST::vector<Float64> NEdN_Estimate;
   Float64 firstWavenumber;
   Float64 wavenumberBinSize;
   UInt32 foldIndex;
   virtual UInt16 getVersionNumber() const = 0;
   static FCEParam fceParameters[Band::TOTAL_BANDS];
   Band::Wavelength spectralBand;
   SceneElement fieldOfView;
   FieldOfRegard fieldOfRegard;
   SweepDirection sweepDirection;
 private:

   Int32 userLocks;
   static bool scalingEnabled;
   Int32 channel;
   RawDataRecordStatusRegister RDR_Status;
   void saveVersion400(std::string& binaryImage) const;
   BOOST::vector<Float64> interferogram_DS_reals;
   BOOST::vector<Float64> interferogram_DS_imags;

};

inline Spectra::Spectra() : 
   response(0.0),
     realWavenumberBin((UInt32)0),
     imaginaryWavenumberBin((UInt32)0),
     NEdN_Estimate((UInt32)0),
     firstWavenumber(0),
     wavenumberBinSize(1),
     foldIndex((UInt32)0),
     spectralBand(Band::LONGWAVE),
     fieldOfView(FOV1),
     fieldOfRegard(ING_TOTAL_SCENES),
     sweepDirection(BOTH_DIRECTIONS),
     userLocks(1),
     channel(0)
{
   //clear status
   ZeroMemory(&RDR_Status, sizeof(RDR_Status));
   ZeroMemory(&SDR_Status, sizeof(SDR_Status));
}

inline void Spectra::detach()
{ 
   userLocks--;
   if(userLocks == 0)
      {
         delete this;
      }
}

inline const std::complex<Float64> Spectra::getComplex(Int32 bin) const
{
   std::complex<Float64> complexNumber(
      (realWavenumberBin.size() > 0u ? realWavenumberBin[bin] : 0.0),
      (imaginaryWavenumberBin.size() > 0u ? imaginaryWavenumberBin[bin] : 0.0)
   );
   return complexNumber;
}

inline void Spectra::configureDesiredBandCenter(
   Float64 lwDesiredBandCenter, 
   Float64 mwDesiredBandCenter, 
   Float64 swDesiredBandCenter)
{
   DESIRED_BAND_CENTER[Band::LONGWAVE] = lwDesiredBandCenter;
   DESIRED_BAND_CENTER[Band::MIDWAVE] = mwDesiredBandCenter;
   DESIRED_BAND_CENTER[Band::SHORTWAVE] = swDesiredBandCenter;
}

inline void Spectra::configureFCEParamAmpThreshRejectLimit(
   FCE_LIMIT type, Float64& theFCEParamAmpLimit, Band::Wavelength theBand)
{
   fceParameters[theBand].ampThreshRejectLimit[type] = theFCEParamAmpLimit;
}

inline void Spectra::configureFCEParamDimensionThresholdLimit(
   Float64& theFCEParamDimLimit, Band::Wavelength theBand)
{
   fceParameters[theBand].dimensionThresholdLimit = theFCEParamDimLimit;
}

inline void Spectra::configureFCEParamFractionalFceThresholdLimit(
   Float64& theFCEFractionalLimit, Band::Wavelength theBand)
{
   fceParameters[theBand].fractionalFceThresholdLimit= theFCEFractionalLimit;
}

inline void Spectra::configureFCEParamGoodLinearFittingThreshLimit(
   Float64& theGoodnessLimit, Band::Wavelength theBand)
{
   fceParameters[theBand].goodLinearFittingThreshLimit = theGoodnessLimit;
}

inline void Spectra::configureFCEParamMaxFceThreshLimit(
   Float64& theMaxFCELimit, Band::Wavelength theBand)
{
   fceParameters[theBand].maxFceThreshLimit = theMaxFCELimit;
}

inline void Spectra::configureFCEParamMaxIndex(
   Int32& theMaxIndex, Band::Wavelength theBand)
{
   fceParameters[theBand].maxIndex = theMaxIndex;
}

inline void Spectra::configureFCEParamMinIndex(
   Int32& theMinIndex, Band::Wavelength theBand)
{
   fceParameters[theBand].minIndex = theMinIndex;
}

inline void Spectra::configureFCEParamDefaultDetectorBand(Int32& theFov)
{
   fceParameters[Band::LONGWAVE].defaultDetectorBand = theFov;
   fceParameters[Band::MIDWAVE].defaultDetectorBand = theFov;
   fceParameters[Band::SHORTWAVE].defaultDetectorBand = theFov;
}
inline void Spectra::configureFCEParamDefaultDetectorFOV(Int32& theFov)
{
   fceParameters[Band::LONGWAVE].defaultDetectorFOV = theFov;
   fceParameters[Band::MIDWAVE].defaultDetectorFOV = theFov;
   fceParameters[Band::SHORTWAVE].defaultDetectorFOV = theFov;
}

// FCE update
inline void Spectra::configureFCEParamEsImagThreshold(
                         Float64& theEsImagThreshold)
{
   fceParameters[Band::LONGWAVE].esImagThreshold = theEsImagThreshold;
}
inline void Spectra::configureFCEParamEsFceNumErrorThreshold(
                         UInt32& theEsFceNumErrorThreshold)
{
   fceParameters[Band::LONGWAVE].esFceNumErrorThreshold = 
                         theEsFceNumErrorThreshold;
}
inline void Spectra::configureFCEParamStartWavenumber(
                         UInt32& theStartWavenumber)
{
   fceParameters[Band::LONGWAVE].startWavenumber = theStartWavenumber;
}
inline void Spectra::configureFCEParamEndWavenumber(UInt32& theEndWavenumber)
{
   fceParameters[Band::LONGWAVE].endWavenumber = theEndWavenumber;
}

/**
 * Inline definition of getRDR_MissingData
 */
inline bool Spectra::getRDR_MissingData() const
{
   return RDR_Status.missingData;
}
#endif /* _INC_SPECTRA_3F2F9E6E028D_INCLUDED */
