/* BEGIN JPSS HEADER */
/*****************************************************************************
*
* Peraton, Inc.  All rights reserved.
*
* GOVERNMENT RIGHTS:
* Per contract 1332KP23CNEEJ0001, CAR 1352.227-70 applies giving full
* copyrights to data, including reports and other copyrighted materials
* to the United States Government.
*
* THIRD PARTY RIGHTS:
* Note: Third party entities may own copyrights in this work.
*
* EXPORT CONTROL:
* Unless otherwise specified beneath this header, this item does not contain
* Technology or Technical Data controlled under either the U.S. International
* Traffic in Arms Regulations or the U.S. Export Administration Regulations.
*
* CLASSIFICATION: B
* Refer to Software Standards and Practices Manual (SSPM) for coding and
* documentation standards.
*
*****************************************************************************/
/* END JPSS HEADER */

/*****************************************************************************
*
*  NAME: ScienceDataProcessor.cpp
*
*  DESCRIPTION:  See description preceeding declaration below
*
*****************************************************************************/

/***********************************************************************
*  HISTORY:
*
*  DATE       PR#     AUTHOR        Build  DESCRIPTION
*  ---------  ------  ------------  -----  ------------------------------------
*  25APR2005          B. Henderson  1.3    Initial Version
*                     D. Elliott
*                     T. Hahn
*  01MAR2006          D. Elliott    1.4    Fringe Count Error
*                     T. Hahn              Follow On Drop
*  02AUG2006  011858  D. Elliott    1.4    Modified ITT header with
*                                          permission from Steven G.
*                                          Minnie from ITT Space
*                                          Systems, LLC
*  05SEP2006  012191  T. Hahn       1.4    Fixed error Correction
*                                          Matrix Error
*  12SEP2006  012216  T. Hahn       1.4    Combine Correction Matrix and
*                                          Engineering Packet items
*  20JUN2007          J. DeLotelle  1.5    Updated copyright and government
*                                          rights info in prologue.
*  27JUN2007          J. DeLotelle  1.5    Changed DS_Symmetry to
*                                          DeepSpace_Symmetry for clarity.
*  29JUN2007          D. Elliott    1.5    Deep Space symmetry QF addition.
*  19DEC2007  016355  T. Gardner    1.5    Fixed a write past end of
*                                          array problem
*  07JAN2008  015748  B. Henderson  1.5x1  ISTN_CRIS_SDR_NGST_1.1
*                                          drop implementation
*  21FEB2008  015748  K. Boswell    1.5x1  ISTN_CRIS_SDR_NGST_1.1
*                                          drop implementation -
*                                          format updates for DFCB compliance
*                                          ref. NP-EMD-2007.510.0027 RevB
*  19MAR2008  017044  K. Boswell    1.5x1  Fixed dereferencing null algorithm
*                                          data pointer from temporary
*                                          variable; fixed fill values in
*                                          DS_Symmetry; removed PRINT_MEM_INFO
*                                          statements.
*  12JUN2008  017888  B. Bohannan          logger refactor APIs
*  20JAN2009  019301  K. Boswell    1.5x1  ISTN_CRIS_SDR_UNIX_NGST_2.1
*                                          drop implementation
*                                          per TM NP-EMD.2008.510.0055
*  27MAY2009          L. Wang     ALG1443  Update for nonlinearity correction
*  03FEB2010  021718  K. Boswell    SenCh  ISTN_CRIS_SDR_UNIX_NGST_2.2
*                                          drop implementation
*                                          per TM NP-EMD.2009.510.0046
*  18MAR2010  022975  R. Evans      SenCh  Commented out theSpectra in
*                                          monitorLaserDiode in order to
*                                          resolve a coring issue until
*                                          a fix can be gotten from the
*                                          science people
*  10MAR2010  021464  K. Boswell    SenCh  Changes in support of
*                                          NP-EMD.2009.510.0046
*                                          CrIS_Scan Baffle Temperature
*                                          Correction
*  19MAY2010  023135  K. Boswell    SenCh  ISTN_CRIS_SDR_UNIX_NGST_2.2.1
*                                          drop implementation per TM
*                                          NP-EMD.2010.510.0006 and
*                                          ISTN_CRIS_SDR_UNIX_NGST_2.2.2 per
*                                          TM NP-EMD.2010.510.0006 Rev A
*                                          and ECR A-275
*  18OCT2010  024814  W. Dare       1.5.4  Assign 0 to unassigned ptrs.
*  04JAN2012  029114  T. Gardner    1.5.6  DR4375 - Correct CMO update error
*                                          when maneuver flag is on.
*  31JUL2012  031562  T. Gardner    Mx6.3  Use sciCalMissing flag
*  19NOV2012  032693  T. Gardner    Mx7    Removed addition of pad to CMO
*  04MAY2013  034609  T. Gardner    Mx8.0  Added checkICT_Status method
*  06JUN2013  034187  T. Gardner    Mx8.0  LaserWavelength is updated only
*             034188                       when CMO is updated and saved
*  04JUN2012  030126  J. Schwartz   Bk1.5  Messaging/TaskData/TK updates
*  14NOV2012  031266  S. Wilderdyke Bk1.5  Replaced BOOST Archive/
*                                          Serialization and refactored
*                                          associated methods
*  02JUL2013  033702  S. Wilderdyke Bk2.0  Changes to work with
*                     S. Manoj             INF Util Time Updates
*  05NOV2013  035224  S. Joo       Bk2.0   Fixed Compiler Warnings in Linux
*  05MAY2014  038866  W. Dare      Mx8.5   474-CCR-14-1707 Time Stamp
*                                          Error
*  27AUG2014  038187  J. DeLotelle  Bk2.0  Updated processSpectra per
*                                          474-CCR-14-1630.
* 26JUN2014           Y. Chen/Y.Han Mx8.5  Updates for full spectral resolution
*                                          Add variable calibrationOrder and
*                                          functionality
* 10AUG2014           Y. Chen       Mx8.5  Separation of inverse SA and other
*                                          matrixes. De-coupling CMO and EngPkt
*                                          file.Added ILSparameters into CMO file
* 24Aug2014           Y. Han        Mx8.5  Clear up codes and remove multiple
*                                          UserLowestWavenumber and
*                                          instrumentLowestWavenumber calculations
*                                          cross the Pro code, which are now only
*                                          computed in updateCorrectionMatrixParameters()
*                                          and calculateWavenumberBinSpacing() in
*                                          this class, respectively
* 15Sep2014           Y. Chen       Mx8.5  Correction of a bug in clearup() by
*                                          keeping the calibration matrix and other
*                                          matrixes when slidewindow rebuilding
* 14OCT2014  042555   SEClark   Blk2.0.0.4 CrIS SSPM Compliance.
* 03JAN2015   41268   J. Eberle    Blk2.0  Header Changes
* 11MAY2015   48586   T. Gardner   Blk2.0  Integrate TR/FR updates into
*                                          block 2.0
* 15MAY2015           L. Wang   V2.0_beta  Fringe count error correction implementation
* 29JUN2015           Y. Chen   V2.0_beta  Integrate FCE correction
* 06JUL2015           Y. Chen   V2.0_beta  DR7982 change maxLunarRadiance to an array
*                                          and check all bands for lunar intrusion
* 14AUG2015           Y. Chen/Y.Han V2.0_beta  Added calibration algorithm
*                                              RadCalFirst_FPfINVsaPf_bigN
* 20AUG2015           Y. Chen/Y.Han V2.0_beta  Add variable
*                                              usePostFilterOrCosineFilter
*                                              and functionality
* 19JAN2016   46719   Tim Fisher   Blk2.0  Static Analysis
* 31MAR2016           Y. Chen      Blk2.0  Integrate TR/FR updates into
*                                          block 2.0
* 5May2015            Y. Chen/Y. Han Blk2.0  Adding variable
*                                            dataPointsTruncatedInterferogram
*                                            and truncation resampling matrix
*                                            clear up code for deltaSigma
*                                            and firstWavenumber in CMO calculation
* 27MAY2016            Y. Han      Blk2.0    Add functionality
*                                            for apodizing interferogram
* 24JUL2017            H. Xu/Y. Chen Blk2.0  Added PCT coefficients to spectra
* 26JUL2017            Y. Chen     Blk2.0    Following UW modification
*                                            adding an additional poster filter
*                                            before truncation matrix F(truncate)
* 26JUL2017            Y. Chen     Blk2.0    Keeping the folding index the same 
*                                            during the run so that reference
*                                            and Earth spectra are consistent  
* 06SEP2017            G. Pachl    Blk2.1    CCRs-17-3543, 3541.
* 30SEP2017            J. Gibbs    Blk2.1    removed unused parameters 
* 31AUG2017   63167    J. Gibbs    Blk2.1    fixed compile warnings
* 26APR2018   65628    G. Pachl    Blk2.1    Merged into Baseline CCR 3542.
* 11APR2018            Y. Chen     Blk2.1    DR8631 add min and max
*                                            frequency for lunar intrusion
*                                            detection in enableLunarIntrusionTracking
* 17JUL2018   66227    G. Pachl    Blk2.1    Merge CCR 3922 Lunar intrusion
* 28NOV2018   66729    Pachl       Blk2.1    Rhel7 updates fixes / compile warnings.
* 17APR2019            Y. Chen     Blk2.1    CCR 4469 Implemented Polarization
*                                            Correction
* 23OCT2019   68993    D. Tislin   Blk2.2    Fixed compile warnings.
* 12FEB2020   66565    G.Pachl     Blk2.3    Removal of unused code.
******************************************************************************/

/*******************************************************************************
* This document contains information proprietary and confidential to ITT Space
* Systems, LLC ("ITT"), or a third party for whom ITT may have a legal
* obligation to protect such information from unauthorized disclosure, use or
* duplication. Any disclosure, use, duplication or diversion contrary
* to U.S. Law of this document or of any of the information contained herein
* for other than the
* specific purpose for which it was disclosed is expressly prohibited,
* except as ITT has otherwise
* agreed in writing or as the U.S. Government has authorized. All copies of
* this document are the
* sole property of ITT and will be returned promptly upon request
******************************************************************************/

#include <ScienceDataProcessor.h>
#include <ProCmnLogger.h>
#include <ProCmnMessage.h>
#include <ProSdrCrisException.h>
#include <ProCmnProcessData.h>
#include <FloatCompare.h>
#include <CorrectionMatrix.h>
#include <ProCmnCallerID.h>
#include <ProCmnTimeUtil.h>
#include <CrISCmoAr.h>
#include <CrISEngAr.h>
#include <CrISILSAr.h>
#include <SpectralResDefs.h>

#define ROUND(num)  ((num >= 0) ? floor(num+0.5) : ceil(num-0.5))

//Science Code calculated constants
static const Float64 PlankConstC1 = 1.1910427e-05;
static const Float64 PlankConstC2 = 1.4387752;
/* static */
const Float64 ScienceDataProcessor::DEFAULT_LASER_WAVELENGTH = 1550.0;

static Float64 desiredBandCenter[Band::TOTAL_BANDS] =
{
    (1095.0 + 650.0)/2.0,
    (1210.0 + 1750.0)/2.0,
    (2155.0 + 2550.0)/2.0
};

static UINT foldIndex[Band::TOTAL_BANDS] = {0,0,0};
static UInt32 savedUnfoldIndex[Band::TOTAL_BANDS] = {0,0,0};
static UInt32 unfoldIndex[Band::TOTAL_BANDS] = {0,0,0};
static bool savedfoldIndex_flag[Band::TOTAL_BANDS] = {false, false, false};

Float64 ScienceDataProcessor::radianceModelOffset[MAX_FOV][Band::TOTAL_BANDS] =
   {{0., 0., 0.}, {0., 0., 0.}, {0., 0., 0.},
    {0., 0., 0.}, {0., 0., 0.}, {0., 0., 0.},
    {0., 0., 0.}, {0., 0., 0.}, {0., 0., 0.}};

// ##METHOD HEADER-START## ****************************************************
//
// ARGUMENTS:
//  INPUTS:
//   None
//
//  OUTPUTS:
//   None.
//
// DESCRIPTION:
//  Constructor
//
//*** ##METHOD HEADER-END## ***************************************************
ScienceDataProcessor::ScienceDataProcessor() : 
                      resampling_laserwavelength(0.0),
                      DMS_Data_Inputs(0),
                      windowSize(30),
                      SCIENCE_CALIBRATION_FRAME_TYPE(1289),
                      ENGINEERING_CALIBRATION_FRAME_TYPE(1290),
                      totalActiveThreads(0),
                      referenceHotSceneReceived(0),
                      initialCorrectionReady(false),
                      appShiftFactorFlag(false), 
                      scanIdx_(0),
                      numCMOBuilds_(0),
                      numEngBuilds_(0),
                      taskedSequentially_(false),
                      scanWhichCausedCMORebuild_(0),
                      scanWhichCausedEngRebuild_(0),
                      currentDSfwd_p(false),
                      currentDSrev_p(false),
                      engPkt_Loaded(false),
                      ilsCorrection_Loaded(false),
                      firstEngPkt(false)
{
    engPacketCount = 0;

    //reset
    scanWhichCausedCMORebuild_ = CRIS_WINDOW_SIZE/2.0;

    // set to first one
    scanWhichCausedEngRebuild_ = 0;

    //initialize wavenumber bin spacing class variables
    for (int bandIndex = 0; bandIndex < Band::TOTAL_BANDS; bandIndex++)
    {
        maxPathDifferential[bandIndex] = 0;
        deltaSigma[bandIndex] = 0;
        lowestWavenumber[bandIndex] = 0;
    }

    //init counters
    ZeroMemory(sequenceCount,
       sizeof(int) * MAX_FOV * Band::TOTAL_BANDS * TOTAL_SCENES *
          TOTAL_DIRECTIONS);
    ZeroMemory(scienceCount,
       sizeof(int) * MAX_FOV * Band::TOTAL_BANDS * TOTAL_SCENES *
          TOTAL_DIRECTIONS);
    ZeroMemory(targetCount,
       sizeof(int) * MAX_FOV * Band::TOTAL_BANDS * TOTAL_SCENES *
          TOTAL_DIRECTIONS);
    ZeroMemory(invalidInterferogramCount,
       sizeof(int) * MAX_FOV * Band::TOTAL_BANDS);
    ZeroMemory(invalidSpectraCount,
       sizeof(int) * MAX_FOV * Band::TOTAL_BANDS);

    ZeroMemory(&validCalTargetDuration.startTime, sizeof(SYSTEMTIME));
    ZeroMemory(&validCalTargetDuration.stopTime, sizeof(SYSTEMTIME));
    ZeroMemory(&validColdTargetDuration.startTime, sizeof(SYSTEMTIME));
    ZeroMemory(&validColdTargetDuration.stopTime, sizeof(SYSTEMTIME));
    ZeroMemory(&validHotTargetDuration.startTime, sizeof(SYSTEMTIME));
    ZeroMemory(&validHotTargetDuration.stopTime, sizeof(SYSTEMTIME));

    ZeroMemory(&maxNpCrossingTime_, sizeof(SYSTEMTIME));
    ZeroMemory(&minNpCrossingTime_, sizeof(SYSTEMTIME));
    ZeroMemory(&validCalTargetTolerence, sizeof(SYSTEMTIME));
    ZeroMemory(&validColdTargetTolerence, sizeof(SYSTEMTIME));
    ZeroMemory(&validHotTargetTolerence, sizeof(SYSTEMTIME));

    ZeroMemory(&AlgorithmParameter, sizeof(SDR_AlgorithmnParameter));
    ZeroMemory(&CalibrationParameter, sizeof(CalibrationParam));
    ZeroMemory(&GeneralParameter, sizeof(SDR_GeneralParameters));
    ZeroMemory(&TestEnvironmentParameter, sizeof(CrIS_TestEnvironmentParameters));

    InstrumentCharacteristic.LinearityCorrectionParameter.
       FIRresponse[Band::LONGWAVE].resize(NUM_FIR_FILTER_RESP_PTS);
    InstrumentCharacteristic.LinearityCorrectionParameter.
       FIRresponse[Band::MIDWAVE].resize(NUM_FIR_FILTER_RESP_PTS);
    InstrumentCharacteristic.LinearityCorrectionParameter.
       FIRresponse[Band::SHORTWAVE].resize(NUM_FIR_FILTER_RESP_PTS);

    InstrumentCharacteristic.firAccumulatorStartBit[Band::LONGWAVE] = 14;
    InstrumentCharacteristic.firAccumulatorStartBit[Band::MIDWAVE] = 15;
    InstrumentCharacteristic.firAccumulatorStartBit[Band::SHORTWAVE] = 17;

    memset(requiresPhaseSync, true,
       sizeof(bool) * MAX_FOV * Band::TOTAL_BANDS * TOTAL_DIRECTIONS);
    // FCE update
    memset(earthScenefceDetectionComplete,false,
       sizeof(bool) * TOTAL_SCENES);
    ZeroMemory(syncAttemptCount,
       sizeof(int) * MAX_FOV * Band::TOTAL_BANDS * TOTAL_DIRECTIONS);
    memset(referenceSceneReceived, false,
       sizeof(bool) * SpectraManager::TOTAL_REF_SPECTRA * TOTAL_DIRECTIONS);
    ZeroMemory(referenceSceneSync,
       sizeof(bool) * MAX_FOV * Band::TOTAL_BANDS * TOTAL_DIRECTIONS);

    // FCE update
    referenceHotSceneReceived = 0;

    setAllowCalibrationTargetDataMissing( true );
    setAllowEngineeringDataPacketsMissing( true );
    setAllowSpaceTargetTemperatureDataMissing( true );
    setDisableTimeStampBasedMovingWindow( true );
    setPerformRadiometricCalibration( true );
    setSkipIctDsPhaseSynchronization( true );
    setUseDeepSpaceRadiance( true );
    setUseIctEnvironmentalCorrectionModel( false );
    setSuppressBaffleProfile(false);
    setUseWavenumberDependentDsEmissivity( false );
    setUseWavenumberDependentIctEmissivity( false );
    setMovingAverageWindowSize( 30 );
    setAllowScienceTlmDataMissing( true );
    setDsTemperatureOrigin( (DsTemperatureOriginSource) DS_CONFIG);
    setIctEmissivityOrigin( (IctEmissivityOriginSource) ICT_CONFIG);
    setInstrumentTemperatureOrigin(
       (InstrumentTemperatureOriginSource) INSTRUMENT_CONFIG);

    setApplyPolarizationCorrections( false );
    setApplyPostCalibrationFilterMatrixCorrection( true );
    setApplyIlsFovEffectsCorrection(false);
    setApplyIlsResidualEffectCorrection(false);
    setApplyResamplingMatrix(false);
    setDisableLaserMonitoring(false);
    setPerformFringeCountErrorHandling(false);
    setPerformPolarizationCorrection(false);
    setPerformSpectralAndSpatialCorrection(false);
    setUseSavedMatrices( false );
    setUsePostFilterOrCosineFilter( 0 );
    setApodizationType( NONE );
    setCalibrationOrder( RadCalFirst_ATBD_versionA );
    setSincFlag( RadCalFirst_ATBD_versionA );
    setLaserDiodeWavelengthOrigin( LASER_CONFIG);
    setImpulseNoiseCountThreshold( 0 );
    setUserClippingSelection( false );

    setEdrDeltaSigma(Band::LONGWAVE, 0.625 );
    setEdrDeltaSigma(Band::MIDWAVE, 1.250 );
    setEdrDeltaSigma(Band::SHORTWAVE, 2.50 );
    setEdrMaximumWavenumber(Band::LONGWAVE, 1095.0 );
    setEdrMaximumWavenumber(Band::MIDWAVE, 1750.0 );
    setEdrMaximumWavenumber(Band::SHORTWAVE, 2550.0 );
    setEdrMinimumWavenumber(Band::LONGWAVE, 650.0 );
    setEdrMinimumWavenumber(Band::MIDWAVE, 1210.0 );
    setEdrMinimumWavenumber(Band::SHORTWAVE, 2155.0 );
    setEdrNumberOfPoints(Band::LONGWAVE, LWNUMBEROFPOINTS-4);
    setEdrNumberOfPoints(Band::MIDWAVE, MW_NUMBER_OF_POINTS-4);
    setEdrNumberOfPoints(Band::SHORTWAVE, SW_NUMBER_OF_POINTS-4);

    enableLunarIntrusionTracking(false, Band::LONGWAVE, 0.0, 863.7 , 901.2);
    enableLunarIntrusionTracking(false, Band::MIDWAVE,  0.0, 1233.7, 1270.8);
    enableLunarIntrusionTracking(false, Band::SHORTWAVE,0.0, 2184.1, 2221.7);

    setBenchMeanIctEmissivity(Band::LONGWAVE, 0.96 );
    setBenchMeanIctEmissivity(Band::MIDWAVE, 0.96 );
    setBenchMeanIctEmissivity(Band::SHORTWAVE, 0.96 );
    setChamberMeanIctEmissivity(Band::LONGWAVE, 0.995 );
    setChamberMeanIctEmissivity(Band::MIDWAVE, 0.995 );
    setChamberMeanIctEmissivity(Band::SHORTWAVE, 0.995 );
    setHotReferenceFieldOfRegard( InternalCalTarget );
    setColdReferenceFieldOfRegard( DeepSpace );
    setDefaultLaserWavelength( 1550.0 );
    setNumberSamplesPerLaserWavelength( 2 );
    setSpaceTargetTemperatureDriftLimit( 0.0 );

    //accessor bypassed to avoid premature triggering of calculations
    InstrumentCharacteristic.decimationFactor[Band::LONGWAVE] =
       LW_DECIMATION_RATES;
    InstrumentCharacteristic.dataPointsUndecimatedInterferogram[Band::LONGWAVE]
       = LW_COLLECTED_SAMPLES;

    InstrumentCharacteristic.decimationFactor[Band::MIDWAVE] =
       MW_DECIMATION_RATES;
    InstrumentCharacteristic.dataPointsUndecimatedInterferogram[Band::MIDWAVE] =
       MW_COLLECTED_SAMPLES;

    InstrumentCharacteristic.decimationFactor[Band::SHORTWAVE] =
       SW_DECIMATION_RATES;
    InstrumentCharacteristic.
       dataPointsUndecimatedInterferogram[Band::SHORTWAVE] =
          SW_COLLECTED_SAMPLES;

    InstrumentCharacteristic.dataPointsDecimatedInterferogram[Band::LONGWAVE] =
       LW_POINTS_DECIMATED_IFGM;
    InstrumentCharacteristic.dataPointsDecimatedInterferogram[Band::MIDWAVE] =
       MW_POINTS_DECIMATED_IFGM;
    InstrumentCharacteristic.dataPointsDecimatedInterferogram[Band::SHORTWAVE] =
       SW_POINTS_DECIMATED_IFGM;

    const Float64 tmpArr[]= { 0.406406, 1.182396, -0.802275, 0.213474 };
    for (int c = 0; c  <  SURF_EMISS_COEFF_CMAX; ++c)
    {
        SurfaceEmissivityCoeffIdxEnum idx = (SurfaceEmissivityCoeffIdxEnum) c;
        setSurfaceEmissivityCoeff(idx, tmpArr[c]);
    }

    setEngCalInstrumentCharacteristics(InstrumentCharacteristic);

    setBeamsplitterTempBench(7.85);
    setBeamsplitterTempChamber(7.85);
    setDsTempBench(4.0);
    setDsTempChamber(95.0);
    setIctTempBench(293.0);
    setIctTempChamber(293.0);
    setMeanDsEmissivityBench(0.995);
    setMeanDsEmissivityChamber(0.960);
    setOmaTempBench(7.85);
    setOmaTempChamber(7.85);
    setScanBaffleTempBench(345.0);
    setScanBaffleTempChamber(345.0);

    setIctPrt1Bias( 0.0 );
    setIctPrt2Bias( 0.0 );
    setComputedWavelengthRejectionThreshold( 28 );
    setLaserWavelengthDriftTolerance( 2.0 );
    setNumberOpdOverscanSamples( 1 );

    setPostCalibrationA1(Band::LONGWAVE, 15 );
    setPostCalibrationA1(Band::MIDWAVE, 22 );
    setPostCalibrationA1(Band::SHORTWAVE, 8 );
    setPostCalibrationA2(Band::SHORTWAVE, 2.0 );
    setPostCalibrationA2(Band::MIDWAVE, 1.0 );
    setPostCalibrationA2(Band::LONGWAVE, 0.5 );
    setPostCalibrationA3(Band::LONGWAVE, 15 );
    setPostCalibrationA3(Band::MIDWAVE, 22 );
    setPostCalibrationA3(Band::SHORTWAVE, 8 );
    setPostCalibrationA4(Band::SHORTWAVE, 2.0 );
    setPostCalibrationA4(Band::MIDWAVE, 1.0 );
    setPostCalibrationA4(Band::LONGWAVE, 0.5 );
    setPostCalibrationK (Band::LONGWAVE, 864  );
    setPostCalibrationK0(Band::LONGWAVE, 77 );
    setPostCalibrationK1(Band::LONGWAVE, 789 );
    setPostCalibrationK (Band::MIDWAVE, 528  );
    setPostCalibrationK0(Band::MIDWAVE, 49 );
    setPostCalibrationK1(Band::MIDWAVE, 481 );
    setPostCalibrationK (Band::SHORTWAVE, 200  );
    setPostCalibrationK0(Band::SHORTWAVE, 22 );
    setPostCalibrationK1(Band::SHORTWAVE, 180 );

    setInstrumentLocation(BENCH);
    setSpectralCalibrationMethod(
       (SpectralCalibrationMethod)CalibrationType_Complex);
    enableNEdN(true);
    setFoldIndexOffset(Band::LONGWAVE, 0);
    setFoldIndexOffset(Band::MIDWAVE, 0);
    setFoldIndexOffset(Band::SHORTWAVE, 0);
    setSAExpansionFactor(Band::LONGWAVE, 1.0);
    setSAExpansionFactor(Band::MIDWAVE, 1.0);
    setSAExpansionFactor(Band::SHORTWAVE, 1.0);

    //calculate and configure spectra's desired band center
    calculateDesiredBandCenter();

    //propagate data to subcomponents if any of the
    //above blue components are altered.
    updateCorrectionMatrixParameters();

    polarizationCorrectionCurve = new UncalibratedSpectra();

    processingPeriod.setLinearityErrorParameters(Band::LONGWAVE,
       theEngineeringCalibrationRecord.
          getEngCalRec_LinearityErrorParameters().getParameters());
    processingPeriod.setLinearityErrorParameters(Band::MIDWAVE,
       theEngineeringCalibrationRecord.
          getEngCalRec_LinearityErrorParameters().getParameters());
    processingPeriod.setLinearityErrorParameters(Band::SHORTWAVE,
       theEngineeringCalibrationRecord.
          getEngCalRec_LinearityErrorParameters().getParameters());

    for(Int32 fov = 0; fov < MAX_FOV; fov++)
    {
      for(Int32 band = 0; band < Band::TOTAL_BANDS; band++)
      {
         processingPeriod.setCalibrationMatrix(
            (SceneElement)fov, (Band::Wavelength)band,
               calibrationMatrix[fov][band]);
      }
    }

    for(Int32 band = 0; band < Band::TOTAL_BANDS; band++)
    {
      processingPeriod.setResamplingMatrix((Band::Wavelength)band,
                          resamplingMatrix[band]);
    }

    // set the pointer to CorrectionParameter
    processingPeriod.setCorrectionParameter(CorrectionParameter);
    // set the pointer to CalibrationParameter
    processingPeriod.setCalibrationParameter(CalibrationParameter);

    InstrumentCharacteristic.dataPointsTruncatedInterferogram[Band::LONGWAVE]
        = 866;
    InstrumentCharacteristic.dataPointsTruncatedInterferogram[Band::MIDWAVE]
        = 1052;
    InstrumentCharacteristic.dataPointsTruncatedInterferogram[Band::SHORTWAVE]
        = 799;

    //Get path of Application Directory
    char szAppPath[MAX_PATH] = "";
    getcwd(szAppPath, sizeof(szAppPath) - 1);
    appDirectory = szAppPath;

    initialCorrectionReady = false; //may not be needed operationally,
                                    //but WILL NOT do any harm

    // plankConstC1 and plankConstC2 need to be set
    setPlankConstantC1(PlankConstC1);
    setPlankConstantC2(PlankConstC2);

    setAppShiftFactorFlag(false);
    ZeroMemory(shiftFactor, sizeof(Float64) * MAX_FOV);
}



// ##METHOD HEADER-START## ****************************************************
//
// ARGUMENTS:
//  INPUTS:
//   None
//
//  OUTPUTS:
//   None.
//
// DESCRIPTION:
//  Destructor
//
//*** ##METHOD HEADER-END## ***************************************************
/*virtual*/ScienceDataProcessor::~ScienceDataProcessor()
{
    if(polarizationCorrectionCurve)
    {
        polarizationCorrectionCurve->detach();
    }

    //Destroy Temperature History
    theScienceCalibrationRecord.clearHistory();
    theTargetTempCalibrationRecord.getProcessingPeriod().clearHistory();
}

// ##METHOD HEADER-START## ****************************************************
//
// ARGUMENTS:
//  INPUTS:
//   None
//
//  OUTPUTS:
//   None.
//
// DESCRIPTION:
//   Clean up of ScienceDataProcessor data for next dispatch
//
//*** ##METHOD HEADER-END## ***************************************************
void ScienceDataProcessor::cleanup()
{
    //init counters
    ZeroMemory(sequenceCount,
       sizeof(int) * MAX_FOV * Band::TOTAL_BANDS * TOTAL_SCENES *
          TOTAL_DIRECTIONS);
    ZeroMemory(scienceCount, sizeof(int) * MAX_FOV * Band::TOTAL_BANDS *
       TOTAL_SCENES * TOTAL_DIRECTIONS);
    ZeroMemory(targetCount, sizeof(int) * MAX_FOV * Band::TOTAL_BANDS *
       TOTAL_SCENES * TOTAL_DIRECTIONS);
    ZeroMemory(invalidInterferogramCount, sizeof(int) * MAX_FOV *
       Band::TOTAL_BANDS);
    ZeroMemory(invalidSpectraCount, sizeof(int) * MAX_FOV *
       Band::TOTAL_BANDS);

    ZeroMemory(&validCalTargetDuration, sizeof(SYSTEMTIME));
    ZeroMemory(&validColdTargetDuration, sizeof(SYSTEMTIME));
    ZeroMemory(&validHotTargetDuration, sizeof(SYSTEMTIME));

    memset(requiresPhaseSync, true,
       sizeof(bool) * MAX_FOV * Band::TOTAL_BANDS * TOTAL_DIRECTIONS);
    ZeroMemory(syncAttemptCount,
       sizeof(int) * MAX_FOV * Band::TOTAL_BANDS * TOTAL_DIRECTIONS);

    currentDSfwd_p = false;
    currentDSrev_p = false;

    engPkt_Loaded = false;
    ilsCorrection_Loaded = false;

    // Free the SpectraManager objects and reinitialize objects
    processingPeriod.cleanup();
    processingPeriod.init();

    // clear vector containers
    for(Int32 scan = 0; scan < CRIS_SCAN_PER_GRAN; scan++)
    {
      for(UInt32 fov = 0; fov < MAX_FOV; fov++)
      {
        for(UInt32 band = 0; band < Band::TOTAL_BANDS; band++)
        {
            forDS_reals[scan][fov][band].clear();
            forDS_imags[scan][fov][band].clear();
            revDS_reals[scan][fov][band].clear();
            revDS_imags[scan][fov][band].clear();
        }
      }
    }
}
// ##METHOD HEADER-START## ****************************************************
//
// ARGUMENTS:
//  INPUTS:
//   Band::Wavelength theBand
//
//  OUTPUTS:
//   None
//
// DESCRIPTION:
//  Calculate the post calibration table.
//  This calculation is only BAND SPECIFIC
//
//*** ##METHOD HEADER-END## ***************************************************/
bool ScienceDataProcessor::buildPostCalibrationTable(Band::Wavelength theBand)
{
    UInt32 binSize = InstrumentCharacteristic.
       dataPointsDecimatedInterferogram[theBand];
    char logMessage[EVENT_LOG_LENGTH];
    sprintf(logMessage,"ATTN: buildPostCalibrationTable binsize  %d", binSize);
    EventLogEngine::append(logMessage);

    //set size of vector
    if( postFilter[theBand].size1() != binSize )
    {
       postFilter[theBand].resize(binSize, binSize, 0, 0);
    }

    for (UInt32 postCalIndex = 0; postCalIndex < binSize; postCalIndex++)
    {
        //Calculate Post Calibration table for FOV1
        (postFilter[theBand])(postCalIndex, postCalIndex) =
           (1.0f / (exp(AlgorithmParameter.postCalibrationA2[theBand] *
                    (AlgorithmParameter.postCalibrationK0[theBand] -
                    AlgorithmParameter.postCalibrationA1[theBand] -
                         (postCalIndex + 1.0f))) + 1.0f)) *
           (1.0f / (exp(AlgorithmParameter.postCalibrationA4[theBand] *
                   ((postCalIndex + 1.0f) -
                    AlgorithmParameter.postCalibrationK1[theBand] -
                    AlgorithmParameter.postCalibrationA3[theBand])) + 1.0f));
    }

    sprintf(logMessage,"ATTN: buildPostCalibrationTable finish" );
    EventLogEngine::append(logMessage);
    return true;
}
// ##METHOD HEADER-START## ****************************************************
//
// ARGUMENTS:
//  INPUTS:
//   Band::Wavelength theBand
//
//  OUTPUTS:
//   None
//
// DESCRIPTION:
//  Calculate the raised cosine post calibration table.
//  This calculation is only BAND SPECIFIC
//
//*** ##METHOD HEADER-END## ***************************************************/
bool ScienceDataProcessor::buildPostCalibrationTable_raisedCos(
                     Band::Wavelength theBand)
{

    UInt32 binSize =
          InstrumentCharacteristic.dataPointsDecimatedInterferogram[theBand];
    char logMessage[EVENT_LOG_LENGTH];
    sprintf(logMessage,"ATTN: buildPostCalibrationTable_raisedCos binsize  %d",
                binSize );
    EventLogEngine::append(logMessage);


    UInt32 cos_width_left_n =
             ceil(AlgorithmParameter.postCalibrationA2[theBand]/
                 deltaSigma[theBand]);
    UInt32 cos_width_right_n =
             ceil(AlgorithmParameter.postCalibrationA4[theBand]/
                 deltaSigma[theBand]);

    // left part
    UInt32 highBinIndex_left =
             ROUND((CorrectionParameter.edrMinimumWavenumber[theBand] +
                  (Float64)AlgorithmParameter.postCalibrationA1[theBand] -
                   lowestWavenumber[theBand])/deltaSigma[theBand]);
    UInt32 lowBinIndex_left = highBinIndex_left - cos_width_left_n;

    // right part
    UInt32 lowBinIndex_right =
             ceil((CorrectionParameter.edrMaximumWavenumber[theBand] -
                 (Float64)AlgorithmParameter.postCalibrationA3[theBand] -
                  lowestWavenumber[theBand])/deltaSigma[theBand]);
    UInt32 highBinIndex_right = lowBinIndex_right + cos_width_right_n;

    // build filter
    //set size of vector
    if( postFilter[theBand].size1() != binSize )
    {
       postFilter[theBand].resize(binSize, binSize, 0.0, 0.0);
    }

    for (UInt32 postCalIndex = 0;
         postCalIndex < lowBinIndex_left; postCalIndex++)
    {
       (postFilter[theBand])(postCalIndex, postCalIndex) = 0.0;
    }

    for (UInt32 postCalIndex = lowBinIndex_left;
         postCalIndex <= highBinIndex_left; postCalIndex++)
    {
       Float64 i = (postCalIndex - lowBinIndex_left);
       Float64 v = 0.5 * cos( M_PI * (1.0 + (Float64)i /
                   (Float64)cos_width_left_n) ) + 0.5;
       if (v <= 0.0) v = 0.0;
       (postFilter[theBand])(postCalIndex, postCalIndex) = v;
    }

    for (UInt32 postCalIndex = highBinIndex_left + 1;
            postCalIndex < lowBinIndex_right; postCalIndex++)
    {
        (postFilter[theBand])(postCalIndex, postCalIndex) = 1.0;
    }

    for (UInt32 postCalIndex = lowBinIndex_right;
          postCalIndex <= highBinIndex_right; postCalIndex++)
    {
       Float64 i = (postCalIndex - lowBinIndex_right);
       Float64 v = 0.5 * cos( M_PI * (Float64)i /
                   (Float64)cos_width_right_n ) + 0.5;
       if (v <= 0.0) v = 0.0;
       (postFilter[theBand])(postCalIndex, postCalIndex) = v;
    }

    for (UInt32 postCalIndex = highBinIndex_right+1;
         postCalIndex < binSize; postCalIndex++)
    {
        (postFilter[theBand])(postCalIndex, postCalIndex) = 0.0;
    }

    sprintf(logMessage,"ATTN: buildPostCalibrationTable_raisedCos finish" );
    EventLogEngine::append(logMessage);
    return true;

}

#define SINC(x)   ((FloatCompare<Float64>::equal(x,0.0) == false) ? (sin((Float64)x * M_PI) / ((Float64)x * M_PI)) : 1.0f)

// ##METHOD HEADER-START## ****************************************************
//
// ARGUMENTS:
//  INPUTS:
//   Band::Wavelength theBand
//
//  OUTPUTS:
//   None
//
// DESCRIPTION:
//  Calculate Resampling Table.
//  This calculation is only BAND SPECIFIC.
//
//*** ##METHOD HEADER-END## ***************************************************/
bool ScienceDataProcessor::buildResamplingTable(Band::Wavelength theBand)
{

    UInt32 binIndex;

    //mathematical protection of divide by zero
    BOOST::vector<Float64> userWavenumber;
    BOOST::vector<Float64> instrumentWavenumber;

    UInt32 binSize = InstrumentCharacteristic.
       dataPointsDecimatedInterferogram[theBand];

    Float64 userFirstWavenumber = correctionTable[0][theBand].
       getUserLowestWavenumber(theBand);
    Float64 instrLowWavenumber  = correctionTable[0][theBand].
       getinstrumentLowestWavenumber(theBand);

    //set size of matrix to rows=points in decimated interferogram(diff for
    //each band) and col=points in decimated interferogram(diff for each band)

    if( resamplingMatrix[theBand].size1() != binSize )
       resamplingMatrix[theBand].resize(binSize, binSize);

    userWavenumber.resize(binSize);
    instrumentWavenumber.resize(binSize);

    //using extension resampling matrix with increasing binsize to
    //theDecimationFactor*binSize
    UInt32 theDecimationFactor=InstrumentCharacteristic.
       decimationFactor[theBand];
    if ( CorrectionParameter.sincFlag )
    {
      theDecimationFactor=1;
    }

    for (binIndex = 0; binIndex < binSize; binIndex++)
    {
        userWavenumber[binIndex] =
           ((Float64)CorrectionParameter.edrDeltaSigma[theBand] * binIndex) +
              userFirstWavenumber;
        instrumentWavenumber[binIndex] =
           ((Float64)deltaSigma[theBand] * binIndex) + instrLowWavenumber;
    }

    Float64 deltaSigmaRatio = (Float64)deltaSigma[theBand] /
       (Float64)CorrectionParameter.edrDeltaSigma[theBand];

    for (binIndex = 0; binIndex < binSize; binIndex++)
    {
        for (UInt32 binIndexPrime = 0; binIndexPrime < binSize; binIndexPrime++)
        {
            Float64 common1 = (instrumentWavenumber[binIndex] -
               userWavenumber[binIndexPrime]) /
                  (Float64)CorrectionParameter.edrDeltaSigma[theBand];

            Float64 common2 = ((instrumentWavenumber[binIndex] -
                            userWavenumber[binIndexPrime]) /
                           (theDecimationFactor*binSize)) /
                              (Float64)deltaSigma[theBand];
            (resamplingMatrix[theBand])(binIndexPrime, binIndex) =
               deltaSigmaRatio * (SINC(common1) / SINC(common2));
        }
    }

    char logMessage[EVENT_LOG_LENGTH];
    sprintf(logMessage,"ATTN:  CMO Resampling deltaSigma %f",
       deltaSigma[theBand]);
    EventLogEngine::append(logMessage);
    return true;
}

// ##METHOD HEADER-START## ****************************************************
//
// ARGUMENTS:
//  INPUTS:
//     Band::Wavelength theBand
//     loat64 lowestWavenumberFrom
//     Float64 deltaSigmaFrom
//     Float64 lowestWavenumberTo
//     Float64 deltaSigmaTo
//     UInt32 bigNto
//  OUTPUTS
//     BOOST::matrix<Float64> &rMatrix  - must have the correct size
//
// DESCRIPTION:
//  Calculate the resampling or truncation matrix
//  This calculation is only BAND SPECIFIC.
//
//*** ##METHOD HEADER-END## ***************************************************/
bool ScienceDataProcessor::buildResamplingTable(Float64 lowestWavenumberFrom,
                            Float64 deltaSigmaFrom,  
                            Float64 lowestWavenumberTo, Float64 deltaSigmaTo,
                            UInt32 bigNto,
                            BOOST::matrix<Float64> &rMatrix)
{

    UInt32 matrixBinSize = rMatrix.size1();

    Float64 deltaSigmaRatio = (Float64)deltaSigmaFrom / deltaSigmaTo;
    Float64 common1, common2, wavenumberTo;
    UInt32 nLowestFrom, nFrom;

    nLowestFrom = ROUND(lowestWavenumberFrom / deltaSigmaFrom);
    for (UInt32 binIndexFrom = 0; binIndexFrom < matrixBinSize; binIndexFrom++)
    {
        nFrom = nLowestFrom + binIndexFrom;
        for (UInt32 binIndexTo = 0; binIndexTo < matrixBinSize; binIndexTo++)
        {
            wavenumberTo = lowestWavenumberTo +
                           (Float64)binIndexTo * deltaSigmaTo;
            common1 = wavenumberTo/deltaSigmaTo -
                          (Float64)nFrom * deltaSigmaRatio;
            common2 = common1/(Float64)bigNto;

            rMatrix(binIndexTo, binIndexFrom) = deltaSigmaRatio *
                      (SINC(common1) / SINC(common2));
        }
    }

    return true;
}

// ##METHOD HEADER-START## ****************************************************
//
// ARGUMENTS:
//  INPUTS:
//
//  OUTPUTS:
//
// DESCRIPTION:
//  Build the apodization function table to be applied to interferograms
//  Raised cosine to taper the extended interferogram points
//
//*** ##METHOD HEADER-END## ***************************************************/
void ScienceDataProcessor::buildInterferogramApodizationTable()
{
  // Minimum total extended interferogram data points beyond 0.8 cm OPD
  Int32 min_extraPoints = 12;

  for(UInt32 theBand = 0; theBand < Band::TOTAL_BANDS; theBand++)
  {
    UInt32 igmLen = InstrumentCharacteristic.dataPointsDecimatedInterferogram[theBand];
    Float64 delta_x = 0.5*(Spectra::LASER_WAVELENGTH[Band::LONGWAVE])*
                  InstrumentCharacteristic.decimationFactor[theBand]*0.0000001;
    Float64 OPD_max = (Float64)igmLen*delta_x;
    Float64 OPD_min = 1.0/CorrectionParameter.edrDeltaSigma[theBand];
    Int32 extra_points = floor((OPD_max - OPD_min)/delta_x);

    if(extra_points >= min_extraPoints)
    {
       if(interferogramApodizationTable[theBand].size() != igmLen)
         interferogramApodizationTable[theBand].resize(igmLen);
       for(UInt32 j = 0; j < igmLen; j++)
          interferogramApodizationTable[theBand](j) = 1.0;

       UInt32 half = extra_points/2;
       Float64 r = half+1.0;
       Float64 cosAp;
       for(UInt32 j = 0; j < half; j++)
       {
          cosAp = 0.5*(1.0 - cos(M_PI*(Float64)(j+1)/r));
          interferogramApodizationTable[theBand](j) = cosAp;
          interferogramApodizationTable[theBand](igmLen-j-1) = cosAp;
       }
     }
   }
}

// ##METHOD HEADER-START## ****************************************************
//
// ARGUMENTS:
//  INPUTS:
//   Band::Wavelength theBand
//   UInt32 interferogramSize
//
//  OUTPUTS:
//     Float64 maxPathDifference
//
// DESCRIPTION:
//  Calculates the user apodization table.
//  This calculation is only BAND SPECIFIC.
//
//*** ##METHOD HEADER-END## ***************************************************/

Float64 ScienceDataProcessor::computeMaxPathDifference(Band::Wavelength theBand,
                                 UInt32 interferogramSize)
{

    Float64 maxPathDifferentialResult =
                          ((Float64)(Spectra::LASER_WAVELENGTH[theBand] / 2.0) *
                          (Float64)interferogramSize / 2.0)*0.0000001;

    return maxPathDifferentialResult;
}

// ##METHOD HEADER-START## ****************************************************
//
// ARGUMENTS:
//  INPUTS:
//   Band::Wavelength theBand
//
//  OUTPUTS:
//   None
//
// DESCRIPTION:
//  Calculates the user apodization table.
//  This calculation is only BAND SPECIFIC.
//
//*** ##METHOD HEADER-END## ***************************************************/
bool ScienceDataProcessor::buildUserApodizationTable(Band::Wavelength theBand)
{
    UInt32 binSize =
       InstrumentCharacteristic.dataPointsDecimatedInterferogram[theBand];

    //set matrix size for each band
    apodizationMatrix[theBand].resize(binSize, binSize);

    for (UInt32 binIndex = 0; binIndex < binSize; binIndex++)
    {
        //populate main diagonal with 1's
        (apodizationMatrix[theBand])(binIndex,binIndex) = 1.0;
    }
    return true;
}

// ##METHOD HEADER-START## ****************************************************
//
// ARGUMENTS:
//  INPUTS:
//   None
//
//  OUTPUTS:
//   None.
//
// DESCRIPTION:
//  Population of high access frequency domain data
//
//*** ##METHOD HEADER-END## ***************************************************
void ScienceDataProcessor::calculateWavenumberBinSpacing(
   Band::Wavelength theBand)
{
    VideoData::EXTRACTION_RECORD theExtractionRec =
       TelemetryProcessor::instance()->theVideoData.getExtractionRecord(
          (VideoData::Wavelength)theBand);

    //recalculate for diagnostic
    if(TelemetryProcessor::instance()->
       getVideoMode() == TelemetryProcessor::DIAGNOSTIC_BURST)
    {
        UInt32 undecimatedInterferogramSize =
               TelemetryProcessor::instance()->theDiagnosticData.getSampleCount(
               (DiagnosticData::Wavelength)theBand);
        maxPathDifferential[theBand] =
               computeMaxPathDifference(theBand, undecimatedInterferogramSize);

        deltaSigma[theBand] = 1.0/(2.0 * maxPathDifferential[theBand]);

        lowestWavenumber[theBand] = 0;

    }
    //recalculate wavenumber bin spacing
    else
    {
       UInt32 undecimatedInterferogramSize = (theExtractionRec.complexSamples -
                 (2 * AlgorithmParameter.numberOpdOverscanSamples)) *
                 theExtractionRec.decimationRate;
        maxPathDifferential[theBand] = computeMaxPathDifference(
                 theBand, undecimatedInterferogramSize);

        deltaSigma[theBand] = 1.0/(2.0 * maxPathDifferential[theBand]);
        UInt32 k = floor( (desiredBandCenter[theBand] - (deltaSigma[theBand] *
                   (Float64)(theExtractionRec.complexSamples -
                     (2 * AlgorithmParameter.numberOpdOverscanSamples))) / 2.0)
                        / deltaSigma[theBand]);
        unfoldIndex[theBand] = k;
        lowestWavenumber[theBand] = (Float64) k * deltaSigma[theBand];
    }

    // to handle serial run, keep the lowestWavenumber the same
    // in the case of foldIndex changed and make the reference spectra
    // consistency
    char logText1[EVENT_LOG_LENGTH];
    sprintf(logText1,"NOTE: %s Laser Wavelength changed to %f"
              " and first wavelength is %f,  unfold point is %d, "
	      " config laser wavelength is %f",
	    BAND[theBand].c_str(), Spectra::LASER_WAVELENGTH[theBand],
	    lowestWavenumber[theBand], unfoldIndex[theBand],
	    InstrumentCharacteristic.defaultLaserWavelength);
    EventLogEngine::append(logText1);

    // laser wavelength is not equal to initial value, and changed
    // then save the foldIndex at first time.
    // K. Zhang,ADR-11194, Use InstrumentCharacteristic.defaultLaserWavelength as initial value,
    // instead of the hard-coded default laser wavelength.
    if( fabs(Spectra::LASER_WAVELENGTH[theBand]
                     - InstrumentCharacteristic.defaultLaserWavelength) > 1e-6 &&
            !savedfoldIndex_flag[theBand])
    {
        savedUnfoldIndex[theBand] = unfoldIndex[theBand];
        savedfoldIndex_flag[theBand] = true;
    }
    // keep the lowestWavenumber the same
    char logText2[EVENT_LOG_LENGTH];
    sprintf(logText2,"NOTE: %s first wavelength changed to %f "
       "saved unfold point is %d and savedflag is %d",
       BAND[theBand].c_str(), lowestWavenumber[theBand],
                              savedUnfoldIndex[theBand],
                           savedfoldIndex_flag[theBand]);
    EventLogEngine::append(logText2);

    if ( savedfoldIndex_flag[theBand] == true )
    {
       unfoldIndex[theBand] = savedUnfoldIndex[theBand];
       foldIndex[theBand] = (UINT)savedUnfoldIndex[theBand]%
            InstrumentCharacteristic.dataPointsDecimatedInterferogram[theBand]
               + InstrumentCharacteristic.foldIndexOffset[theBand];
       lowestWavenumber[theBand] = (Float64) unfoldIndex[theBand] *
                                             deltaSigma[theBand];
    }
    // K. Zhang, ADR-11194, 03/18/2025
    // In case of Neon lamp mis-firing, calculate foldIndex anyway,
    // but not update savedfoldIndex_flag.
    if (fabs(Spectra::LASER_WAVELENGTH[theBand]
	     - InstrumentCharacteristic.defaultLaserWavelength) < 1e-6 &&
	savedfoldIndex_flag[theBand] == false)
    {
      foldIndex[theBand] = (UINT)unfoldIndex[theBand]%
           InstrumentCharacteristic.dataPointsDecimatedInterferogram[theBand]
              + InstrumentCharacteristic.foldIndexOffset[theBand];
    }

    char logText3[EVENT_LOG_LENGTH];
    sprintf(logText3,"NOTE: Final %s first wavelength is %f "
                     " and fold point is %d and deltaSigma is %f",
       BAND[theBand].c_str(), lowestWavenumber[theBand],
                              foldIndex[theBand],
                              deltaSigma[theBand]);
    EventLogEngine::append(logText3);

    buildFirFilterGainTable(theBand);

    // initialize or re-initialize instrument first wavenumber and delta sigma
    // for the correctionTable object
    for (int fovIndex = 0; fovIndex < MAX_FOV; fovIndex++)
    {
      correctionTable[fovIndex][theBand].
         setInstrumentLowestWavenumber(theBand, lowestWavenumber[theBand]);
      correctionTable[fovIndex][theBand].
         setInstrumentDeltaSigma(theBand, deltaSigma[theBand]);
    }

    // Build interferogram apodization table
    if(CorrectionParameter.calibrationOrder == RadCalFirst_FPfINVsaPf_bigN)
       buildInterferogramApodizationTable();

}

// ##METHOD HEADER-START## ****************************************************
//
// ARGUMENTS:
//  INPUTS:
//   Float64 newWavelength
//
//  OUTPUTS:
//   None
//
// DESCRIPTION:
//  Update the value of laser wavelength defined in different member objects.
//
//  K. Zhang, ADR-11194, do update only if static savedfoldIndex_flag is false.
//
//*** ##METHOD HEADER-END## ***************************************************/
void ScienceDataProcessor::setDefaultLaserWavelength(Float64 newWavelength)
{
    InstrumentCharacteristic.defaultLaserWavelength = newWavelength;

    if(savedfoldIndex_flag[Band::LONGWAVE]  == false ||
       savedfoldIndex_flag[Band::MIDWAVE]   == false ||
       savedfoldIndex_flag[Band::SHORTWAVE] == false)
    {
	theEngineeringCalibrationRecord.setAverageMetrologyWavelength(newWavelength);
	Spectra::configureLaserWavelength(newWavelength);
    }
}

// ##METHOD HEADER-START## ****************************************************
//
// ARGUMENTS:
//  INPUTS:
//   unsigned short packetType
//
//  OUTPUTS:
//   Spectra* theSpectra - The processed spectra
//
// DESCRIPTION:
//   This operation holds the intelligence to route any RDR packet to assistant
//   class processing
//
//   Throws ProSdrCrisException if there is an error updating the CMO
//
//*** ##METHOD HEADER-END## ***************************************************
Spectra* ScienceDataProcessor::processRawDataRecord
(
    UInt16 packetType,
    bool   missingPacket,
    CCSDSFormat* theTelemetryFormat,
    CrisSdrCoefficients* cfgParmsPtr,
    CrisSdrAlgDataType *algDataPtr,
    UInt32 scanIdx
)
{
    Int32 sceneIndex;

    static TelemetryProcessor::BurstModes lastVideoMode[Band::TOTAL_BANDS] =
        {TelemetryProcessor::NO_VIDEO_DATA,
         TelemetryProcessor::NO_VIDEO_DATA,
         TelemetryProcessor::NO_VIDEO_DATA};

    static UINT detector;
    static UINT band;
    Spectra* theSpectra = 0;

    static Int32 sciPacketCount = 0;

    //identify packetType
    TelemetryProcessor::instance()->rdrLookup(packetType, &detector, &band);

    //identify incoming packet content
    if((band != VideoData::Unknown) && (detector > 0))
    {
        //Video mode has changed, update wavenumber bin spacing
        if (lastVideoMode[band] !=
              TelemetryProcessor::instance()->getVideoMode() &&
            !missingPacket)
        {
            calculateWavenumberBinSpacing((Band::Wavelength)band);

            lastVideoMode[band] =
               TelemetryProcessor::instance()->getVideoMode();
        }

        // can the same packetType be used twice
        // in an Interferogram constructor
        Interferogram theInterferogram(packetType);
        theInterferogram.setMissingData(missingPacket);

        SceneElement thisFOV = theInterferogram.getFieldOfView();
        Band::Wavelength thisBand = theInterferogram.getBand();
        FieldOfRegard  thisFOR = theInterferogram.getFieldOfRegard();
        SweepDirection thisPSD = theInterferogram.getPorchSwingDirection();

        // keep count of each interferogram
        sequenceCount[thisFOV][thisBand][thisFOR][thisPSD]++;

        // keep count of invalid raw interferograms
        if(theInterferogram.getInvalidData())
        {
            invalidInterferogramCount[thisFOV][thisBand]++;
        }

        Spectra* newSpectra;

        // apodize the interferograms
        if(CorrectionParameter.calibrationOrder == RadCalFirst_FPfINVsaPf_bigN)
        {
           theInterferogram.apodize_interferogram(interferogramApodizationTable[thisBand]);
        }

        //select calibration method
        if(GeneralParameter.spectralCalMethod == CalibrationType_Complex)
        {
            newSpectra = new CalibratedSpectra(theInterferogram,
                         foldIndex[thisBand], lowestWavenumber[thisBand],
                         deltaSigma[thisBand],
                         CorrectionParameter.calibrationOrder,
                         cfgParmsPtr);// pass PCT coefficients to spectra
        }
        else
        {
            newSpectra = new UncalibratedSpectra(theInterferogram,
                         foldIndex[thisBand],
                         lowestWavenumber[thisBand], deltaSigma[thisBand],
                         CorrectionParameter.calibrationOrder,
                         cfgParmsPtr);// pass PCT coefficients to spectra
        }

        // FCE update
        //this is an interferogram packet
        theSpectra = addSpectra(newSpectra, cfgParmsPtr);
    }
    else
    {
        //this is some kind of housekeeping packet
        if (packetType == ENGINEERING_CALIBRATION_FRAME_TYPE)
        {
            ++engPacketCount;

            //this is a 4 min packet
            const EngCalRec_ScienceTelemetryConversionCoefficients*
               theConvCoeff = &theEngineeringCalibrationRecord.
                  getEngCalRec_ScienceTelemetryConversionCoefficients();
            const EngCalRec_ScienceTelemetryLimits* theSciTeleLimits =
               &theEngineeringCalibrationRecord.
                  getEngCalRec_ScienceTelemetryLimits();

            //checksum changed
            char tmpString1[EVENT_LOG_LENGTH];
            sprintf(tmpString1,
               "NOTE:: 4 min cal packet checksum last is %d, current is %d.",
                   theEngineeringCalibrationRecord.getLastCheckSumValue(),
                      TelemetryStateConverter::get4minuteUpdateValue());
            EventLogEngine::append(tmpString1);

            if ((Int32)TelemetryStateConverter::get4minuteUpdateValue() !=
               theEngineeringCalibrationRecord.getLastCheckSumValue())
            {
                char tmpString[EVENT_LOG_LENGTH];
                sprintf(tmpString,
                   "NOTE:: 4 min cal packet (v%d) checksum changed.",
                        TelemetryStateConverter::getEngPacketVersion());
                EventLogEngine::append(tmpString);

                theEngineeringCalibrationRecord.setLastCheckSumValue(
                     (Int32)TelemetryStateConverter::get4minuteUpdateValue());

                // added the calibrationOrfer parameter
                // refreshData to update all the calibration parameters
                if (theEngineeringCalibrationRecord.refreshData(packetType,
                   theTelemetryFormat, CorrectionParameter.calibrationOrder))
                {
                    // Update Engineering Packet
                    if (engPacketCount > 0 )
                    {
                       numEngBuilds_++;
                       scanWhichCausedEngRebuild_ = scanIdx_;

                       char tmpString2[EVENT_LOG_LENGTH];
                       sprintf(tmpString2,
                          "NOTE: scan %d contain new Engpacket.",
                             scanWhichCausedEngRebuild_);
                       EventLogEngine::append(tmpString2);

                       EventLogEngine::append(
                          "NOTE: Saving EngineeringCalibrationRecord "
                             "parameters started");
                       refreshTime(theTelemetryFormat);

                       theEngineeringCalibrationRecord.
                          setEngCalLaserWavelength();

                       sprintf(tmpString2,
                               "NOTE: IETTime: %lld and LaserWavelength: %f.",
                             theEngineeringCalibrationRecord.getIETTime(),
                                theEngineeringCalibrationRecord.
                                   getEngCalRec_LaserMetrologyInfo().
                                      getLaserWavelength());
                       EventLogEngine::append(tmpString2);

                       saveSerializedEngPkt();
                       engPkt_Loaded = true;
                       EventLogEngine::append(
                          "NOTE: Saving EngineeringCalibrationRecord "
                          "parameters complete");
                    }

                    //Engineering packet detected, check if need to update
                    //correction matrix
                    theILSParametersRecord.
                       refreshData(packetType, theTelemetryFormat, true);
                    
                    if ((engPacketCount > 0 && 
                       ((theILSParametersRecord.getEngCalRec_ILSCurveFitParameters().getValuesChanged() ||
                        theILSParametersRecord.getEngCalRec_ILSFOVParameters().getValuesChanged()) &&
                        !CorrectionParameter.useSavedMatrices)) || 
                        DMS_Data_Inputs->CMOPtr_in == 0)
                    {

                        if (theILSParametersRecord.
                           getEngCalRec_ILSCurveFitParameters().
                              getValuesChanged() ||
                                 theILSParametersRecord.
                                    getEngCalRec_ILSFOVParameters().
                                       getValuesChanged())
                        {
                           DEBUG_HIGH((ProCmnLogger::getLogger()),
                              "Change in ILS FOV or Curve parameters was detected");
                        }

                       // Do not rebuild CMO during maneuvers, unless we do
                       // not have a CMO at all
                        if (!DMS_Data_Inputs->isDuringManeuver ||
                            DMS_Data_Inputs->CMOPtr_in == 0)
                        {
                             theILSParametersRecord.refreshData(
                                packetType,theTelemetryFormat, false);
                             Int8 retVal = refresh_Invsa_CorrectionMatrix(false);

                             if(retVal <= 0)
                             {
                                //send a debug message to INF letting user know
                                //that 4-min CMO update failed
                                DEBUG_HIGH((ProCmnLogger::getLogger()),
                                   "CMO update failed!  processing will stop.");

                                //program should stop if no inverse SA
                                throw ProSdrCrisException("CMO update failed",
                                   ProCmnCallerID::ANONYMOUS );
                             }
                             else if (retVal > 0)
                             {
                                //we don't need to reload CMO and eng packet
                                ilsCorrection_Loaded = true;
                                refreshCorrectionMatrix(true);
                             }
                       }
                       else
                       {
                          std::ostringstream msg;
                          msg << "ATTN:  Engineering packet data indicates"
                              << " rebuild of CMO necessary, but data occurs"
                              << " during a maneuver, so CMO will"
                              << " NOT rebuild.";
                          (void)EventLogEngine::append(msg.str());
                       }
                    }
                    // For case that has invSA but without EngPkt as inputs
                    // files
                    if ( (DMS_Data_Inputs->CMOPtr_in != 0) && 
                         (DMS_Data_Inputs->CMOPtr_in != DM_BADADDRESS) &&
                         engPkt_Loaded && ilsCorrection_Loaded )
                    {
                       EventLogEngine::append(
                          "NOTE: refreshCorrectionMatrix if have CMO with new "
                          "Engpkt");
                       refreshCorrectionMatrix(true);
                    }


                    //Echos data from EngCalRec into SciCalRec
                    theScienceCalibrationRecord.
                       setIctPrt1Ro(theConvCoeff->getIctPrt1Ro());
                    theScienceCalibrationRecord.
                       setIctPrt1A(theConvCoeff->getIctPrt1A());
                    theScienceCalibrationRecord.
                       setIctPrt1B(theConvCoeff->getIctPrt1B());
                    theScienceCalibrationRecord.
                       setIctPrt2Ro(theConvCoeff->getIctPrt2Ro());
                    theScienceCalibrationRecord.
                       setIctPrt2A(theConvCoeff->getIctPrt2A());
                    theScienceCalibrationRecord.
                       setIctPrt2B(theConvCoeff->getIctPrt2B());
                    theScienceCalibrationRecord.
                       setIctLowRangeCalibrationResistorRo(theConvCoeff->
                          getIctLowRangeCalibrationResistorRo());
                    theScienceCalibrationRecord.
                       setIctLowRangeCalibrationResistorA(theConvCoeff->
                          getIctLowRangeCalibrationResistorA());
                    theScienceCalibrationRecord.
                       setIctHighRangeCalibrationResistorRo(theConvCoeff->
                          getIctHighRangeCalibrationResistorRo());
                    theScienceCalibrationRecord.
                       setIctHighRangeCalibrationResistorA(theConvCoeff->
                          getIctHighRangeCalibrationResistorA());
                    theScienceCalibrationRecord.
                       setIctCalibrationResistorRtdRo(theConvCoeff->
                          getIctCalibrationResistorRtdRo());
                    theScienceCalibrationRecord.
                       setIctCalibrationResistorRtdA(theConvCoeff->
                          getIctCalibrationResistorRtdA());
                    theScienceCalibrationRecord.
                       setLaserDiodeTemperatureSlope(theConvCoeff->
                          getLaserDiodeTemperatureSlope());
                    theScienceCalibrationRecord.
                       setLaserDiodeBiasCurrentSlope(theConvCoeff->
                          getLaserDiodeBiasCurrentSlope());
                    theScienceCalibrationRecord.
                       setBeamsplitterTempIntercept(theConvCoeff->
                          getBeamsplitterTempIntercept());
                    theScienceCalibrationRecord.
                       setBeamsplitterTempSlope(theConvCoeff->
                          getBeamsplitterTempSlope());
                    theScienceCalibrationRecord.
                       setScanMirrorTempIntercept(theConvCoeff->
                          getScanMirrorTempIntercept());
                    theScienceCalibrationRecord.
                       setScanMirrorTempSlope(theConvCoeff->
                          getScanMirrorTempSlope());
                    theScienceCalibrationRecord.
                       setScanBaffleTempIntercept(theConvCoeff->
                          getScanBaffleTempIntercept());
                    theScienceCalibrationRecord.
                       setScanBaffleTempSlope(theConvCoeff->
                          getScanBaffleTempSlope());
                    theScienceCalibrationRecord.
                       setOmaStructureTemp1Intercept(theConvCoeff->
                          getOmaStructureTemp1Intercept());
                    theScienceCalibrationRecord.
                       setOmaStructureTemp1Slope(theConvCoeff->
                          getOmaStructureTemp1Slope());
                    theScienceCalibrationRecord.
                       setOmaStructureTemp2Intercept(theConvCoeff->
                          getOmaStructureTemp2Intercept());
                    theScienceCalibrationRecord.
                       setOmaStructureTemp2Slope(theConvCoeff->
                          getOmaStructureTemp2Slope());
                    theScienceCalibrationRecord.
                       setTelescopeTempSlope(theConvCoeff->
                          getTelescopeTempSlope());
                    theScienceCalibrationRecord.
                       setStage1CoolerTempSlope(theConvCoeff->
                          getStage1CoolerTempSlope());
                    theScienceCalibrationRecord.
                       setStage2CoolerTempSlope(theConvCoeff->
                          getStage2CoolerTempSlope());
                    theScienceCalibrationRecord.
                       setStage3CoolerTempSlope(theConvCoeff->
                          getStage3CoolerTempSlope());
                    theScienceCalibrationRecord.
                       setStage4CoolerTempSlope(theConvCoeff->
                          getStage4CoolerTempSlope());
                    theScienceCalibrationRecord.
                       setCrossTrackSsmPointingErrorIntercept(theConvCoeff->
                          getCrossTrackSsmPointingErrorIntercept());
                    theScienceCalibrationRecord.
                       setCrossTrackSsmPointingErrorSlope(theConvCoeff->
                          getCrossTrackSsmPointingErrorSlope());
                    theScienceCalibrationRecord.
                       setInTrackSsmPointingErrorIntercept(theConvCoeff->
                          getInTrackSsmPointingErrorIntercept());
                    theScienceCalibrationRecord.
                       setInTrackSsmPointingErrorSlope(theConvCoeff->
                          getInTrackSsmPointingErrorSlope());

                    theScienceCalibrationRecord.
                       setIctTemp1DriftLimit(theSciTeleLimits->
                          getIctTemp1DriftLimit());
                    theScienceCalibrationRecord.
                       setIctTemp2DriftLimit(theSciTeleLimits->
                          getIctTemp2DriftLimit());
                    theScienceCalibrationRecord.
                       setScanMirrorTempDriftLimit(theSciTeleLimits->
                          getScanMirrorTempDriftLimit());
                    theScienceCalibrationRecord.
                       setBeamsplitterTemp1DriftLimit(theSciTeleLimits->
                          getBeamsplitterTemp1DriftLimit());
                    theScienceCalibrationRecord.
                       setScanBaffleTempDriftLimit(theSciTeleLimits->
                          getScanBaffleTempDriftLimit());
                    theScienceCalibrationRecord.
                       setOmaStruct1TempDriftLimit(theSciTeleLimits->
                          getOmaStruct1TempDriftLimit());
                    theScienceCalibrationRecord.
                       setOmaStruct2TempDriftLimit(theSciTeleLimits->
                          getOmaStruct2TempDriftLimit());
                    theScienceCalibrationRecord.
                       setTelescopeTempDriftLimit(theSciTeleLimits->
                          getTelescopeTempDriftLimit());
                    theScienceCalibrationRecord.
                       setStage1CoolerTempDriftLimit(theSciTeleLimits->
                          getStage1CoolerTempDriftLimit());
                    theScienceCalibrationRecord.
                       setStage2CoolerTempDriftLimit(theSciTeleLimits->
                          getStage2CoolerTempDriftLimit());
                    theScienceCalibrationRecord.
                       setStage3CoolerTempDriftLimit(theSciTeleLimits->
                          getStage3CoolerTempDriftLimit());
                    theScienceCalibrationRecord.
                       setStage4CoolerTempDriftLimit(theSciTeleLimits->
                          getStage4CoolerTempDriftLimit());
                    theScienceCalibrationRecord.
                       setLaserDiodeWavelengthDriftLimit(theSciTeleLimits->
                          getLaserDiodeWavelengthDriftLimit());

                    Spectra::configureTimeStampBias(
                       TelemetryStateConverter::get4minuteTimestampBiasValue());

                    if(CalibrationParameter.instrumentTemperatureOrigin ==
                       INSTRUMENT_TLM
                            && CalibrationParameter.
                               suppressSsmBaffleProfile == false)
                    {
                        if(theEngineeringCalibrationRecord.
                           getEngCalRec_ICTEnvironmentalModel().
                              getEngCalBaffleTempOffsetDecimatedY().size() ==
                                 static_cast<UInt32>(NUM_ECM_BAFFLE_PTS))
                        {
                            buildBaffleTemperatureOffset(
                               (&theEngineeringCalibrationRecord.
                                  getEngCalRec_ICTEnvironmentalModel())->
                                     getEngCalBaffleTempOffsetDecimatedY(),
                                    (&theEngineeringCalibrationRecord.
                                       getEngCalRec_ICTEnvironmentalModel())->
                                          getEngCalOrbitalPeriod());
                        }
                        else
                        {
                            EventLogEngine::append(
                               "EngCal format doe not support USER Requested "
                                  "ECM Baffle Compensation");
                        }
                    }

                }
                else
                {
                    (void)EventLogEngine::append(
                     "WARN:: Unable to refresh all values of 4 minute packet");
                }
            }
            else
            {
                //4minute packet had same content as last time
                (void)EventLogEngine::append(
                   "NOTE:: 4 min cal packet unchanged.");
                // For case that has invSA and EngPkt as inputs files
                if ( engPkt_Loaded && ilsCorrection_Loaded )
                {
                   EventLogEngine::append("NOTE: refreshCorrectionMatrix if "
                                          "have inSA and Engpkt as inputs");
                   refreshCorrectionMatrix(true);
                }

            }
        }
        else if (packetType == SCIENCE_CALIBRATION_FRAME_TYPE
                    && GeneralParameter.spectralCalMethod ==
                       CalibrationType_Complex)
        {
            //this is a 8 sec packet
            if(theScienceCalibrationRecord.refreshData(
                        packetType,
                        algDataPtr,
                        cfgParmsPtr,
                        scanIdx))
            {
                if (CalibrationParameter.dsTemperatureOrigin != DS_CONFIG)
                {
                    processEnvironmentalData(TelemetryProcessor::instance()->
                       getLastFrameTime());
                }
                else
                {
                    //get from configuration data
                    (void)theTargetTempCalibrationRecord.getProcessingPeriod().
                       addTemperature((GeneralParameter.
                          instrumentLocation == BENCH) ?
                                (TestEnvironmentParameter.dsTempBench) :
                               (TestEnvironmentParameter.dsTempChamber));
                }

                for(Int32 thisFOV = 0; thisFOV  <  MAX_FOV; thisFOV++)
                {
                    for(Int32 spectralBand = 0;
                       spectralBand < Band::TOTAL_BANDS; spectralBand++)
                    {
                        Spectra* hotRadiance = calculateHotRadiance(
                           (SceneElement)thisFOV,
                              (Band::Wavelength)spectralBand);
                        Spectra* coldRadiance;

                        if (CalibrationParameter.useDeepSpaceRadiance)
                        {
                            coldRadiance = calculateColdRadiance(
                               (SceneElement)thisFOV,
                                  (Band::Wavelength)spectralBand);
                        }
                        else
                        {
                            coldRadiance = new UncalibratedSpectra();

                            coldRadiance->
                               setFieldOfView((SceneElement) thisFOV);
                            coldRadiance->setBand(
                               (Band::Wavelength)spectralBand);
                            coldRadiance->setSize(
                               TelemetryProcessor::instance()->
                                  theVideoData.getExtractionRecord(
                                     (VideoData::Wavelength)spectralBand).
                                       complexSamples - (2 * AlgorithmParameter.
                                           numberOpdOverscanSamples));
                        }
                        processingPeriod.
                           addTargetRadianceCurves(hotRadiance, coldRadiance);
                    }
                }

                const EngCalRec_PolarizationCalibrationWavenumbers&
                   engPolarizationInfo = theEngineeringCalibrationRecord.
                      getEngCalRec_PolarizationCalibrationWavenumbers();

                //Schedule Polarization Correction
                for (sceneIndex = 1; sceneIndex <= EarthScene30; sceneIndex++)
                {
                    for (Int32 bandIndex = 0;
                       bandIndex < Band::TOTAL_BANDS; bandIndex++)
                    {
                        if(engPolarizationInfo.getValuesChanged() &&
                           (engPolarizationInfo.
                              getChangeWavenumberPosition() ||
                            engPolarizationInfo.
                               getChangedPolarizationChange(sceneIndex)))
                        {
                            //create new polarization spectra
                            Spectra* polarSpectra = new UncalibratedSpectra(
                               FOV1, (Band::Wavelength) bandIndex,
                                  (FieldOfRegard)sceneIndex, BOTH_DIRECTIONS);

                            buildPolarizationCurve(polarSpectra,
                               (FieldOfRegard)sceneIndex,
                                  (Band::Wavelength)bandIndex);

                            //schedule new correction
                            processingPeriod.schedulePolarizationCorrection(
                               polarSpectra, (FieldOfRegard)sceneIndex,
                                  (Band::Wavelength)bandIndex);
                        }
                        else
                        {
                            //continue with current correction
                            processingPeriod.schedulePolarizationCorrection(0,
                               (FieldOfRegard)sceneIndex,
                                  (Band::Wavelength)bandIndex);
                        }
                    }
                }

                //Polarization change can only occur from a 4min packet change.
                //Reset valuesChanged flag so that polarization change will not
                //be recalculated until next 4min packet change.
                theEngineeringCalibrationRecord.
                   resetPolarizationValuesChanged();

                //Schedule geolocation information
                for (sceneIndex = 0; sceneIndex < TOTAL_SCENES; sceneIndex++)
                {
                    for (int fovIndex = 0; fovIndex < MAX_FOV; fovIndex++)
                    {
                        if (sciPacketCount == 0 ||
                           theScienceCalibrationRecord.getServoValuesChanged())
                        {
                            //create new geolocation spectra
                            Spectra* geoSpectra = new UncalibratedSpectra(
                               (SceneElement)fovIndex, Band::TOTAL_BANDS,
                                  (FieldOfRegard)sceneIndex, BOTH_DIRECTIONS);

                            processingPeriod.scheduleGeolocation(
                               calculateGeometricCalibration(geoSpectra),
                                  (FieldOfRegard)sceneIndex,
                                     (SceneElement)fovIndex);
                        }
                        else
                        {
                            processingPeriod.scheduleGeolocation(0,
                               (FieldOfRegard)sceneIndex,
                                  (SceneElement)fovIndex);
                        }

                    }
                }
                theScienceCalibrationRecord.resetServoValuesChanged();
            }
            else
            {
                (void)EventLogEngine::append(
                   "WARN:: Failed to refresh all 8 sec data values.");
            }


            //Apply Correction Matrix
            if(GeneralParameter.spectralCalMethod == CalibrationType_Complex)
            {
                //when user has request any correction contribution
                if ((CorrectionParameter.apodizationType != NONE) ||
                    (CorrectionParameter.applyIlsResidualEffectCorrection) ||
                    (CorrectionParameter.
                       applyPostCalibrationFilterMatrixCorrection) ||
                    (CorrectionParameter.applyResamplingMatrix) ||
                    (CorrectionParameter.applyIlsFovEffectsCorrection))
                {
                    //register new 8 seconds correction value
                    for (int fovIndex = 0; fovIndex < MAX_FOV; fovIndex++)
                    {
                        for(int bandIndex = 0;
                           bandIndex < Band::TOTAL_BANDS; bandIndex++)
                        {
                            //register receipt
                            scienceCount[fovIndex][bandIndex]
                               [InstrumentCharacteristic.
                                  hotReferenceIdentifer][FORWARD]++;
                            scienceCount[fovIndex][bandIndex]
                               [InstrumentCharacteristic.
                                  hotReferenceIdentifer][REVERSE]++;
                        } // for every band
                    } // for every fov
                }
                else
                {
                    //no corrections requested
                    for (int fovIndex = 0; fovIndex < MAX_FOV; fovIndex++)
                    {
                        for(int bandIndex = 0;
                           bandIndex < Band::TOTAL_BANDS; bandIndex++)
                        {
                            //register receipt
                            scienceCount[fovIndex][bandIndex]
                               [InstrumentCharacteristic.hotReferenceIdentifer]
                                  [FORWARD]++;
                            scienceCount[fovIndex][bandIndex]
                               [InstrumentCharacteristic.hotReferenceIdentifer]
                                  [REVERSE]++;
                        }
                    }
                }
            }
            else
            {
                // no calibration was performed
            }

            sciPacketCount++;
        }
        else
        {
            //packet does not contain content of interest
        }
    }

    return theSpectra;
}


// ##METHOD HEADER-START## ****************************************************
//
// ARGUMENTS:
//  INPUTS:
//   SYSTEMTIME theSpectraFrameTime - the timestamp of the last read packet
//
//  OUTPUTS:
//   None
//
// DESCRIPTION:
//  Read external calibration temperatures from a file
//
//*** ##METHOD HEADER-END## ***************************************************
bool ScienceDataProcessor::processEnvironmentalData(
   const SYSTEMTIME&) // theSpectraFrameTime
{
    //Read from selected files to find a timestamp that is within the
    //tolerance specified in the config file.
    bool foundTimeFrame = false;

    Float64 externalTargetTemp = 0.0;

    if (foundTimeFrame)
    {
        if (CalibrationParameter.dsTemperatureOrigin == DS_CCS)
        {
            //CCS files are in Celsius, converting to Kelvin
            externalTargetTemp += CONVERT_C_TO_K;

            //Add CCS DS temperature to target temperature calibration record
            theTargetTempCalibrationRecord.setCcsDsTemp(externalTargetTemp);
        }
        else
        {
            //Add CTC DS temperature to target temperature calibration record
            theTargetTempCalibrationRecord.setCtcDsTemp(externalTargetTemp);
        }

        (void)theTargetTempCalibrationRecord.addTemperature(externalTargetTemp);

        //register receipt
        for (int fovIndex = 0; fovIndex < MAX_FOV; fovIndex++)
        {
            for(int bandIndex = 0; bandIndex < Band::TOTAL_BANDS; bandIndex++)
            {
                //register receipt
                targetCount[fovIndex][bandIndex][InstrumentCharacteristic.
                   coldReferenceIdentifer][FORWARD]++;
                targetCount[fovIndex][bandIndex][InstrumentCharacteristic.
                   coldReferenceIdentifer][REVERSE]++;
            }
        }
    }
    else
    {
        if (CalibrationParameter.dsTemperatureOrigin == DS_CCS)
        {
            //Average of the temperatures from current and previous
            //space target temperatures
            theTargetTempCalibrationRecord.setCcsDsTemp(
               theTargetTempCalibrationRecord.getProcessingPeriod().
                  calculateAverageTemperature());
        }
        else
        {
            //Average of the temperatures from current and previous
            //space target temperatures
            theTargetTempCalibrationRecord.setCtcDsTemp(
               theTargetTempCalibrationRecord.getProcessingPeriod().
                  calculateAverageTemperature());
        }

        EventLogEngine::append(
           "ATTN:  Current Space Target Temperature readings "
           "are outside the time tolerance");
        (void)theTargetTempCalibrationRecord.markMissingCal();
    }
    return true;
}


// ##METHOD HEADER-START## ****************************************************
//
// ARGUMENTS:
//  INPUTS:
//   None
//
//  OUTPUTS:
//   None.
//
// DESCRIPTION:
//
//
//*** ##METHOD HEADER-END## ***************************************************
void ScienceDataProcessor::buildPolarizationCurve(
   Spectra* newSpectra, FieldOfRegard theScene, Band::Wavelength theBand)
{
    BOOST::vector<Float64> FitData;
    BOOST::vector<Float64> Coefficients;
    BOOST::vector<Float64> Wavenumbers;

    const EngCalRec_PolarizationCalibrationWavenumbers& engPolarizationInfo =
       theEngineeringCalibrationRecord.
          getEngCalRec_PolarizationCalibrationWavenumbers();

    int binSize = InstrumentCharacteristic.
       dataPointsDecimatedInterferogram[theBand];

    Wavenumbers.resize(binSize);

    //Calculate all wavenumbers for a particular band
    if (engPolarizationInfo.getValuesChanged())
    {
        for (int binIndex = 0; binIndex < binSize; binIndex++)
        {
            Wavenumbers[binIndex] =
               lowestWavenumber[theBand] + (deltaSigma[theBand] * binIndex);
        }
    }


    Coefficients.resize(AlgorithmParameter.polarizationCorrectionFitOrder+1);
    int Status = polyFitDriver(engPolarizationInfo.getWavenumberPosition(
       (Band::Wavelength)theBand),
          engPolarizationInfo.getPolarizationChange(
             theScene, (Band::Wavelength)theBand), &Coefficients);
    if(Status != 0)
    {
        EventLogEngine::append("Polarization Curve Build Failed.");
        newSpectra->setSDR_InvalidData(true);
    }

    PolynomialEval(Wavenumbers,Coefficients,&(newSpectra->bindReal()));
}

// ##METHOD HEADER-START## ****************************************************
//
// ARGUMENTS:
//  INPUTS:
//   unsigned short packetType - APID of packet
//
//  OUTPUTS:
//   Spectra* oldSpectra - The spectra leaving the window with appropriate
//                         exiting processing applied
//
// DESCRIPTION:
//  Creates and adds a new spectra to the spectra sliding window.  Reference
//  spectra are simply returned, but scene data will have optional calibration
//  and corrections applied
//
//*** ##METHOD HEADER-END## ***************************************************
// FCE update
Spectra* ScienceDataProcessor::addSpectra(Spectra* newSpectra,
         CrisSdrCoefficients* cfgParmsPtr)
{
    SceneElement thisFOV = newSpectra->getFieldOfView();
    Band::Wavelength thisBand = newSpectra->getBand();
    FieldOfRegard  thisFOR = newSpectra->getFieldOfRegard();
    SweepDirection thisPSD = newSpectra->getSweepDirection();

    newSpectra->setSDR_CalibrationType(GeneralParameter.spectralCalMethod);
    newSpectra->setSDR_ApodType(CorrectionParameter.apodizationType);
    newSpectra->setSDR_CalOrder(CorrectionParameter.calibrationOrder);

    //quantity validation of reference spectra
    if(!CalibrationParameter.allowCalibrationTargetDataMissing)
    {
        if(!SpectraManager::isReferenceScene(*newSpectra))
        {
            //make sure reference scenes are in sync with earth scene content
            if((sequenceCount[thisFOV][thisBand][InstrumentCharacteristic.
               hotReferenceIdentifer][thisPSD] + 1) !=
                  sequenceCount[thisFOV][thisBand][thisFOR][thisPSD])
            {
                char logMessage[EVENT_LOG_LENGTH];
                sprintf(logMessage,"SYNC:  %s %s %d HOT reference missing.",
                   PSD[thisPSD].c_str(), BAND[thisBand].c_str(), thisFOV + 1);
                (void)EventLogEngine::append(logMessage);

                //insert placeholder spectra
                Spectra* fakeSpectra = new CalibratedSpectra(thisFOV,thisBand,
                   InstrumentCharacteristic.hotReferenceIdentifer,thisPSD);

                fakeSpectra->setSDR_InvalidData(true);
                fakeSpectra->setSize(newSpectra->getReal().size());
                fakeSpectra->setFrameTime(newSpectra->getFrameTime());

                // set values so fake spectra are added to the window
                fakeSpectra->
                   setSDR_CalibrationType(GeneralParameter.spectralCalMethod);
                fakeSpectra->
                   setSDR_ApodType(CorrectionParameter.apodizationType);
                fakeSpectra->setSDR_CalOrder(CorrectionParameter.
                   calibrationOrder);

                delete fakeSpectra;
                fakeSpectra = 0;

                sequenceCount[thisFOV][thisBand][InstrumentCharacteristic.
                   hotReferenceIdentifer][thisPSD]++;
            }

            if((sequenceCount[thisFOV][thisBand][InstrumentCharacteristic.
               coldReferenceIdentifer][thisPSD] + 1) !=
                  sequenceCount[thisFOV][thisBand][thisFOR][thisPSD])
            {
                char logMessage[EVENT_LOG_LENGTH];
                sprintf(logMessage,"SYNC:  %s %s %d %d COLD reference missing.",
                   PSD[thisPSD].c_str(), BAND[thisBand].c_str(),
                      thisFOR, thisFOV + 1);
                (void)EventLogEngine::append(logMessage);

                //insert placeholder spectra
                Spectra* fakeSpectra = new CalibratedSpectra(thisFOV,thisBand,
                   InstrumentCharacteristic.coldReferenceIdentifer,thisPSD);

                fakeSpectra->setSDR_InvalidData(true);
                fakeSpectra->setSize(newSpectra->getReal().size());
                fakeSpectra->setFrameTime(newSpectra->getFrameTime());

                // set values so fake spectra are added to the window
                fakeSpectra->setSDR_CalibrationType(
                   GeneralParameter.spectralCalMethod);
                fakeSpectra->setSDR_ApodType(
                   CorrectionParameter.apodizationType);
                fakeSpectra->setSDR_CalOrder(CorrectionParameter.
                   calibrationOrder);

                 delete fakeSpectra;
                fakeSpectra = 0;

                sequenceCount[thisFOV][thisBand][InstrumentCharacteristic.
                   coldReferenceIdentifer][thisPSD]++;
            }//if((sequenceCount[thisFOV][thisBand][InstrumentCharacteristic.
               // coldReferenceIdentifer][thisPSD] + 1) !=
               //sequenceCount[thisFOV][thisBand][thisFOR][thisPSD])
        }//if(!SpectraManager::isReferenceScene(*newSpectra))
    }//if(!CalibrationParameter.allowCalibrationTargetDataMissing)

    //quantity validation of ICT target measurement
    if(!CalibrationParameter.allowScienceTlmDataMissing)
    {
        if(SpectraManager::isHotReferenceScene(*newSpectra))
        {
            //make sure reference scenes are in sync with earth scene content
            if((scienceCount[thisFOV][thisBand][thisFOR][thisPSD] + 1) !=
               sequenceCount[thisFOV][thisBand][thisFOR][thisPSD])
            {
                if((scienceCount[thisFOV][thisBand][thisFOR][thisPSD] + 1) <
                   sequenceCount[thisFOV][thisBand][thisFOR][thisPSD])
                {
                    char logMessage[EVENT_LOG_LENGTH];
                    sprintf(logMessage,
                       "SYNC:  %s %s %d %d Missing Science "
                       "Calibration record detected.", PSD[thisPSD].c_str(),
                       BAND[thisBand].c_str(), thisFOR, thisFOV + 1);
                    (void)EventLogEngine::append(logMessage);

                    if (! theScienceCalibrationRecord.isSciCalMissing())
                    {
                        //insert placeholder measurement
                        theScienceCalibrationRecord.markMissingCal();
                        theScienceCalibrationRecord.
                           setSciCalMissing(true); // To gaurantee markMissing
                                                   // once within the scan
                    }

                    Spectra* oldSpectra = processingPeriod.
                       getMeasuredSpectra(
                          processingPeriod.
                             getWindowDepth(thisFOV,thisBand,thisFOR,thisPSD),
                                            thisFOV,
                                            thisBand,
                                            thisFOR,
                                            thisPSD);
                    if (oldSpectra)
                    {
                      //remove invalid spectra from rawSumationSpectra
                      //and squareSummationSpectrao
                      processingPeriod.removeInvalidSpectra(oldSpectra,
                         newSpectra->getSDR_CalibrationType());

                      //invalidate associated reference scene
                      oldSpectra->setSDR_InvalidData(true);
                    }
                }
                else
                {
                    char logMessage[EVENT_LOG_LENGTH];
                    sprintf(logMessage,
                    "SYNC:  %s %s %d Extra Science Calibration "
                    "record Detected.", PSD[thisPSD].c_str(),
                    BAND[thisBand].c_str(), thisFOV + 1);
                    (void)EventLogEngine::append(logMessage);

                    //insert placeholder measurement
                    theScienceCalibrationRecord.removeLastCal();
                }

                scienceCount[thisFOV][thisBand][thisFOR][thisPSD] =
                   sequenceCount[thisFOV][thisBand][thisFOR][thisPSD] - 1;
            }
        }
    }

    //quantity validation of ICT target measurement
    if(!CalibrationParameter.allowSpaceTargetTemperatureDataMissing)
    {
        if(SpectraManager::isColdReferenceScene(*newSpectra))
        {
            //make sure reference scenes are in sync with earth scene content
            if((targetCount[thisFOV][thisBand][thisFOR][thisPSD] + 1) !=
               sequenceCount[thisFOV][thisBand][thisFOR][thisPSD])
            {
                char logMessage[EVENT_LOG_LENGTH];
                sprintf(logMessage,
                   "SYNC:  %s %s %d Missing Space Target reading detected.",
                    PSD[thisPSD].c_str(), BAND[thisBand].c_str(), thisFOV + 1);
                (void)EventLogEngine::append(logMessage);

                Spectra* oldSpectra = processingPeriod.getMeasuredSpectra(
                    processingPeriod.getWindowDepth(thisFOV,thisBand,thisFOR,
                                                    thisPSD),
                    thisFOV,
                    thisBand,
                    thisFOR,
                    thisPSD);

                if (oldSpectra)
                {
                    //remove invalid spectra from rawSumationSpectra
                    //and squareSummationSpectra
                    processingPeriod.removeInvalidSpectra(
                       oldSpectra, newSpectra->getSDR_CalibrationType());

                    //invalidate associated reference scene
                    oldSpectra->setSDR_InvalidData(true);
                }

                targetCount[thisFOV][thisBand][thisFOR][thisPSD]++;
            }
        }
    }

    Spectra* oldSpectra;

    // FCE update
    if(CorrectionParameter.performFringeCountErrorHandling == true &&
       SpectraManager::isReferenceScene(*newSpectra))
    {
        //alternate reference scene process when FCE is enabled
        //Reference scenes are placed into a holder for later processing.
        processingPeriod.addHoldingReferenceSpectra(newSpectra);

        SpectraManager::ReferenceSpectra spectraType;
        spectraType = (thisFOR ==
           InstrumentCharacteristic.coldReferenceIdentifer) ?
           SpectraManager::coldReference : SpectraManager::hotReference;

        referenceSceneReceived[spectraType][thisPSD] = true;

        // FCE update 
        //after hot scences with psd = 1 (reverse) saved in 
        //holdReferenceSpectra array, we want to add all spectra 
        //(LW/MW/SW, FOV1-9) into "basket" immediately instead of waiting
        //until the next scan. So adding a counter referenceHotSceneReceived
        if(referenceSceneReceived[SpectraManager::hotReference][thisPSD])
        {
            referenceHotSceneReceived++;
        }

        if ((referenceSceneReceived[SpectraManager::coldReference][REVERSE] +
             referenceSceneReceived[SpectraManager::coldReference][FORWARD] +
             referenceSceneReceived[SpectraManager::hotReference][REVERSE] +
             referenceSceneReceived[SpectraManager::hotReference][FORWARD]) > 1)
        {
            //Detect and Correct all reference scene/PSD
            //EXCEPT the current Reference Scene/PSD
            launchBatchReferenceDetect(spectraType, thisPSD);

        }

        if(referenceHotSceneReceived == 2 * Band::TOTAL_BANDS * MAX_FOV)
        {
           launchBatchReferenceDetect(SpectraManager::UNSPECIFIED_REFERENCE, thisPSD);
           referenceHotSceneReceived = 0;
        }

        //returns current reference scene so that it can be written to disk.
        //The new batch FC detection for reference scenes has changed the
        //philosophy of one spectra in one spectra pops off the window.
        oldSpectra = newSpectra;
    }
    else
    {

        //FCE update -- 
        //As the performFringeCountErrorHandling == false, just directly adding 
        //newSpectra register new window spectra
        oldSpectra = processingPeriod.addSpectra(newSpectra);
    }

    if(!CalibrationParameter.disableTimeStampBasedMovingWindow)
    {
        if(!referenceSceneSync[thisFOV][thisBand][thisPSD] &&
           (processingPeriod.getWindowDepth(thisFOV, thisBand,
              InternalCalTarget, thisPSD) == 2))
        {
            referenceSceneSync[thisFOV][thisBand][thisPSD] = true;
            //this was the second hot reference, therefore there should
            //at minimum of 1 of all other expected scenes
            Spectra* measuredSpectra = processingPeriod.getMeasuredSpectra(
                2, thisFOV, thisBand, InternalCalTarget, thisPSD);
            if(measuredSpectra &&
               processingPeriod.syncReferenceSpectra(*measuredSpectra,
                   CorrectionParameter.performFringeCountErrorHandling))
            {
                //data set was already alligned
            }
            else
            {
                //adjustments have been made therefore resequencing is required
                for(int scene = 0; scene < TOTAL_SCENES; scene++)
                {
                    //address only earth scenes
                    sequenceCount[thisFOV][thisBand]
                       [(FieldOfRegard)scene][thisPSD] = processingPeriod.
                          getWindowDepth(thisFOV, thisBand,
                             (FieldOfRegard)scene, thisPSD);
                }
            }
        }
    }

    //temporal validation of spectra
    if(SpectraManager::isReferenceScene(*newSpectra) &&
       !CalibrationParameter.disableTimeStampBasedMovingWindow &&
       processingPeriod.getLastSpectra(thisFOV, thisBand, thisFOR, thisPSD))
    {
        SYSTEMTIME temporalSize = processingPeriod.getWindowTimeSpan(
           thisFOV, thisBand, thisFOR, thisPSD);

        if(!validCalTargetDuration.isInside(temporalSize))
        {
            std::string sceneUnderInspection;

            if(thisFOR == InstrumentCharacteristic.hotReferenceIdentifer)
            {
                sceneUnderInspection = "ICT";
            }
            else
            {
                sceneUnderInspection = "DeepSpace";
            }

            char logMessage[EVENT_LOG_LENGTH];
            sprintf(logMessage,
               "TIME:  %s reference failed time validation with "
               "size %02d:%02d:%02d.%03d.", sceneUnderInspection.c_str(),
                    temporalSize.wHour,
                    temporalSize.wMinute,
                    temporalSize.wSecond,
                    temporalSize.wMilliseconds);
            (void)EventLogEngine::append(logMessage);

            //remove invalid spectra from rawSumationSpectra and
            // squareSummationSpectrao
            processingPeriod.removeInvalidSpectra(newSpectra, newSpectra->
               getSDR_CalibrationType());

            newSpectra->setSDR_InvalidData(true);
        }

    }

    //one-time phase synchronization of reference scenes
    //FCE update -- Only using flag performFringeCountErrorHandling to 
    //control FCE correction process
    if(CorrectionParameter.performFringeCountErrorHandling == true &&
       requiresPhaseSync[thisFOV][thisBand][thisPSD])
    {
        if(oldSpectra && !SpectraManager::isReferenceScene(*oldSpectra))
        {
            //this is a earth scene
            if(CalibrationParameter.maximumNumberOfIctDsSynchronizationTries ==
               syncAttemptCount[thisFOV][thisBand][thisPSD])
            {
                for(UINT theBand = 0; theBand < Band::TOTAL_BANDS; theBand++)
                {
                    for(UINT theFOV = 0; theFOV < MAX_FOV; theFOV++)
                    {
                        //ICT DS sync exceeded number of tries, clear
                        //requiresPhaseSync flags to stop further synchronizing
                        requiresPhaseSync[theFOV][theBand][thisPSD] = false;
                    }
                }
                char logMessage[EVENT_LOG_LENGTH];
                sprintf(logMessage,
                   "ERR :  %s %s %d Maximum Syncronization attempt exceeded.",
                   PSD[thisPSD].c_str(), BAND[thisBand].c_str(), thisFOV + 1);
                (void)EventLogEngine::append(logMessage);

            }
            else
            {
                if(!processingPeriod.batchICTDSSynchronization(oldSpectra,
                   CalibrationParameter.
                      maximumNumberOfFceTriesDuringIctDsSynchronization))
                {
                    syncAttemptCount[thisFOV][thisBand][thisPSD]++;
                }
                else
                {
                    for(UINT theBand = 0;
                       theBand < Band::TOTAL_BANDS; theBand++)
                    {
                        for(UINT theFOV = 0; theFOV < MAX_FOV; theFOV++)
                        {
                            //ICT DS is now synchronized, clear
                            //ScienceDataProcessor's requiresPhaseSync flags.
                            requiresPhaseSync[theFOV][theBand][thisPSD] = false;
                        }
                    }
                }
            }
        }
    }

    // FCE update
    return (processSpectra(oldSpectra, cfgParmsPtr));

}



// ##METHOD HEADER-START## ****************************************************
//
// ARGUMENTS:
//  INPUTS:
//   None
//
//  OUTPUTS:
//   bool indicating any value replaced.
//
// DESCRIPTION:
//  Build each child table for every band and fov, and also call
//  the calculateCalibrationMatrix method to build the final
//  calibration_matrix (CMO)
//
//*** ##METHOD HEADER-END## ***************************************************
int ScienceDataProcessor::refreshCorrectionMatrix(bool isLoading)
{
    // Update laser wavelength
    Float64 prevLaserWavelength = theEngineeringCalibrationRecord.
       getEngCalRec_LaserMetrologyInfo().getLaserWavelength();
    theEngineeringCalibrationRecord.setEngCalLaserWavelength();
    Spectra::configureLaserWavelength(theEngineeringCalibrationRecord.
       getEngCalRec_LaserMetrologyInfo().getLaserWavelength());
    Spectra::buildResamplingWavelength(InstrumentCharacteristic.
       numberSamplesPerLaserWavelength);

    if ( FloatCompare<Float64>::equal(
          resampling_laserwavelength,Spectra::LASER_WAVELENGTH[Band::LONGWAVE])
             == false )
    {
      char logMessage[EVENT_LOG_LENGTH];
      sprintf(logMessage, "Laser Wavelength UPDATED to LW(%f) MW(%f) SW(%f) "
                          "prevLaserWavelength(%f) LaserWavelength(%f) "
                          "resampling_laserwavelength (%f)",
         Spectra::LASER_WAVELENGTH[Band::LONGWAVE],
            Spectra::LASER_WAVELENGTH[Band::MIDWAVE],
               Spectra::LASER_WAVELENGTH[Band::SHORTWAVE], prevLaserWavelength,
                  theEngineeringCalibrationRecord.
                     getEngCalRec_LaserMetrologyInfo().getLaserWavelength(),
                        resampling_laserwavelength);
      (void)EventLogEngine::append(logMessage);

      for(int everyBand = 0; everyBand  <  Band::TOTAL_BANDS; everyBand++)
      {
         calculateWavenumberBinSpacing((Band::Wavelength)everyBand);
      }

      updateCorrectionMatrixParameters();

      (void)EventLogEngine::append("NOTE:  Correction Matrix refresh started");

      calculateCalibrationMatrix();
    }
    else
    {
      (void)EventLogEngine::append("NOTE:laserwavelength doesn't change");
    }
    (void)EventLogEngine::append("NOTE:  Correction Matrix refresh complete");

    // Save the CMO if update was successful, otherwise set laserwavelength
    // back to pre-update value since we are not going to save a new CMO.
    if (!isLoading)
    {
        EventLogEngine::append("NOTE:  Saving serialized CMO file started");
        saveSerializedCorrectionMatrix();
        EventLogEngine::append("NOTE:  Saving serialized CMO file complete");

    }

    return (1);
}

// only refresh the Self-Apodization correction matrix
int ScienceDataProcessor::refresh_Invsa_CorrectionMatrix(bool isLoading)
{
    int newCorrectionValues = 0;
    int tmpNewCorrectionValues = 0;
    int fovIndex = FOV1;
    int bandIndex = 0;

    // Update laser wavelength
    Float64 prevLaserWavelength = theEngineeringCalibrationRecord.
       getEngCalRec_LaserMetrologyInfo().getLaserWavelength();
    theEngineeringCalibrationRecord.setEngCalLaserWavelength();
    Spectra::configureLaserWavelength(theEngineeringCalibrationRecord.
       getEngCalRec_LaserMetrologyInfo().getLaserWavelength());
    Spectra::buildResamplingWavelength(
       InstrumentCharacteristic.numberSamplesPerLaserWavelength);

    char logMessage[EVENT_LOG_LENGTH];
    sprintf(logMessage, "in InvSa Laser Wavelength update to LW(%f) MW(%f) "
                        "SW(%f) prevLaserWavelength(%f) LaserWavelength(%f)",
    Spectra::LASER_WAVELENGTH[Band::LONGWAVE],
       Spectra::LASER_WAVELENGTH[Band::MIDWAVE],
          Spectra::LASER_WAVELENGTH[Band::SHORTWAVE], prevLaserWavelength,
             theEngineeringCalibrationRecord.getEngCalRec_LaserMetrologyInfo().
                getLaserWavelength());
    (void)EventLogEngine::append(logMessage);

    for(int everyBand = 0; everyBand  <  Band::TOTAL_BANDS; everyBand++)
    {
       calculateWavenumberBinSpacing((Band::Wavelength)everyBand);
    }

    updateCorrectionMatrixParameters();

    (void)EventLogEngine::append(
       "NOTE:  InvSa Correction Matrix refresh started");

    ////////////////////////////////////////////////////////////////////////////
    //Finish calculations for FOV1-FOV9 for SELF_APODIZATION which are
    //band and FOV Specific.
    ////////////////////////////////////////////////////////////////////////////
    Float64 correctionTableLowestWavenumber, correctionTableDeltaSigma;
    Float64 maxPathDifference_trunc;
    UInt32 bigN_trunc;
    for(bandIndex = 0; bandIndex < Band::TOTAL_BANDS; bandIndex++)
    {
       // compute lowestWavenumber and deltaSigma for SA-1 matrix
       if( CorrectionParameter.calibrationOrder == RadCalFirst_ATBD_versionA ||
           CorrectionParameter.calibrationOrder == RadCalFirst_INVsaFPf_bigN)
       {
          correctionTableLowestWavenumber =
                 correctionTable[0][bandIndex].getUserLowestWavenumber(
                                            (Band::Wavelength)bandIndex);
          correctionTableDeltaSigma =
                 correctionTable[0][bandIndex].getUserDeltaSigma(
                                       (Band::Wavelength)bandIndex);
       }
       else
       {
          bigN_trunc =
           InstrumentCharacteristic.dataPointsTruncatedInterferogram[bandIndex]*
           InstrumentCharacteristic.decimationFactor[bandIndex];
          maxPathDifference_trunc =
           computeMaxPathDifference( (Band::Wavelength)bandIndex, bigN_trunc);
          correctionTableDeltaSigma = 0.5/maxPathDifference_trunc;
          correctionTableLowestWavenumber = floor((desiredBandCenter[bandIndex]-
           correctionTableDeltaSigma*
           InstrumentCharacteristic.dataPointsDecimatedInterferogram[bandIndex]*
           0.5)/correctionTableDeltaSigma) * correctionTableDeltaSigma;
       }

       for (fovIndex = FOV1; fovIndex < MAX_FOV; fovIndex++)
       {
            correctionTable[fovIndex][bandIndex].setCorrectionTableDeltaSigma(
                      (Band::Wavelength)bandIndex, correctionTableDeltaSigma);
            correctionTable[fovIndex][bandIndex].
                   setCorrectionTableLowestWavenumber(
                      (Band::Wavelength)bandIndex, correctionTableLowestWavenumber);

            tmpNewCorrectionValues +=  correctionTable[fovIndex][bandIndex].
               buildILSCorrectionTable(theEngineeringCalibrationRecord,
                                       (SceneElement)fovIndex,
                                       (Band::Wavelength)bandIndex);
            if(tmpNewCorrectionValues < 0){
                return -1;//failure
            }
            newCorrectionValues += tmpNewCorrectionValues;
        }
    }

    (void)EventLogEngine::append(
       "NOTE:  InvSa Correction Matrix compiling started.");

    if(newCorrectionValues > 0)
    {
        numCMOBuilds_++;
        scanWhichCausedCMORebuild_ = scanIdx_;
    }

    (void)EventLogEngine::append(
       "NOTE:  InvSa Correction Matrix refresh complete");

    // Save the CMO if update was successful, otherwise set laserwavelength
    // back to pre-update value since we are not going to save a new CMO.
    if ((newCorrectionValues > 0) && !isLoading)
    {
        EventLogEngine::append("NOTE:  Saving serialized CMO file started");
        saveSerializedCorrectionMatrix();
        EventLogEngine::append("NOTE:  Saving serialized CMO file complete");

    }
    else
    {
        Spectra::configureLaserWavelength(prevLaserWavelength);
        Spectra::buildResamplingWavelength(
           InstrumentCharacteristic.numberSamplesPerLaserWavelength);
    }

    return (newCorrectionValues);
}


// ##METHOD HEADER-START## ****************************************************
//
// ARGUMENTS:
//  INPUTS:
//   None
//
//  OUTPUTS:
//   None
//
// DESCRIPTION:
//  Multiply child correction tables to calculate the final calibration matrix
//
//*** ##METHOD HEADER-END## ***************************************************
void ScienceDataProcessor::calculateCalibrationMatrix()
{
    totalActiveThreads = 0;
    UInt32 binSize;

    char logMessage[EVENT_LOG_LENGTH];
    sprintf(logMessage,"NOTE:  Begin CMO Calculation  ");
    ProCmnMessage::getInstance()->updateHealth(logMessage);
    EventLogEngine::append(logMessage);

    for(Int32 bandIndex = 0; bandIndex < Band::TOTAL_BANDS; bandIndex++)
    {
      // 0: PostFilter (Default) 1: Cosine Filter
      if ( CorrectionParameter.usePostFilterOrCosineFilter == 0 )
      {
          buildPostCalibrationTable((Band::Wavelength)bandIndex);
      }
      else
      {
          buildPostCalibrationTable_raisedCos((Band::Wavelength)bandIndex);
      }

      if (CorrectionParameter.apodizationType != NONE)
      {
         buildUserApodizationTable((Band::Wavelength)bandIndex);
      }
    }

    if( CorrectionParameter.calibrationOrder == RadCalFirst_ATBD_versionA ||
        CorrectionParameter.calibrationOrder == RadCalFirst_INVsaFPf_bigN)
    {
       for (Int32 bandIndex = 0; bandIndex < Band::TOTAL_BANDS; bandIndex++)
       {
          buildResamplingTable((Band::Wavelength)bandIndex);
          binSize = resamplingMatrix[bandIndex].size1();
          BOOST::matrix<Float64> tmp_matrix =
             boost::numeric::ublas::prod(resamplingMatrix[bandIndex],
                                         postFilter[bandIndex]);
          for (Int32 fovIndex = 0; fovIndex < MAX_FOV; fovIndex++)
          {
            if(!(calibrationMatrix[fovIndex][bandIndex].size1() == binSize) ||
               !(calibrationMatrix[fovIndex][bandIndex].size2() == binSize))
              calibrationMatrix[fovIndex][bandIndex].resize(binSize, binSize);
              CorrectionMatrix::matrixProduct(correctionTable[fovIndex][bandIndex].
                 getInvsaCalibrationMatrix(), tmp_matrix,
                    calibrationMatrix[fovIndex][bandIndex]);
          }
       }
    }
    else
    {
      if( CorrectionParameter.calibrationOrder ==
             SpecCalFirst_FPfINVsaPf_dFIR_bigN_JP ||
          CorrectionParameter.calibrationOrder == RadCalFirst_FPfINVsaPf_bigN )
      {
         //*********************************************************************
         // Build correction matrix(CMO): 
         //       CMO = F(s->u) # f # SA-1 # f # F(truncate) # f or
         //       CMO = F(s->u) # f # SA-1 # f 
         //    All the matrices have the same dimension Nd x Nd, 
         //    where Nd is the decimated interferogram size 
         //*********************************************************************
         UInt32 matrixBinSize, bigNto;
         Float64 lowestWavenumberTo, deltaSigmaTo;

         for (Int32 bandIndex = 0; bandIndex < Band::TOTAL_BANDS; bandIndex++)
         {
            matrixBinSize = InstrumentCharacteristic.
                            dataPointsDecimatedInterferogram[bandIndex];

            BOOST::matrix<Float64> matrix_tmp;

            // compute the truncation matrix F(truncate) if needed
            if( matrixBinSize != InstrumentCharacteristic.dataPointsTruncatedInterferogram[bandIndex])
            {
              bigNto   = InstrumentCharacteristic.
                         dataPointsTruncatedInterferogram[bandIndex] *
                         InstrumentCharacteristic.decimationFactor[bandIndex];
              lowestWavenumberTo = correctionTable[0][bandIndex].
                  getCorrectionTableLowestWavenumber((Band::Wavelength)bandIndex);
              deltaSigmaTo = correctionTable[0][bandIndex].
                  getCorrectionTableDeltaSigma((Band::Wavelength)bandIndex);
              BOOST::matrix<Float64> rMatrix(matrixBinSize, matrixBinSize);
              // build the truncation/interpolation matrix F(truncate)
              buildResamplingTable(lowestWavenumber[bandIndex],
                                   deltaSigma[bandIndex],  
                                   lowestWavenumberTo, deltaSigmaTo, bigNto,
                                   rMatrix);

              // Following UW modification
              // adding an additional poster before truncation matrix F(truncate)
              BOOST::matrix<Float64> tmp_matrix =
                boost::numeric::ublas::prod(rMatrix,
                                         postFilter[bandIndex]);
              matrix_tmp = boost::numeric::ublas::prod(
                                   postFilter[bandIndex], tmp_matrix);


            }
            else
            {
              matrix_tmp = postFilter[bandIndex];
 
              bigNto = matrixBinSize *
                       InstrumentCharacteristic.decimationFactor[bandIndex];
              lowestWavenumberTo = lowestWavenumber[bandIndex];
              deltaSigmaTo = deltaSigma[bandIndex];
            }

            if( resamplingMatrix[bandIndex].size1() != matrixBinSize )
               resamplingMatrix[bandIndex].resize(matrixBinSize, matrixBinSize);

            //build the resampling matrix to change the resolution and grid size
            buildResamplingTable( lowestWavenumberTo, deltaSigmaTo,  
                          correctionTable[0][bandIndex].
                           getUserLowestWavenumber((Band::Wavelength)bandIndex),
                          correctionTable[0][bandIndex].
                           getUserDeltaSigma((Band::Wavelength)bandIndex),
                          bigNto,
                          resamplingMatrix[bandIndex]);
 
            for (Int32 fovIndex = 0; fovIndex < MAX_FOV; fovIndex++)
            {
              BOOST::matrix<Float64> correction_matrix =
                boost::numeric::ublas::prod(correctionTable[fovIndex][bandIndex].
                    getInvsaCalibrationMatrix(), matrix_tmp);

              BOOST::matrix<Float64> resampling_matrix =
                 boost::numeric::ublas::prod(resamplingMatrix[bandIndex],
                                             postFilter[bandIndex]);
 
              if(!(calibrationMatrix[fovIndex][bandIndex].size1()
                          == matrixBinSize) ||
                 !(calibrationMatrix[fovIndex][bandIndex].size2()
                          == matrixBinSize))
              {
                 calibrationMatrix[fovIndex][bandIndex].
                      resize(matrixBinSize, matrixBinSize);
              }

              CorrectionMatrix::matrixProduct(resampling_matrix,
                                        correction_matrix,
                                        calibrationMatrix[fovIndex][bandIndex]);
 
            }
         }
      }
    }

    // update ScienceDataProcessor resampling_laserwavelength for 
    // rebuilding the Resampling Matrix
    resampling_laserwavelength = Spectra::LASER_WAVELENGTH[Band::LONGWAVE];
    
    sprintf(logMessage,"ATTN:  CMO Resampling reset to Laser Wavelength %f",
       Spectra::LASER_WAVELENGTH[Band::LONGWAVE]);
    EventLogEngine::append(logMessage);

    sprintf(logMessage,"NOTE:  End CMO Calculation  ");
    ProCmnMessage::getInstance()->updateHealth(logMessage);
    EventLogEngine::append(logMessage);
 }

// ##METHOD HEADER-START## ****************************************************
//
// ARGUMENTS:
//  INPUTS:
//   None
//
//  OUTPUTS:
//   None
//
// DESCRIPTION:
//  Updates correction matrix parameters from config
//
//*** ##METHOD HEADER-END## ***************************************************
void ScienceDataProcessor::updateCorrectionMatrixParameters()
{
    for (Int32 fovIndex = 0; fovIndex < MAX_FOV; ++fovIndex)
    {
        for(Int32 bandIndex = 0; bandIndex < Band::TOTAL_BANDS; ++bandIndex)
        {
            //setting Correction Parameters in Sci Data Proc
            //from theCorrectionParameterTab
            correctionTable[fovIndex][bandIndex].
               setApplyPolarizationCorrections(
                  CorrectionParameter.applyPolarizationCorrections );
            correctionTable[fovIndex][bandIndex].
               setApplyIlsFovEffectsCorrection(
                  CorrectionParameter.applyIlsFovEffectsCorrection );
            correctionTable[fovIndex][bandIndex].
               setApplyIlsResidualEffectCorrection(
                  CorrectionParameter.applyIlsResidualEffectCorrection );
            correctionTable[fovIndex][bandIndex].
               setDisableLaserMonitoring(
                  CorrectionParameter.disableLaserMonitoring );
            correctionTable[fovIndex][bandIndex].
               setPerformFringeCountErrorHandling(
                  CorrectionParameter.performFringeCountErrorHandling );
            correctionTable[fovIndex][bandIndex].
               setPerformPolarizationCorrection(
                  CorrectionParameter.performPolarizationCorrection );
            correctionTable[fovIndex][bandIndex].
               setPerformSpectralAndSpatialCorrection(
                  CorrectionParameter.performSpectralAndSpatialCorrection );
            correctionTable[fovIndex][bandIndex].
               setUseSavedMatrices( CorrectionParameter.useSavedMatrices );
            correctionTable[fovIndex][bandIndex].
               setCalibrationOrder( (CalibrationOrder)
                  CorrectionParameter.calibrationOrder );
            correctionTable[fovIndex][bandIndex].
               setSincFlag( (CalibrationOrder)
                  CorrectionParameter.calibrationOrder );
            correctionTable[fovIndex][bandIndex].
               setLaserDiodeWavelengthOrigin(
                  CorrectionParameter.laserDiodeWavelengthOrigin );
            correctionTable[fovIndex][bandIndex].
               setImpulseNoiseCountThreshold(
                  CorrectionParameter.impulseNoiseCountThreshold );
            correctionTable[fovIndex][bandIndex].
               setSAExpansionFactor((Band::Wavelength)bandIndex,
                  CorrectionParameter.saExpansionFactor[bandIndex] );

            correctionTable[fovIndex][bandIndex].setFceParamDefaultDetectorBand
               ( AlgorithmParameter.fceParamDefaultDetectorBand );
            correctionTable[fovIndex][bandIndex].setFceParamDefaultDetectorFOV
               ( AlgorithmParameter.fceParamDefaultDetectorFOV );
            correctionTable[fovIndex][bandIndex].
               setPolarizationCorrectionFitOrder
                  ( AlgorithmParameter.polarizationCorrectionFitOrder );
            correctionTable[fovIndex][bandIndex].
               setMaximumFractionRejections
                  ( AlgorithmParameter.maximumFractionRejections );
            correctionTable[fovIndex][bandIndex].
               setComputedWavelengthRejectionThreshold
                  ( AlgorithmParameter.computedWavelengthRejectionThreshold );
            correctionTable[fovIndex][bandIndex].
               setLaserWavelengthDriftTolerance
                  ( AlgorithmParameter.laserWavelengthDriftTolerance );
            correctionTable[fovIndex][bandIndex].
               setNumberOpdOverscanSamples
                  ( AlgorithmParameter.numberOpdOverscanSamples );

            if (CorrectionParameter.applyResamplingMatrix)
            {
                Float64 userFirstWavenumber =
                (Float64) ( ROUND((((CorrectionParameter.
                   edrMinimumWavenumber[bandIndex] +
                      CorrectionParameter.edrMaximumWavenumber[bandIndex]) /
                         CorrectionParameter.edrDeltaSigma[bandIndex])-
                            InstrumentCharacteristic.
                               dataPointsDecimatedInterferogram[bandIndex] ) /
                                  2.0f) ) *
                                   CorrectionParameter.edrDeltaSigma[bandIndex];


                //adjust metadata of resampling
                correctionTable[fovIndex][bandIndex].setUserDeltaSigma(
                   (Band::Wavelength)bandIndex,
                      CorrectionParameter.edrDeltaSigma[bandIndex]);
                correctionTable[fovIndex][bandIndex].setUserLowestWavenumber(
                   (Band::Wavelength)bandIndex, userFirstWavenumber);

            }
            else
            {
                correctionTable[fovIndex][bandIndex].setUserDeltaSigma(
                   (Band::Wavelength)bandIndex, deltaSigma[bandIndex]);
                correctionTable[fovIndex][bandIndex].setUserLowestWavenumber(
                   (Band::Wavelength)bandIndex, lowestWavenumber[bandIndex]);
            }

            correctionTable[fovIndex][bandIndex].setEdrNumberOfPoints(
               (Band::Wavelength)bandIndex, CorrectionParameter.
                  edrNumberOfPoints[bandIndex] );
            correctionTable[fovIndex][bandIndex].setEdrDeltaSigma(
               (Band::Wavelength)bandIndex, CorrectionParameter.
                  edrDeltaSigma[bandIndex] );
            correctionTable[fovIndex][bandIndex].setEdrMaximumWavenumber(
               (Band::Wavelength)bandIndex, CorrectionParameter.
                  edrMaximumWavenumber[bandIndex] );
            correctionTable[fovIndex][bandIndex].setEdrMinimumWavenumber(
               (Band::Wavelength)bandIndex, CorrectionParameter.
                  edrMinimumWavenumber[bandIndex] );

            correctionTable[fovIndex][bandIndex].
               setDecimationFactor((Band::Wavelength)bandIndex,
                  InstrumentCharacteristic.decimationFactor[bandIndex] );
            correctionTable[fovIndex][bandIndex].
               setDataPointsDecimatedInterferogram(
                  (Band::Wavelength)bandIndex, InstrumentCharacteristic.
                     dataPointsDecimatedInterferogram[bandIndex] );
            correctionTable[fovIndex][bandIndex].
               setDataPointsUndecimatedInterferogram(
                  (Band::Wavelength)bandIndex, InstrumentCharacteristic.
                     dataPointsUndecimatedInterferogram[bandIndex] );

            correctionTable[fovIndex][bandIndex].
               setPostCalibrationA1((Band::Wavelength)bandIndex,
                  AlgorithmParameter.postCalibrationA1[bandIndex] );
            correctionTable[fovIndex][bandIndex].
               setPostCalibrationA2((Band::Wavelength)bandIndex,
                  AlgorithmParameter.postCalibrationA2[bandIndex] );
            correctionTable[fovIndex][bandIndex].
               setPostCalibrationA3((Band::Wavelength)bandIndex,
                  AlgorithmParameter.postCalibrationA3[bandIndex] );
            correctionTable[fovIndex][bandIndex].
               setPostCalibrationA4((Band::Wavelength)bandIndex,
                  AlgorithmParameter.postCalibrationA4[bandIndex] );
            correctionTable[fovIndex][bandIndex].
               setPostCalibrationK ((Band::Wavelength)bandIndex,
                  AlgorithmParameter.postCalibrationK[bandIndex]  );
            correctionTable[fovIndex][bandIndex].
               setPostCalibrationK0((Band::Wavelength)bandIndex,
                  AlgorithmParameter.postCalibrationK0[bandIndex] );
            correctionTable[fovIndex][bandIndex].
               setPostCalibrationK1((Band::Wavelength)bandIndex,
                  AlgorithmParameter.postCalibrationK1[bandIndex] );
            correctionTable[fovIndex][bandIndex].
               setFceParamAmpThreshRejectLimit(Spectra::REF_THRESHOLD,
                  (Band::Wavelength)bandIndex,
                     AlgorithmParameter.fceParamAmpThreshRejectLimit
                        [bandIndex][Spectra::REF_THRESHOLD] );
            correctionTable[fovIndex][bandIndex].
               setFceParamAmpThreshRejectLimit(
                  Spectra::CAL_THRESHOLD,
                     (Band::Wavelength)bandIndex,
                        AlgorithmParameter.fceParamAmpThreshRejectLimit
                           [bandIndex][Spectra::CAL_THRESHOLD] );

            correctionTable[fovIndex][bandIndex].
               setFceParamDimensionThresholdLimit(
                  (Band::Wavelength)bandIndex, AlgorithmParameter.
                     fceParamDimensionThresholdLimit[bandIndex] );
            correctionTable[fovIndex][bandIndex].
               setFceParamFractionalFceThresholdLimit(
                  (Band::Wavelength)bandIndex, AlgorithmParameter.
                     fceParamFractionalFceThresholdLimit[bandIndex] );
            correctionTable[fovIndex][bandIndex].
               setFceParamGoodLinearFittingThreshLimit(
                  (Band::Wavelength)bandIndex, AlgorithmParameter.
                     fceParamGoodLinearFittingThreshLimit[bandIndex] );
            correctionTable[fovIndex][bandIndex].
               setFceParamMaxIndex(
                  (Band::Wavelength)bandIndex, AlgorithmParameter.
                     fceParamMaxIndex[bandIndex] );
            correctionTable[fovIndex][bandIndex].
               setFceParamMaxFceThreshLimit(
                  (Band::Wavelength)bandIndex, AlgorithmParameter.
                     fceParamMaxFceThreshLimit[bandIndex] );
            correctionTable[fovIndex][bandIndex].
               setFceParamMinIndex(
                  (Band::Wavelength)bandIndex, AlgorithmParameter.
                     fceParamMinIndex[bandIndex] );

            correctionTable[fovIndex][bandIndex].
               setInstrumentDeltaSigma(
                  (Band::Wavelength)bandIndex, deltaSigma[bandIndex]);
            correctionTable[fovIndex][bandIndex].
               setInstrumentLowestWavenumber((Band::Wavelength)bandIndex,
                  lowestWavenumber[bandIndex]);
            correctionTable[fovIndex][bandIndex].
               setMaxPathDifferential(
                  (Band::Wavelength)bandIndex, maxPathDifferential[bandIndex]);

        }
    }
}



// ##METHOD HEADER-START## ****************************************************
//
// ARGUMENTS:
//  INPUTS:
//   Band::Wavelength theBand
//
//  OUTPUTS:
//   Spectra* hotRadianceCurve
//
// DESCRIPTION:
//  Calculates the hot radiance
//
//*** ##METHOD HEADER-END## ***************************************************

// calculate ICT radiance on sensor or user grids depending
// on the calibration order
Spectra* ScienceDataProcessor::calculateHotRadiance(
   SceneElement theFOV, Band::Wavelength theBand)
{
    TelemetryProcessor *theTelemetryProcessor = TelemetryProcessor::instance();

    const EngCalRec_ICTEnvironmentalModel* theIctEnvModel =
       &theEngineeringCalibrationRecord.getEngCalRec_ICTEnvironmentalModel();
    const EngCalRec_ICTEmissivityParameters* theIctEmissParam =
       &theEngineeringCalibrationRecord.getEngCalRec_ICTEmissivityParameters();

    const BOOST::vector < Float32> *baffleTemperatureOffset =
       &(theIctEnvModel->getEngCalBaffleTempOffset());

    //get the sample count from the most recently read apid
    // Note: The algorithm parameter value number of
    //       numberOpdOverscanSamples is trimmed from the
    //       the back and front, so 2 * numberOpdOverscanSamples is used.
    UInt32 binSize;

    if(TelemetryProcessor::instance()->getVideoMode() ==
       TelemetryProcessor::DIAGNOSTIC_BURST)
    {
        binSize = theTelemetryProcessor->theDiagnosticData.
           getSampleCount((DiagnosticData::Wavelength)theBand) / 2;
        // recalculate delta sigma and sigma min for for diagnostic
        maxPathDifferential[theBand] =
           (Spectra::LASER_WAVELENGTH[theBand] / 2.0) * binSize * 0.0000001;

        deltaSigma[theBand] = 1.0/(2.0 * maxPathDifferential[theBand]);

        lowestWavenumber[theBand] = 0;
    }
    else
    {
        binSize = theTelemetryProcessor->theVideoData.getExtractionRecord(
           (VideoData::Wavelength)theBand).complexSamples -
              (2 * AlgorithmParameter.numberOpdOverscanSamples);
    }

    //get average ict temp from sliding window
    Float64 ictTemp = theScienceCalibrationRecord.
       getProcessingIctPeriod().calculateAverageTemperature();

    // updated for different calibration order
    Float64 first_wn = 0.0;
    Float64 deltaSigma_u = 0.0;

    if(CorrectionParameter.calibrationOrder == RadCalFirst_ATBD_versionA   ||
       CorrectionParameter.calibrationOrder == RadCalFirst_INVsaFPf_bigN   ||
       CorrectionParameter.calibrationOrder == RadCalFirst_FPfINVsaPf_bigN )
    { //in sensor grid
        first_wn = this->lowestWavenumber[theBand];
        deltaSigma_u = this->deltaSigma[theBand];
    }
    else if ( CorrectionParameter.calibrationOrder == SpecCalFirst_FPfINVsaPf_dFIR_bigN_JP )

    {//in user grid
        first_wn = this->correctionTable[theFOV][theBand].
           getUserLowestWavenumber(theBand);
        deltaSigma_u = this->correctionTable[theFOV][theBand].
           getUserDeltaSigma(theBand);
    }

    Spectra* hotRadianceCurve = new UncalibratedSpectra();

    hotRadianceCurve->setFieldOfView(theFOV);
    hotRadianceCurve->setBand(theBand);
    hotRadianceCurve->setSize(binSize);

    hotRadianceCurve->setWavenumberSpacing(deltaSigma_u);
    hotRadianceCurve->setFirstWavenumber(first_wn);

    Float64 scaleFactor = 1.0;
    bool appScaleFactor = (appShiftFactorFlag == true &&
          (CorrectionParameter.calibrationOrder == RadCalFirst_ATBD_versionA   ||
           CorrectionParameter.calibrationOrder == RadCalFirst_INVsaFPf_bigN   ||
           CorrectionParameter.calibrationOrder == RadCalFirst_FPfINVsaPf_bigN ) );
    if( appScaleFactor == true ) scaleFactor = 1.0 + shiftFactor[theFOV];

    //calculate ICT Radiance
    for(UInt32 binIndex = 0; binIndex < binSize; binIndex++)
    {
        Float64 wavenumber = (first_wn + (binIndex * deltaSigma_u)) -
           (deltaSigma_u *radianceModelOffset[theFOV][theBand]);
        if( appScaleFactor == true )
        {
           wavenumber *= scaleFactor;
        }

        if (CalibrationParameter.ictEmissivityOrigin ==
           ICT_CONFIG && !CalibrationParameter.
              useWavenumberDependentIctEmissivity)
        {
            //use Instrument parameter Mean ICT Emissivity (emissivity * plank)
            *(*hotRadianceCurve)[binIndex] =
               ((GeneralParameter.instrumentLocation == BENCH) ?
                  (InstrumentCharacteristic.benchMeanIctEmissivity[theBand]) :
                     (InstrumentCharacteristic.
                        chamberMeanIctEmissivity[theBand])) *
                           ((getPlankConstantC1() * (wavenumber * wavenumber *
                              wavenumber)) /
                           ((exp(getPlankConstantC2() *
                              wavenumber / ictTemp)) - 1));
        }
        else
        {
            //use values in engineering packet (emissivity * plank)
            *(*hotRadianceCurve)[binIndex] =
               (theIctEmissParam->getEffectiveEmissivity(theBand))[binIndex] *
                ((getPlankConstantC1() *
                   (wavenumber * wavenumber * wavenumber)) /
                      ((exp(getPlankConstantC2() *
                         wavenumber / ictTemp)) - 1));
        }
    }

    //Calculate total ICT Radiance
    if (CalibrationParameter.useIctEnvironmentalCorrectionModel)
    {
        Float64 omaContribution;
        Float64 beamsplitterContribution;
        Float64 scanBaffleContribution;
        Float64 ictBaffleContribution;
        Float64 spaceContribution;

        //average temperatures from sliding windows
        Float64 omaTemp = theScienceCalibrationRecord.
           getProcessingOmaPeriod().calculateAverageTemperature();
        Float64 baffleTemp = theScienceCalibrationRecord.
           getProcessingScanBafflePeriod().calculateAverageTemperature();

        Float64 scanMirrorReflectivity =
           theIctEnvModel->getScanMirrorReflectivity(theBand);

        Float64 coldBeamsplitterViewedSurface =
           (CalibrationParameter.instrumentTemperatureOrigin == INSTRUMENT_TLM)
            ?
            (theIctEnvModel->getColdBeamsplitterViewFactor())
            :
            TestEnvironmentParameter.coldBeamSplitterViewFactor;

        Float64 warmBeamsplitterViewedSurface =
           (CalibrationParameter.instrumentTemperatureOrigin == INSTRUMENT_TLM)
            ?
            (theIctEnvModel->getWarmBeamsplitterViewFactor())
            :
            TestEnvironmentParameter.warmBeamSplitterViewFactor;

        Float64 scanBaffleViewedSurface =
           (CalibrationParameter.instrumentTemperatureOrigin == INSTRUMENT_TLM)
            ?
            (theIctEnvModel->getScanBaffleViewFactor())
            :
            TestEnvironmentParameter.scanBaffleViewFactor;

        Float64 omaViewedSurface =
           (CalibrationParameter.instrumentTemperatureOrigin == INSTRUMENT_TLM)
            ?
            (theIctEnvModel->getFrameViewFactor())
            :
            TestEnvironmentParameter.omaViewFactor;

        Float64 ictBaffleViewedSurface =
           (CalibrationParameter.instrumentTemperatureOrigin == INSTRUMENT_TLM)
            ?
            (theIctEnvModel->getIctBaffleViewFactor())
            :
            TestEnvironmentParameter.ictBaffleViewFactor;

        Float64 spaceViewedSurface =
           (CalibrationParameter.instrumentTemperatureOrigin == INSTRUMENT_TLM)
            ?
            (theIctEnvModel->getSpaceViewFactor())
            :
            TestEnvironmentParameter.spaceViewFactor;

        Float64 effectiveEmissivity;

        //use values in engineering packet unless the emissivity is < 0 or > 1,
        //then use default values from config file
        Float64 theScanBaffleEmissivity   =
           (CalibrationParameter.instrumentTemperatureOrigin ==
             INSTRUMENT_TLM &&
           (theIctEnvModel->getScanBaffleEmissivity(theBand) > 0 &&
           theIctEnvModel->getScanBaffleEmissivity(theBand) <= 1))
            ?
           (theIctEnvModel->getScanBaffleEmissivity(theBand))
            :
           TestEnvironmentParameter.scanBaffleEmissivity[theBand];

        Float64 theOmaEmissivity          =
           (CalibrationParameter.instrumentTemperatureOrigin ==
             INSTRUMENT_TLM &&
             (theIctEnvModel->getInterferometerHousingEmissivity(theBand) > 0 &&
             theIctEnvModel->getInterferometerHousingEmissivity(theBand) <= 1))
              ?
             (theIctEnvModel->getInterferometerHousingEmissivity(theBand))
              :
             TestEnvironmentParameter.omaEmissivity[theBand];

        Float64 theIctBaffleEmissivity =
           (CalibrationParameter.instrumentTemperatureOrigin ==
             INSTRUMENT_TLM &&
             (theIctEnvModel->getIctBaffleEmissivity(theBand) > 0 &&
             theIctEnvModel->getIctBaffleEmissivity(theBand) <=1))
              ?
             (theIctEnvModel->getIctBaffleEmissivity(theBand))
              :
             TestEnvironmentParameter.ictBaffleEmissivity[theBand];

         Float64 theEarthEmissivity     =
           (TestEnvironmentParameter.earthTargetOverride ||
             (CalibrationParameter.
               instrumentTemperatureOrigin == INSTRUMENT_CONFIG))
                                            ?
             TestEnvironmentParameter.ssmTargetEmissivity[theBand]
                                            :
             (theIctEnvModel->getSsmTargetEmissivity(theBand) > 0 &&
             theIctEnvModel->getSsmTargetEmissivity(theBand) <= 1)
                                                 ?
             (theIctEnvModel->getSsmTargetEmissivity(theBand))
                                                 :
             TestEnvironmentParameter.ssmTargetEmissivity[theBand];

         Float64 theEarthTemperature    =
           (TestEnvironmentParameter.earthTargetOverride ||
             (CalibrationParameter.
               instrumentTemperatureOrigin == INSTRUMENT_CONFIG))
                                            ?
             ((GeneralParameter.instrumentLocation == BENCH)
                                                 ?
             (TestEnvironmentParameter.ssmTargetTempBench)
                                                 :
             (TestEnvironmentParameter.ssmTargetTempChamber))
                                            :
                                          (theIctEnvModel->getSsmTargetTemp());

        //conditional calibration of on-orbit environmental structure
        //introduced by spacecraft position with respect to the sun
        if(!CalibrationParameter.suppressSsmBaffleProfile)
        {
            const Int32 ORBIT_DURATION =
                theEngineeringCalibrationRecord
                    .getEngCalRec_ICTEnvironmentalModel()
                    .getEngCalOrbitalPeriod();

            // Get the current packet time.
            SYSTEMTIME now =
                TelemetryProcessor::instance()->getLastFrameTime();

            // Use the maximum North Pole crossing time if the current
            // packet's time is greater than the maximum North Pole crossing
            // time.
            SYSTEMTIME when =
                SystemTimeUtilities::quickTest(
                        now,
                        maxNpCrossingTime_,
                        GT)
                ? maxNpCrossingTime_
                : minNpCrossingTime_;

            // Determine the difference of the frame time and the
            // North Pole crossing.
            SYSTEMTIME where = SystemTimeUtilities::subtract(now, when);

            // Determine the baffle temperature lookup index.
            const Int32 secondsInDiff =
                (Float64)SystemTimeUtilities::getTotalMilliSeconds(where)
                        / SystemTimeUtilities::MILLISECONDS_PER_SECOND + 0.5;
            Int32 profileIndex = secondsInDiff % ORBIT_DURATION;

            // Log only scans that belong to the current granule and
            // throttle those messages so that only one is printed per
            // scan.
            const Int32 NUM_SCANS =
                ProCmnProcessData::getInstance()->getActScan();
            if ((scanIdx_ >= CRIS_WINDOW_SIZE/2) &&
                (scanIdx_ < (CRIS_WINDOW_SIZE/2 + NUM_SCANS)))
            {
                Int32 scanIdxTmp = scanIdx_ - CRIS_WINDOW_SIZE/2;
                if (!orbitMsgThrottle_[scanIdxTmp])
                {
                    std::ostringstream oss;
                    oss << "Polar crossing time = "
                        << SystemTimeUtilities::Format(when)
                        << "; Current time = "
                        << SystemTimeUtilities::Format(now)
                        << "; Difference = "
                        << SystemTimeUtilities::Format(where)
                        << "; Total seconds in difference = "
                        << secondsInDiff
                        << "; profile index = "
                        << profileIndex;
                    EventLogEngine::append(oss.str());

                    orbitMsgThrottle_[scanIdxTmp] = true;
                }
            }

            //adjust SSM baffle temperature with orbital profile
            baffleTemp += (*baffleTemperatureOffset)[profileIndex];
       }

        for (UInt32 binIndex = 0; binIndex < binSize; ++binIndex)
        {
            Float64 wavenumber =
               (first_wn + (binIndex * deltaSigma_u)) -
                  (deltaSigma_u *radianceModelOffset[theFOV][theBand]);
            if(appScaleFactor == true)
            {
               wavenumber *= scaleFactor;
            }

            Float64 plankCommon =
               getPlankConstantC1() * (wavenumber * wavenumber * wavenumber);

            if (CalibrationParameter.ictEmissivityOrigin ==
               ICT_CONFIG && !CalibrationParameter.
                  useWavenumberDependentIctEmissivity)
            {
                //use Instrument parameter Mean ICT Emissivity
                effectiveEmissivity = ((GeneralParameter.
                  instrumentLocation == BENCH) ?
                  (InstrumentCharacteristic.benchMeanIctEmissivity[theBand]) :
                  (InstrumentCharacteristic.chamberMeanIctEmissivity[theBand]));

            }
            else
            {
                //use values in engineering packet unless the emissivity
                //is < 0 or > 1 then use default values from config file
                effectiveEmissivity = ((theIctEmissParam->
                  getEffectiveEmissivity(theBand))[binIndex] > 0 &&
                    (theIctEmissParam->
                      getEffectiveEmissivity(theBand))[binIndex] <= 1) ?
                        ((theIctEmissParam->
                          getEffectiveEmissivity(theBand))[binIndex]) :
                            ((GeneralParameter.instrumentLocation == BENCH) ?
                              (InstrumentCharacteristic.
                                benchMeanIctEmissivity[theBand]) :
                                  (InstrumentCharacteristic.
                                    chamberMeanIctEmissivity[theBand]));


            }

            scanBaffleContribution =
               (1.0 - effectiveEmissivity) * theScanBaffleEmissivity *
                  scanBaffleViewedSurface *
                     (plankCommon / ((exp(getPlankConstantC2() * wavenumber /
                        baffleTemp)) - 1.0));

            omaContribution =
               (1.0 - effectiveEmissivity) * theOmaEmissivity *
                  scanMirrorReflectivity *
                     (omaViewedSurface + warmBeamsplitterViewedSurface) *
                        (plankCommon / ((exp(getPlankConstantC2() *
                           wavenumber / omaTemp)) - 1.0));

            beamsplitterContribution = (1.0 - effectiveEmissivity) *
               scanMirrorReflectivity * scanMirrorReflectivity *
                  coldBeamsplitterViewedSurface * ((plankCommon /
                     ((exp(getPlankConstantC2() * wavenumber / ictTemp)) -
                        1.0) / 2.0));

            ictBaffleContribution = (1.0 - effectiveEmissivity) *
               theIctBaffleEmissivity * ictBaffleViewedSurface * (plankCommon /
                  ((exp(getPlankConstantC2() * wavenumber / ictTemp)) - 1.0));

            spaceContribution = (1.0 - effectiveEmissivity) *
               theEarthEmissivity * spaceViewedSurface * (plankCommon /
                  ((exp(getPlankConstantC2() * wavenumber /
                     theEarthTemperature)) - 1.0));

            *(*hotRadianceCurve)[binIndex] += scanBaffleContribution +
               omaContribution + beamsplitterContribution +
                  ictBaffleContribution + spaceContribution;
            // scale the ICT radiance if the self-apodization needed
            // to be considered
            *(*hotRadianceCurve)[binIndex] *= scaleFactor;
        }
    }

    //setting the target significance with
    //size of the temperature sliding window
    hotRadianceCurve->setTargetSignificance(
       (BYTE)theScienceCalibrationRecord.
          getProcessingBeamsplitterPeriod().getHistoricalWindowSize(), 0);

    return hotRadianceCurve;
}



// ##METHOD HEADER-START## ****************************************************
//
// ARGUMENTS:
//  INPUTS:
//   Band::Wavelength theBand
//
//  OUTPUTS:
//   Spectra* coldRadianceCurve
//
// DESCRIPTION:
//  Calculate Cold Radiance Curve on sensor or user
//  grids depending on the calibration order
//
//*** ##METHOD HEADER-END## ***************************************************
Spectra* ScienceDataProcessor::calculateColdRadiance(
   SceneElement theFOV, Band::Wavelength theBand)
{
    //Average Temperature for ds temp sliding window
    Float64 dsTemp = theTargetTempCalibrationRecord.
       getProcessingPeriod().calculateAverageTemperature();

    TelemetryProcessor *theTelemetryProcessor =
       TelemetryProcessor::instance();

    //get the sample count from the most recently read apid
    // Note: The algorithm parameter value number of
    //       numberOpdOverscanSamples is trimmed from the
    //       the back and front, so 2 * numberOpdOverscanSamples is used.

    UInt32 binSize;

    if(TelemetryProcessor::instance()->getVideoMode() ==
       TelemetryProcessor::DIAGNOSTIC_BURST)
    {
        binSize = theTelemetryProcessor->theDiagnosticData.
           getSampleCount((DiagnosticData::Wavelength)theBand) / 2;

        // recalculate delta sigma and sigma min for for diagnostic diagnostic
        maxPathDifferential[theBand] =
           (Spectra::LASER_WAVELENGTH[theBand] / 2.0) * binSize * 0.0000001;

        deltaSigma[theBand] = 1.0/(2.0 * maxPathDifferential[theBand]);

        lowestWavenumber[theBand] = 0;
    }
    else
    {
        binSize = theTelemetryProcessor->theVideoData.getExtractionRecord(
           (VideoData::Wavelength)theBand).complexSamples -
              (2 * AlgorithmParameter.numberOpdOverscanSamples);
    }

    // updated for different calibration order
    Float64 first_wn = 0.0;
    Float64 deltaSigma_u = 0.0;

    if(CorrectionParameter.calibrationOrder == RadCalFirst_ATBD_versionA   ||
       CorrectionParameter.calibrationOrder == RadCalFirst_INVsaFPf_bigN   ||
       CorrectionParameter.calibrationOrder == RadCalFirst_FPfINVsaPf_bigN )
    { //in sensor grid
        first_wn = this->lowestWavenumber[theBand];
        deltaSigma_u = this->deltaSigma[theBand];
    }
    else if ( CorrectionParameter.calibrationOrder == SpecCalFirst_FPfINVsaPf_dFIR_bigN_JP )
    {//in user grid
        first_wn = this->correctionTable[theFOV][theBand].
           getUserLowestWavenumber(theBand);
        deltaSigma_u = this->correctionTable[theFOV][theBand].
           getUserDeltaSigma(theBand);
    }

    Spectra* coldRadianceCurve = new UncalibratedSpectra();

    coldRadianceCurve->setFieldOfView(theFOV);
    coldRadianceCurve->setBand(theBand);
    coldRadianceCurve->setSize(binSize);

    coldRadianceCurve->setWavenumberSpacing(deltaSigma_u);
    coldRadianceCurve->setFirstWavenumber(first_wn);

    //calculate cold Radiance
    for(UInt32 binIndex = 0; binIndex < binSize; binIndex++)
    {
        Float64 wavenumber = ( first_wn + (binIndex * deltaSigma_u))-
           ( deltaSigma_u *radianceModelOffset[theFOV][theBand]);

        if (CalibrationParameter.useWavenumberDependentDsEmissivity)
        {
            //use values in engineering packet
            *(*coldRadianceCurve)[binIndex] =
               TestEnvironmentParameter.
                  dsEffectiveEmissivity[theBand][binIndex] *
                    ((getPlankConstantC1() * (wavenumber *
                       wavenumber * wavenumber)) /
                         ((exp(getPlankConstantC2() *
                            wavenumber / dsTemp)) - 1));
        }
        else
        {
            //use Instrument parameter Mean cold Emissivity
            *(*coldRadianceCurve)[binIndex] =
               ((GeneralParameter.instrumentLocation == BENCH) ?
                      (TestEnvironmentParameter.meanDsEmissivityBench) :
                         (TestEnvironmentParameter.meanDsEmissivityChamber)) *
                            ((getPlankConstantC1() *
                               (wavenumber * wavenumber * wavenumber)) /
                                  ((exp(getPlankConstantC2() *
                                     wavenumber / dsTemp)) - 1));
        }
    }

    //setting the target significance with size
    //of the temperature sliding window
    coldRadianceCurve->
       setTargetSignificance(0, (BYTE)theScienceCalibrationRecord.
          getProcessingIctPeriod().getHistoricalWindowSize());

    return coldRadianceCurve;
}


// ##METHOD HEADER-START## ****************************************************
//
// ARGUMENTS:
//  INPUTS:
//   SpectralCalibrationMethod newMethod
//
//  OUTPUTS:
//   None
//
// DESCRIPTION:
//  Sets the window size base on the spectral calibration method
//
//*** ##METHOD HEADER-END## ***************************************************
void ScienceDataProcessor::setSpectralCalibrationMethod(
   SpectralCalibrationMethod newMethod)
{
    GeneralParameter.spectralCalMethod = newMethod;
}


// ##METHOD HEADER-START## ****************************************************
//
// ARGUMENTS:
//  INPUTS:
//   SceneElement       fov
//   Band::Wavelength   band
//   FieldOfRegard      scene
//   SweepDirection     psd
//
//  OUTPUTS:
//   pointer to processed spectra removed from window
//
// DESCRIPTION:
//  Removes a spectra from the sliding window without adding a spectra.
//  Generally  used to extract buffered content upon process completion.
//
//*** ##METHOD HEADER-END## ***************************************************
Spectra* ScienceDataProcessor::flushWindow(
   SceneElement fov, Band::Wavelength band,
   FieldOfRegard scene, SweepDirection psd)
{
    Spectra* oldSpectra = processingPeriod.purgeSpectra(fov, band, scene, psd);

    if(oldSpectra)
    {
        sequenceCount[fov][band][scene][psd]--;
        if(sequenceCount[fov][band][scene][psd] < 0)
        {
            sequenceCount[fov][band][scene][psd] = 0;
        }

        if(SpectraManager::isReferenceScene(scene))
        {
            targetCount[fov][band][scene][psd]--;
            scienceCount[fov][band][scene][psd]--;
            if(targetCount[fov][band][scene][psd] < 0)
            {
                targetCount[fov][band][scene][psd] = 0;
            }

            if(scienceCount[fov][band][scene][psd] < 0)
            {
                scienceCount[fov][band][scene][psd] = 0;
            }
        }
    }
    return(oldSpectra);
}


// ##METHOD HEADER-START## ****************************************************
//
// ARGUMENTS:
//  INPUTS:
//   Spectra*  newSpectra  -  The exiting spectra to have processed
//
//  OUTPUTS:
//   same spectra
//
// DESCRIPTION:
//  Processing Earth Scene spectra according to configuration parameters.
//  Calibration,Correction and validity are evaluated.
//
//*** ##METHOD HEADER-END## ***************************************************
// FCE update -- Bring CrisSdrCoefficients* cfgParmsPtr here for ImagLWThreshold
Spectra* ScienceDataProcessor::processSpectra(Spectra* newSpectra,
                   CrisSdrCoefficients* cfgParmsPtr)
{
    SceneElement     oldFOV;
    Band::Wavelength oldBand;
    FieldOfRegard    oldScene;
    SweepDirection   oldPSD;

    bool refScene=false;

    if(newSpectra)
    {
        //spectra leaving window will be of same type as the one entering
        oldFOV = newSpectra->getFieldOfView();
        oldBand = newSpectra->getBand();
        oldScene = newSpectra->getFieldOfRegard();
        oldPSD = newSpectra->getSweepDirection();


        //add available geolocation information to spectra
        newSpectra->setLosPitchAngle(processingPeriod.
           getGeolocationSpectra(0, oldScene, oldFOV)->getLosPitchAngle());
        newSpectra->setLosRollAngle(processingPeriod.
           getGeolocationSpectra(0,oldScene, oldFOV)->getLosRollAngle());
    }

    if((newSpectra) && (newSpectra->isValid()))
    {
        if(GeneralParameter.spectralCalMethod != CalibrationType_None)
        {
            if(!SpectraManager::isReferenceScene(*newSpectra) &&
                scanIdx_ >= CRIS_WINDOW_SIZE)
            {
                // FCE update -- Save raw LW spectrum for FCE detection
                CalibratedSpectra rawSpectra;
                refScene = SpectraManager::isReferenceScene(*newSpectra);
                if (!refScene && oldBand == Band::LONGWAVE &&
                    !earthScenefceDetectionComplete[oldScene] )
                {
                    rawSpectra = *newSpectra;
                }
                UncalibratedSpectra  hotRefSpectra(*(processingPeriod.
                              getReferenceSpectra(SpectraManager::hotReference,
                                                  oldFOV, oldBand, oldPSD)));
                UncalibratedSpectra coldRefSpectra(*(processingPeriod.
                              getReferenceSpectra(SpectraManager::coldReference,
                                                  oldFOV, oldBand, oldPSD)));
                if(InstrumentCharacteristic.
                      performLinearityCorrectionControl[oldBand])
                {
                   //apply LinErr corrections before calibration
                   hotRefSpectra.applyLinearityErrorCorrection();
                   coldRefSpectra.applyLinearityErrorCorrection();
                   newSpectra->applyLinearityErrorCorrection();
                }
                // FCE update --- 
                //Wrapping different spectral calibration calls into one function
                //Performing radiometric and spectral calibrations
                processingPeriod.calibrate(newSpectra, &hotRefSpectra,
                                           &coldRefSpectra, oldFOV, oldBand);

                Int16 hotSize  = (Int16)processingPeriod.getValidCount(
                       oldFOV, oldBand,
                       InstrumentCharacteristic.hotReferenceIdentifer, oldPSD);
                Int16 coldSize = (Int16)processingPeriod.getValidCount(
                       oldFOV, oldBand,
                       InstrumentCharacteristic.coldReferenceIdentifer, oldPSD);

                //encode calibration validity
                newSpectra->setTargetSignificance(hotSize, coldSize);
                if((processingPeriod.getCalibrationWindowSize() > 0) &&
                       ((hotSize < getValidWindowSize()) ||
                       (coldSize < getValidWindowSize())))
                {
                    newSpectra->setSDR_InvalidData(true);
                }
                // update First Wavenumber (value will change
                // if resampling Matrix applied)
                newSpectra->setFirstWavenumber(
                    correctionTable[oldFOV][oldBand].
                    getUserLowestWavenumber(oldBand));
                // update Wavenumber bin size (value will change
                // if resampling Matrix applied)
                newSpectra->setWavenumberSpacing(
                    correctionTable[oldFOV][oldBand].
                    getUserDeltaSigma(oldBand));

		// K. Zhang, ADR-11194, update SDR_Status if the default laser wavelength was used
		if(fabs(Spectra::LASER_WAVELENGTH[oldBand]
			- InstrumentCharacteristic.defaultLaserWavelength) < 1e-6 &&
		   savedfoldIndex_flag[oldBand] == false)
		{
		   newSpectra->setSDR_useConfigLaserWavelength(true);
		}
		
                // FCE update -- Move earth FCE detection here
                if(CorrectionParameter.performFringeCountErrorHandling)
                {
                    refScene = SpectraManager::isReferenceScene(*newSpectra);


                    if (requiresPhaseSync[oldFOV][oldBand][oldPSD] == false &&
                        !earthScenefceDetectionComplete[oldScene])
                    {
                        if( !refScene && oldBand == Band::LONGWAVE)
                        {
                            // Add batchEarthSceneFCE function
                            if(batchEarthSceneFCE(newSpectra,
                                   &rawSpectra, cfgParmsPtr))
                            {

                                char logMsg[EVENT_LOG_LENGTH];
                                sprintf(logMsg,"FCE:  Batch processing of %s %s"
                                " Fringe Count detected. All Bands and FOVs of "
                                "reference spectra for that FOR have been shifted.",
                                FOR[oldScene].c_str(), PSD[oldPSD].c_str());
                                EventLogEngine::append(logMsg);
                            }

                            earthScenefceDetectionComplete[oldScene] = true;

                        }
                    }
                }

                 //Calculate estimated NEdN
                if(GeneralParameter.requiresNEdN)
                {
                    const Spectra* hotRadiance =
                       processingPeriod.getRadianceSpectra(
                          SpectraManager::hotRadiance,
                          oldFOV,
                          oldBand);
                    const Spectra* coldRadiance =
                       processingPeriod.getRadianceSpectra(
                          SpectraManager::coldRadiance,
                          oldFOV,
                          oldBand);

                    UInt32 totalBins = newSpectra->getReal().size();

                    //check bin size match
                    //FCE update -- change hotReference to hotRefSpectra,
                    //coldReference to coldRefSpectra
                    if((hotRefSpectra.getReal().size() == totalBins)  &&
                       (coldRefSpectra.getReal().size() == totalBins) &&
                       (hotRadiance->getReal().size() == totalBins)   &&
                       (coldRadiance->getReal().size() == totalBins))
                    {
                        //call estimate NEdN with the slopes calculated above
                        Spectra* nEdNReference =
                           processingPeriod.getReferenceSpectra(
                              SpectraManager::nednPrediction,
                              oldFOV,
                              oldBand,
                              oldPSD);

                        if (nEdNReference->getReal().size() == totalBins)
                        {
                           newSpectra->bindNEdN() = nEdNReference->getReal();
                        }
                        else
                        {

                           BOOST::vector<Float64>& NEdN_tmp =
                              newSpectra->bindNEdN();
                           NEdN_tmp.resize(totalBins);

                           for(UInt32 i=0; i<totalBins; i++)
                           {
                              NEdN_tmp[i] = MISS_FLOAT32_FILL;
                           }

                        }
                    }
                    else
                    {
                        BOOST::vector<Float64>& NEdN_tmp =
                           newSpectra->bindNEdN();
                        NEdN_tmp.resize(totalBins);

                        for(UInt32 i=0; i<totalBins; i++)
                        {
                           NEdN_tmp[i] = MISS_FLOAT32_FILL;
                        }

                        newSpectra->setSDR_ExcessiveNEdN(true);
                    }
                }

                //apply user selected clipping and apply polarization correction
                if (CorrectionParameter.userSelectedClipping)
                {
                    if(!newSpectra->clipGuardBands(
                       CorrectionParameter.edrMinimumWavenumber[oldBand],
                       CorrectionParameter.edrMaximumWavenumber[oldBand]))
                    {
                       newSpectra->setSDR_InvalidData(true);
                    }

                    else 
                    {
                      if (CorrectionParameter.applyPolarizationCorrections)
                      {

                         CalibratedSpectra ictSpectra(*(         
                             processingPeriod.getRadianceSpectra(
                                SpectraManager::hotRadiance,     
                                oldFOV,                          
                                oldBand)));                      
                     
                         if(CorrectionParameter.calibrationOrder == RadCalFirst_ATBD_versionA   ||
                            CorrectionParameter.calibrationOrder == RadCalFirst_INVsaFPf_bigN   ||
                            CorrectionParameter.calibrationOrder == RadCalFirst_FPfINVsaPf_bigN )
                         { 
                           // ICT in sensor grid need to apply SA correction
                           // and resampling to user grid 
             
                           UInt32 totalBins = ictSpectra.getReal().size();
                           BOOST::vector<Float64> ictRealWavenumberBin;            
                           ictRealWavenumberBin = ictSpectra.bindReal();           
                           BOOST::matrix<Float64> testConvMatrix(totalBins, 1);    

                           if ( calibrationMatrix[oldFOV][oldBand].size1() == totalBins &&
                                   calibrationMatrix[oldFOV][oldBand].size2() == totalBins)
                           {
                              BOOST::matrix_column<BOOST::matrix<Float64> >
                                  (testConvMatrix, 0) = ictRealWavenumberBin;
                              BOOST::matrix<Float64> ictMatrix_real(totalBins, 1);
                              ictMatrix_real = BOOST::prod(calibrationMatrix[oldFOV][oldBand],
                                                testConvMatrix);
                              ictRealWavenumberBin = BOOST::matrix_column
                                   <BOOST::matrix<Float64> > (ictMatrix_real, 0);
            
                              for(UInt32 wavenumberBin = 0; wavenumberBin < totalBins; wavenumberBin++)
                              {
                                 ictSpectra.bindReal()[wavenumberBin] = 
                                        ictRealWavenumberBin[wavenumberBin];
                              } 
                           
                              // update First Wavenumber (value will change
                              // if resampling Matrix applied)
                              ictSpectra.setFirstWavenumber(
                                  correctionTable[oldFOV][oldBand].
                                  getUserLowestWavenumber(oldBand));
                              // update Wavenumber bin size (value will change
                              // if resampling Matrix applied)
                              ictSpectra.setWavenumberSpacing(
                                  correctionTable[oldFOV][oldBand].
                                  getUserDeltaSigma(oldBand));
                    
                           }
                         }

                         // ict in user grid
                         // apply user selected clipping for ict spectra                                                  
                         if(!ictSpectra.clipGuardBands(
                            CorrectionParameter.edrMinimumWavenumber[oldBand],
                            CorrectionParameter.edrMaximumWavenumber[oldBand]))
                         {
                            newSpectra->setSDR_InvalidData(true);
                         }
                         else
                         {
                            processingPeriod.polarizationCorrection(newSpectra, &ictSpectra);
                         }
                      }     
                    }
                }

                //apply Thermal drift monitoring
                newSpectra->setSDR_ThermalDriftExcess(
                   theScienceCalibrationRecord.hasExcessThermalDrift() ||
                   theTargetTempCalibrationRecord.hasExcessThermalDrift());

                //validate temporal alignment
                if(!CalibrationParameter.disableTimeStampBasedMovingWindow)
                {
                    if(processingPeriod.getWindowDepth(oldFOV, oldBand,
                       InstrumentCharacteristic.hotReferenceIdentifer,
                          oldPSD) == windowSize)
                    {
                        int hotCenterPosition = processingPeriod.
                           getWindowDepth(oldFOV, oldBand,
                              InstrumentCharacteristic.hotReferenceIdentifer,
                                 oldPSD) / 2;
                        int coldCenterPosition = processingPeriod.
                           getWindowDepth(oldFOV, oldBand,
                              InstrumentCharacteristic.coldReferenceIdentifer,
                                 oldPSD) / 2;
                        SYSTEMTIME hotCenter = {0,0,0,0,0,0,0,0};
                        SYSTEMTIME coldCenter= {0,0,0,0,0,0,0,0};

                        if(processingPeriod.getWindowDepth(oldFOV, oldBand,
                           InstrumentCharacteristic.hotReferenceIdentifer,
                              oldPSD) > hotCenterPosition)
                        {
                            hotCenter  = processingPeriod.getMeasuredSpectra(
                               hotCenterPosition + 1, oldFOV, oldBand,
                                  InstrumentCharacteristic.
                                     hotReferenceIdentifer,
                                        oldPSD)->getFrameTime();

                        }

                        if(processingPeriod.getWindowDepth(oldFOV, oldBand,
                           InstrumentCharacteristic.coldReferenceIdentifer,
                              oldPSD) > coldCenterPosition)
                        {
                            coldCenter = processingPeriod.getMeasuredSpectra(
                               coldCenterPosition + 1, oldFOV, oldBand,
                                  InstrumentCharacteristic.
                                     coldReferenceIdentifer,
                                        oldPSD)->getFrameTime();
                        }

                        SYSTEMTIME calTime    = SystemTimeUtilities::subtract(
                           newSpectra->getFrameTime(), hotCenter);

                        if(SystemTimeUtilities::quickTest(
                           calTime, validCalTargetTolerence,GT))
                        {
                            newSpectra->setSDR_InvalidData(true);

                            //missed center of hot reference
                            char logMessage[EVENT_LOG_LENGTH];
                            sprintf(logMessage,
                               "TIME:  %s %s %s calibration failed HOT time "
                               "validation with %02d:%02d:%02d.%03d",
                               BAND[oldBand].c_str(),
                               FOV[oldFOV].c_str(),
                               FOR[oldScene].c_str(),
                               calTime.wHour,
                               calTime.wMinute,
                               calTime.wSecond,
                               calTime.wMilliseconds);
                            (void)EventLogEngine::append(logMessage);

                        }

                        calTime    = SystemTimeUtilities::subtract(
                           newSpectra->getFrameTime(), coldCenter);

                        if(SystemTimeUtilities::quickTest(
                           calTime, validCalTargetTolerence,GT))
                        {
                            newSpectra->setSDR_InvalidData(true);

                            //missed center of cold reference
                            char logMessage[EVENT_LOG_LENGTH];
                            sprintf(logMessage,
                               "TIME:  %s %s %s calibration failed COLD time "
                               "validation with %02d:%02d:%02d.%03d",
                               BAND[oldBand].c_str(),
                               FOV[oldFOV].c_str(),
                               FOR[oldScene].c_str(),
                               calTime.wHour,
                               calTime.wMinute,
                               calTime.wSecond,
                               calTime.wMilliseconds);
                            (void)EventLogEngine::append(logMessage);

                        }
                    }
                    else
                    {
                        // time validation was not requested
                    }
                }
            }
            else
            {
                //this was a reference scene and cannot be calibrated
                if(InstrumentCharacteristic.
                   performLinearityCorrectionControl[oldBand] &&
                   !CorrectionParameter.performFringeCountErrorHandling)
                {
                   //LinErr adjustment to this ref scene
                   //when FCE handling is NOT active
                   //(correction is not available to reference
                   //scenes during FCE handling)
                   newSpectra->applyLinearityErrorCorrection();
                }
            }
        }
        else
        {
            //NO CALIBRATION
            if(GeneralParameter.requiresNEdN)
            {
               SceneElement     thisFOV  = newSpectra->getFieldOfView();
               Band::Wavelength thisBand = newSpectra->getBand();
               FieldOfRegard    thisFOR  = newSpectra->getFieldOfRegard();
               SweepDirection   thisPSD  = newSpectra->getSweepDirection();
               newSpectra->generateUSN(
                  processingPeriod.
                     getRawNoCalSum(thisFOR, thisFOV, thisBand, thisPSD),
                     processingPeriod.getRawNoCalSquareSum(
                     thisFOR, thisFOV, thisBand, thisPSD),processingPeriod.
                     getWindowDepth(thisFOV, thisBand, thisFOR, thisPSD));
            }

            //apply user selected clipping
            if (CorrectionParameter.userSelectedClipping)
            {
                (void)newSpectra->clipGuardBands(
                   CorrectionParameter.edrMinimumWavenumber[oldBand],
                   CorrectionParameter.edrMaximumWavenumber[oldBand]);

                //apply user selected resampling
                if(CorrectionParameter.applyResamplingMatrix)
                {
                    (void)newSpectra->resample(
                       CorrectionParameter.edrMinimumWavenumber[oldBand],
                       CorrectionParameter.edrMaximumWavenumber[oldBand],
                       CorrectionParameter.edrDeltaSigma[oldBand]);
                }
            }
            else if(CorrectionParameter.applyResamplingMatrix)
            {
                //only resample
                (void)newSpectra->resample(
                    lowestWavenumber[oldBand],
                    lowestWavenumber[oldBand]
                    + (deltaSigma[oldBand] * newSpectra->getReal().size()),
                    CorrectionParameter.edrDeltaSigma[oldBand]);
            }
        }

        //keep count of total invalid spectra
        if(newSpectra->getSDR_InvalidData())
        {
            invalidSpectraCount[oldFOV][oldBand]++;
        }
    }

    return newSpectra;
}

// ##METHOD HEADER-START## ****************************************************
//
// ARGUMENTS:
//  INPUTS:
//   None
//
//  OUTPUTS:
//   bool updateCMO - true tells Science Data Processor to update CMO
//
// DESCRIPTION:
//  Monitors the laser diode to determine if the CMO needs to be updated.
//  This method also updates the SDR Monitored Laser Wavelength and
//  SDR Spectral Resampling Laser Wavelength
//
//*** ##METHOD HEADER-END## ***************************************************
bool ScienceDataProcessor::monitorLaserDiode()
{
    bool updateCMO = false;

    static Float64 milConversion = 1000000;

    if(theScienceCalibrationRecord.isLaserDiodeCurrentPeriodFull() &&
       theScienceCalibrationRecord.isLaserDiodeCurrentPeriodValid() &&
       theScienceCalibrationRecord.isLaserDiodeTempPeriodFull() &&
       theScienceCalibrationRecord.isLaserDiodeTempPeriodValid())
    {
        Float64 deltaLaserTemp =
           (theScienceCalibrationRecord.
              getPreviousAveragedLaserDiodeTemp() -
                 theScienceCalibrationRecord.getAveragedLaserDiodeTemp());
        Float64 deltaLaserCurrent =
           (theScienceCalibrationRecord.
              getPreviousAveragedLaserDiodeCurrent() -
                 theScienceCalibrationRecord.getAveragedLaserDiodeCurrent());

        Float64 laserDrift =
           (theEngineeringCalibrationRecord.
              getEngCalRec_ScienceTelemetryConversionCoefficients().
                 getLaserDiodeTemperatureSlope() * deltaLaserTemp)  +
           (theEngineeringCalibrationRecord.
              getEngCalRec_ScienceTelemetryConversionCoefficients().
                 getLaserDiodeBiasCurrentSlope() * deltaLaserCurrent);

        Float64 lambdaMonitored =
           Spectra::LASER_WAVELENGTH[Band::LONGWAVE] * (1.0 +
              (laserDrift / milConversion));

        Float64 lambdaMonitoredResampling =
           Spectra::LASER_WAVELENGTH[Band::LONGWAVE] /
              InstrumentCharacteristic.numberSamplesPerLaserWavelength;

        Float64 lambdaMonitoringDrift = fabs(((lambdaMonitoredResampling -
           (lambdaMonitored / InstrumentCharacteristic.
              numberSamplesPerLaserWavelength)) / (lambdaMonitored /
                 InstrumentCharacteristic.numberSamplesPerLaserWavelength)) *
                    milConversion);

        Float64 laserWavelengthTolerance =
           (CorrectionParameter.laserDiodeWavelengthOrigin == LASER_CONFIG) ?
              (AlgorithmParameter.laserWavelengthDriftTolerance) :
              (theEngineeringCalibrationRecord.
                 getEngCalRec_ScienceTelemetryLimits().
                    getLaserDiodeWavelengthDriftLimit());

        char logText[EVENT_LOG_LENGTH];
        sprintf(logText,
           "INFO:  8sec Monitored Wavelength = %f", lambdaMonitored);
        EventLogEngine::append(logText);

        if (FloatCompare<Float64>::equal(lambdaMonitored,0.0) == false &&
            lambdaMonitoringDrift > laserWavelengthTolerance)
        {
            if (!CorrectionParameter.disableLaserMonitoring)
            {
                updateCMO = true;
            }

            char logMessage[EVENT_LOG_LENGTH];
            sprintf(logMessage,
               "NOTE:  8sec Laser Drift %f > %f, Laser Wavelength = %f ",
               lambdaMonitoringDrift,
               laserWavelengthTolerance,
               lambdaMonitored);
            EventLogEngine::append(logMessage);
        }

        for (Int32 bandIndex = 0; bandIndex < Band::TOTAL_BANDS; bandIndex++)
        {
            for (Int32 fovIndex = 0; fovIndex < MAX_FOV; fovIndex++)
            {
                for (Int32 forIndex = 0; forIndex < TOTAL_SCENES; forIndex++)
                {
                    for (Int32 psdIndex = 0;
                       psdIndex < BOTH_DIRECTIONS; psdIndex++)
                    {
                        Spectra* oldSpectra = processingPeriod.
                           getMeasuredSpectra(
                              processingPeriod.getWindowDepth(
                                                  (SceneElement)fovIndex,
                                                  (Band::Wavelength)bandIndex,
                                                  (FieldOfRegard)forIndex,
                                                  (SweepDirection)psdIndex),
                             (SceneElement)fovIndex,
                             (Band::Wavelength)bandIndex,
                             (FieldOfRegard)forIndex,
                             (SweepDirection)psdIndex);

                        if (oldSpectra)
                        {
                            oldSpectra->
                               setMonitoredLaserWavelength(lambdaMonitored);
                            oldSpectra->SDR_Status.
                               lambda_monitored_quality =
                                (lambdaMonitoringDrift >
                                   laserWavelengthTolerance) ? 1 : 0;
                            if (updateCMO)
                            {
                                oldSpectra->
                                   setSpectralResamplingLaserWavelength(
                                      lambdaMonitored /
                                         InstrumentCharacteristic.
                                            numberSamplesPerLaserWavelength);
                            }
                        }
                    }
                }
            }

            if (updateCMO)
            {
                calculateWavenumberBinSpacing((Band::Wavelength)bandIndex);
            }
        }
    }

    return updateCMO;
}



// ##METHOD HEADER-START## ****************************************************
//
// ARGUMENTS:
//  INPUTS:
//   None
//
//  OUTPUTS:
//   bool laserWavelengthUpdate - true if the laser wavelength should be updated
//
// DESCRIPTION:
//  Determines if the laser wavelength needs to be updated
//  and updates if needed.
//
//*** ##METHOD HEADER-END## ***************************************************
bool ScienceDataProcessor::monitorLaserWavelength()
{
    static Float64 ppmConversion = 0.000001;
    bool laserWavelengthUpdate = false;

    // If laser wavelength has not been updated beyond
    // initialization value, set it to the value in the
    // engineering packet if one is available.
    if ((Spectra::usingDefaultLW) && (DMS_Data_Inputs->EngPtr_in != 0))
    {
        Spectra::configureLaserWavelength(theEngineeringCalibrationRecord.
           getEngCalRec_LaserMetrologyInfo().getLaserWavelength());
        Spectra::buildResamplingWavelength(InstrumentCharacteristic.
           numberSamplesPerLaserWavelength);
        Spectra::usingDefaultLW = false;

        char logMessage[EVENT_LOG_LENGTH];
        sprintf(logMessage,
           "monitorLaserWavelength(): Laser Wavelength UPDATED from CMO "
           "value to LW(%f) MW(%f) SW(%f)",
           Spectra::LASER_WAVELENGTH[Band::LONGWAVE],
           Spectra::LASER_WAVELENGTH[Band::MIDWAVE],
           Spectra::LASER_WAVELENGTH[Band::SHORTWAVE]);
        (void)EventLogEngine::append(logMessage);

        calculateWavenumberBinSpacing(Band::LONGWAVE);
        calculateWavenumberBinSpacing(Band::MIDWAVE);
        calculateWavenumberBinSpacing(Band::SHORTWAVE);
    }

    Float64 actualDrift =
       fabs((Spectra::LASER_WAVELENGTH[Band::LONGWAVE] -
             theEngineeringCalibrationRecord.getAverageMetrologyWavelength()) /
             theEngineeringCalibrationRecord.getAverageMetrologyWavelength());
    Float64 tolerence   =
       AlgorithmParameter.laserWavelengthDriftTolerance * ppmConversion;

    if (theEngineeringCalibrationRecord.getRejectedWavelengthsRatio() >
       AlgorithmParameter.maximumFractionRejections)
    {
        //Invalid neon calibration has been detected
        char logMessage[EVENT_LOG_LENGTH];
        sprintf(logMessage,
           "WARN::Invalid Neon Calibration has been detected. (%f%% Rejected)",
           theEngineeringCalibrationRecord.getRejectedWavelengthsRatio() * 100);
        (void)EventLogEngine::append(logMessage);

        Spectra::configureInvalidNeonCal(true);
    }
    else if (actualDrift > tolerence)
    {
        char logMessage[EVENT_LOG_LENGTH];
        sprintf(logMessage,
                "NOTE::Neon Calibration Data (%f) is (%.3f > %.3f ppm)",
                theEngineeringCalibrationRecord.getAverageMetrologyWavelength(),
                actualDrift / ppmConversion,
                AlgorithmParameter.laserWavelengthDriftTolerance);
        EventLogEngine::append(logMessage);

        // If laser wavelength is set to initialization value here,
        // we are starting up and there is no
        // initial CMO, so set it from the engineering packet here
        // and it will be set with the first correction
        // matrix build and read from there after.
        if (Spectra::usingDefaultLW)
        {
            Spectra::configureLaserWavelength(theEngineeringCalibrationRecord.
               getAverageMetrologyWavelength());
            Spectra::buildResamplingWavelength(InstrumentCharacteristic.
               numberSamplesPerLaserWavelength);
            Spectra::usingDefaultLW = false;

            sprintf(logMessage,
               "monitorLaserWavelength(): Laser Wavelength UPDATED "
               "to LW(%f) MW(%f) SW(%f)",
               Spectra::LASER_WAVELENGTH[Band::LONGWAVE],
               Spectra::LASER_WAVELENGTH[Band::MIDWAVE],
               Spectra::LASER_WAVELENGTH[Band::SHORTWAVE]);
            (void)EventLogEngine::append(logMessage);
        }

        laserWavelengthUpdate = true;

        //saving the Laser Diode Temperature and Current from the
        //Science Data Packet
        theScienceCalibrationRecord.setCaptureNextLaserInfo();

        Spectra::configureInvalidNeonCal(false);

        //since wavelength changed recalculate sigma min and delta
        //sigma for each band
        calculateWavenumberBinSpacing(Band::LONGWAVE);
        calculateWavenumberBinSpacing(Band::MIDWAVE);
        calculateWavenumberBinSpacing(Band::SHORTWAVE);
    }
    else
    {
        char logMessage[EVENT_LOG_LENGTH];
        sprintf(logMessage,
           "NOTE::Neon Calibration Data (%f) is (%.3f < %.3fppm). "
           "Current CMO maintained...",
           theEngineeringCalibrationRecord.getAverageMetrologyWavelength(),
           actualDrift / ppmConversion,
           AlgorithmParameter.laserWavelengthDriftTolerance);
        EventLogEngine::append(logMessage);
        sprintf(logMessage,
           "ATTN::Neon Calibration validity maintained LW(%f) MW(%f) SW(%f)",
           Spectra::LASER_WAVELENGTH[Band::LONGWAVE],
           Spectra::LASER_WAVELENGTH[Band::MIDWAVE],
           Spectra::LASER_WAVELENGTH[Band::SHORTWAVE]);
        EventLogEngine::append(logMessage);

        Spectra::configureInvalidNeonCal(false);
    }

    return laserWavelengthUpdate;
}



// ##METHOD HEADER-START## ****************************************************
//
// ARGUMENTS:
//  INPUTS:
//   int newSize  -  the moving window average size to maintain
//
//  OUTPUTS:
//   none
//
// DESCRIPTION:
//  Propagates temporal window size to assistant classes
//
//*** ##METHOD HEADER-END## ***************************************************
void ScienceDataProcessor::setWindowSize(int newSize)
{
    windowSize = newSize;

    this->processingPeriod.setWindowSize(newSize);
    this->theEngineeringCalibrationRecord.setWindowSize(newSize);
    this->theScienceCalibrationRecord.setWindowSize(newSize);
    this->theTargetTempCalibrationRecord.setWindowSize(newSize);
}


// ##METHOD HEADER-START## ****************************************************
//
// ARGUMENTS:
//  INPUTS:
//     UINT referenceSize     -  forced reference size
//     UINT targetTempSize    -  forced temperature size
//     UINT calibrationSize   -  forced scene size
//
//  OUTPUTS:
//   none
//
// DESCRIPTION:
//  Propagates temporal window size to assistant classes
//
//*** ##METHOD HEADER-END## ***************************************************
void ScienceDataProcessor::declareWindowSize(UInt32 referenceSize,
                                             UInt32 targetTempSize,
                                             UInt32 calibrationSize)
{
    windowSize = referenceSize;

    this->processingPeriod.declareWindowSize(referenceSize, calibrationSize);
    this->theScienceCalibrationRecord.setWindowSize(targetTempSize);
    this->theTargetTempCalibrationRecord.setWindowSize(targetTempSize);
}

// ##METHOD HEADER-START## ****************************************************
//
// ARGUMENTS:
//  INPUTS:
//   None
//
//  OUTPUTS:
//   bool
//
// DESCRIPTION:
//  Loads EngPkt from a DMS
//
//*** ##METHOD HEADER-END## ***************************************************
bool ScienceDataProcessor::loadEngPkt()
{
    bool retVal = false;

    const EngCalRec_ScienceTelemetryConversionCoefficients* theConvCoeff =
       &theEngineeringCalibrationRecord.
          getEngCalRec_ScienceTelemetryConversionCoefficients();
    const EngCalRec_ScienceTelemetryLimits* theSciTeleLimits =
       &theEngineeringCalibrationRecord.getEngCalRec_ScienceTelemetryLimits();

    calculateWavenumberBinSpacing(Band::LONGWAVE);
    calculateWavenumberBinSpacing(Band::MIDWAVE);
    calculateWavenumberBinSpacing(Band::SHORTWAVE);

    // Check if anything retrieved from DMS
    if ((DMS_Data_Inputs->EngPtr_in != 0) && 
            (DMS_Data_Inputs->EngPtr_in != DM_BADADDRESS))
    {
        //--------------------------------------------------------------------
        // DMS data is not in the format needed for re-populating
        // the CrIS Science data structures so transform the DMS input:
        //
        // First, convert to archive version of eng type
        CrISEngAr* pEng = static_cast<CrISEngAr *>(&(DMS_Data_Inputs->
           EngPtr_in->theEng));

        // Then, create an intermediate output archive so the rest of the code
        //  can extract as it did from the BOOST archive
        CrISDataStream ss;
        cris_oarchive car(ss);
        pEng->serializeEng(car);

        // Make an input archive of the data.
        cris_iarchive icar(ss);

        retVal = true;

        (void)EventLogEngine::append("NOTE:  Engineering packet load started");

        theEngineeringCalibrationRecord.serialize(icar);

        engPacketCount = 1;  //register that a valid record was retrieved

        // baffleTemperatureOffset size is retrieved from Eng Packet
        // dynamically, we have to re-calculate it here
        if(CalibrationParameter.instrumentTemperatureOrigin == INSTRUMENT_TLM
           && CalibrationParameter.suppressSsmBaffleProfile == false)
        {
           if(theEngineeringCalibrationRecord.
              getEngCalRec_ICTEnvironmentalModel().
                 getEngCalBaffleTempOffsetDecimatedY().size() ==
                    static_cast<UInt32>(NUM_ECM_BAFFLE_PTS))
           {

              buildBaffleTemperatureOffset((&theEngineeringCalibrationRecord.
                 getEngCalRec_ICTEnvironmentalModel())->
                    getEngCalBaffleTempOffsetDecimatedY(),
                 (&theEngineeringCalibrationRecord.
                    getEngCalRec_ICTEnvironmentalModel())->
                       getEngCalOrbitalPeriod());

           }
           else
           {
              EventLogEngine::append(
                 "EngCal format doe not support USER Requested ECM Baffle "
                 "Compensation in loadEngPkt()");
           }
        }

        //Echos data from EngCalRec into SciCalRec
        theScienceCalibrationRecord.setIctPrt1Ro(theConvCoeff->getIctPrt1Ro());
        theScienceCalibrationRecord.setIctPrt1A(theConvCoeff->getIctPrt1A());
        theScienceCalibrationRecord.setIctPrt1B(theConvCoeff->getIctPrt1B());
        theScienceCalibrationRecord.setIctPrt2Ro(theConvCoeff->getIctPrt2Ro());
        theScienceCalibrationRecord.setIctPrt2A(theConvCoeff->getIctPrt2A());
        theScienceCalibrationRecord.setIctPrt2B(theConvCoeff->getIctPrt2B());
        theScienceCalibrationRecord.setIctLowRangeCalibrationResistorRo(
           theConvCoeff->getIctLowRangeCalibrationResistorRo());
        theScienceCalibrationRecord.setIctLowRangeCalibrationResistorA(
           theConvCoeff->getIctLowRangeCalibrationResistorA());
        theScienceCalibrationRecord.setIctHighRangeCalibrationResistorRo(
           theConvCoeff->getIctHighRangeCalibrationResistorRo());
        theScienceCalibrationRecord.setIctHighRangeCalibrationResistorA(
           theConvCoeff->getIctHighRangeCalibrationResistorA());
        theScienceCalibrationRecord.setIctCalibrationResistorRtdRo(
           theConvCoeff->getIctCalibrationResistorRtdRo());
        theScienceCalibrationRecord.setIctCalibrationResistorRtdA(
           theConvCoeff->getIctCalibrationResistorRtdA());
        theScienceCalibrationRecord.setLaserDiodeTemperatureSlope(
           theConvCoeff->getLaserDiodeTemperatureSlope());
        theScienceCalibrationRecord.setLaserDiodeBiasCurrentSlope(
           theConvCoeff->getLaserDiodeBiasCurrentSlope());
        theScienceCalibrationRecord.setBeamsplitterTempIntercept(
           theConvCoeff->getBeamsplitterTempIntercept());
        theScienceCalibrationRecord.setBeamsplitterTempSlope(
           theConvCoeff->getBeamsplitterTempSlope());
        theScienceCalibrationRecord.setScanMirrorTempIntercept(
           theConvCoeff->getScanMirrorTempIntercept());
        theScienceCalibrationRecord.setScanMirrorTempSlope(
           theConvCoeff->getScanMirrorTempSlope());
        theScienceCalibrationRecord.setScanBaffleTempIntercept(
           theConvCoeff->getScanBaffleTempIntercept());
        theScienceCalibrationRecord.setScanBaffleTempSlope(
           theConvCoeff->getScanBaffleTempSlope());
        theScienceCalibrationRecord.setOmaStructureTemp1Intercept(
           theConvCoeff->getOmaStructureTemp1Intercept());
        theScienceCalibrationRecord.setOmaStructureTemp1Slope(
           theConvCoeff->getOmaStructureTemp1Slope());
        theScienceCalibrationRecord.setOmaStructureTemp2Intercept(
           theConvCoeff->getOmaStructureTemp2Intercept());
        theScienceCalibrationRecord.setOmaStructureTemp2Slope(
           theConvCoeff->getOmaStructureTemp2Slope());
        theScienceCalibrationRecord.setTelescopeTempSlope(
           theConvCoeff->getTelescopeTempSlope());
        theScienceCalibrationRecord.setStage1CoolerTempSlope(
           theConvCoeff->getStage1CoolerTempSlope());
        theScienceCalibrationRecord.setStage2CoolerTempSlope(
           theConvCoeff->getStage2CoolerTempSlope());
        theScienceCalibrationRecord.setStage3CoolerTempSlope(
           theConvCoeff->getStage3CoolerTempSlope());
        theScienceCalibrationRecord.setStage4CoolerTempSlope(
           theConvCoeff->getStage4CoolerTempSlope());
        theScienceCalibrationRecord.setCrossTrackSsmPointingErrorIntercept(
           theConvCoeff->getCrossTrackSsmPointingErrorIntercept());
        theScienceCalibrationRecord.setCrossTrackSsmPointingErrorSlope(
           theConvCoeff->getCrossTrackSsmPointingErrorSlope());
        theScienceCalibrationRecord.setInTrackSsmPointingErrorIntercept(
           theConvCoeff->getInTrackSsmPointingErrorIntercept());
        theScienceCalibrationRecord.setInTrackSsmPointingErrorSlope(
           theConvCoeff->getInTrackSsmPointingErrorSlope());

        theScienceCalibrationRecord.setIctTemp1DriftLimit(
           theSciTeleLimits->getIctTemp1DriftLimit());
        theScienceCalibrationRecord.setIctTemp2DriftLimit(
           theSciTeleLimits->getIctTemp2DriftLimit());
        theScienceCalibrationRecord.setScanMirrorTempDriftLimit(
           theSciTeleLimits->getScanMirrorTempDriftLimit());
        theScienceCalibrationRecord.setBeamsplitterTemp1DriftLimit(
           theSciTeleLimits->getBeamsplitterTemp1DriftLimit());
        theScienceCalibrationRecord.setScanBaffleTempDriftLimit(
           theSciTeleLimits->getScanBaffleTempDriftLimit());
        theScienceCalibrationRecord.setOmaStruct1TempDriftLimit(
           theSciTeleLimits->getOmaStruct1TempDriftLimit());
        theScienceCalibrationRecord.setOmaStruct2TempDriftLimit(
           theSciTeleLimits->getOmaStruct2TempDriftLimit());
        theScienceCalibrationRecord.setTelescopeTempDriftLimit(
           theSciTeleLimits->getTelescopeTempDriftLimit());
        theScienceCalibrationRecord.setStage1CoolerTempDriftLimit(
           theSciTeleLimits->getStage1CoolerTempDriftLimit());
        theScienceCalibrationRecord.setStage2CoolerTempDriftLimit(
           theSciTeleLimits->getStage2CoolerTempDriftLimit());
        theScienceCalibrationRecord.setStage3CoolerTempDriftLimit(
           theSciTeleLimits->getStage3CoolerTempDriftLimit());
        theScienceCalibrationRecord.setStage4CoolerTempDriftLimit(
           theSciTeleLimits->getStage4CoolerTempDriftLimit());
        theScienceCalibrationRecord.setLaserDiodeWavelengthDriftLimit(
           theSciTeleLimits->getLaserDiodeWavelengthDriftLimit());

        //update laser wavelength from values in engineering packet
        if(CorrectionParameter.laserDiodeWavelengthOrigin == LASER_TLM)
        {
            theEngineeringCalibrationRecord.calculateMetrologyWavelength();
        }
        char logMessage[EVENT_LOG_LENGTH];
        sprintf(logMessage, "Laser Wavelength in EngPkt (%f)",
                theEngineeringCalibrationRecord.
                   getEngCalRec_LaserMetrologyInfo().getLaserWavelength());
        (void)EventLogEngine::append(logMessage);

        calculateWavenumberBinSpacing(Band::LONGWAVE);
        calculateWavenumberBinSpacing(Band::MIDWAVE);
        calculateWavenumberBinSpacing(Band::SHORTWAVE);

    }
    else
    {
        (void)EventLogEngine::append(
           "WARN::Engineering Calibration failed to load");
    }
    return retVal;
}

// ##METHOD HEADER-START## ****************************************************
//
// ARGUMENTS:
//  INPUTS:
//   None
//
//  OUTPUTS:
//   bool
//
// DESCRIPTION:
//  Loads invSA matrix and ILSParameters from a DMS
//
//*** ##METHOD HEADER-END## ***************************************************

bool ScienceDataProcessor::loadCMOs()
{
    bool retVal = false;

    // Check if anything retrieved from DMS
    if ((DMS_Data_Inputs->CMOPtr_in != 0) && 
            (DMS_Data_Inputs->CMOPtr_in != DM_BADADDRESS))
    {
        retVal = true;
        (void)EventLogEngine::append("NOTE:  Correction Matrix reading");

        // Load the CMO data from DMS
        loadSerializedCorrectionMatrixFromDMS();

        (void)EventLogEngine::append(
           "NOTE:  Correction Matrix reading finished");

    }
    else
    {
        (void)EventLogEngine::append("WARN::Correction Matrix failed to load");
    }
    return retVal;
}


// ##METHOD HEADER-START## ****************************************************
//
// ARGUMENTS:
//  INPUTS:
//   None
//
//  OUTPUTS:
//   none
//
// DESCRIPTION:
//  calculate desired band center, and configure spectra's desired band center
//
//*** ##METHOD HEADER-END## ***************************************************
void ScienceDataProcessor::calculateDesiredBandCenter()
{
    Float64 lwDesiredBandCenter =
       (CorrectionParameter.edrMaximumWavenumber[Band::LONGWAVE] +
          CorrectionParameter.edrMinimumWavenumber[Band::LONGWAVE]) / 2;
    Float64 mwDesiredBandCenter =
       (CorrectionParameter.edrMaximumWavenumber[Band::MIDWAVE] +
          CorrectionParameter.edrMinimumWavenumber[Band::MIDWAVE]) / 2;
    Float64 swDesiredBandCenter =
       (CorrectionParameter.edrMaximumWavenumber[Band::SHORTWAVE] +
          CorrectionParameter.edrMinimumWavenumber[Band::SHORTWAVE]) / 2;

    //Configure spectra's static desired band center variable
    Spectra::configureDesiredBandCenter(
       lwDesiredBandCenter, mwDesiredBandCenter, swDesiredBandCenter);

    //Configure the scienceDataProcessors desired band center
    desiredBandCenter[Band::LONGWAVE] = lwDesiredBandCenter;
    desiredBandCenter[Band::MIDWAVE] = mwDesiredBandCenter;
    desiredBandCenter[Band::SHORTWAVE] = swDesiredBandCenter;

}

/////////1/////////2/////////3/////////4/////////5/////////6/////////7/////////8
// NAME
//    ScienceDataProcessor::saveSerializedCorrectionMatrix
//
// DESCRIPTION
//    This method archives the CrIS Correction Matrix to DMS.
//    The data is serialized and streamed to an output buffer.
//    This buffer needs to be copied to the DMS shared-memory block.
//
// ARGUMENTS
//     none
//
/////////1/////////2/////////3/////////4/////////5/////////6/////////7/////////8
void ScienceDataProcessor::saveSerializedCorrectionMatrix()
{

    std::stringstream oss(std::ios_base::out|std::ios_base::in|std::ios_base::binary);
    CrISDataStream ss;

    // Aggregate algorithm CMO data via BOOST Archive class
    serializeCmo(ss);

    // Extract CrISCMOEngStruct from BOOST archive for DMS
    archiveCmo(ss);

}

////////////////////////////////////////////////////////////////////////////////
// NAME
//     ScienceDataProcessor::serializeCmo
//
// DESCRIPTION
//     This method serializes the Correction Matrix data.
//
//
////////////////////////////////////////////////////////////////////////////////
void ScienceDataProcessor::serializeCmo(CrISDataStream& ss)
{

    cris_oarchive car(ss);

    //1 >>----> Add the Correction Matrix
    UInt16 twoBytes = 0;

    car & twoBytes;
    car & twoBytes;

    for(Int32 bandIndex = 0; bandIndex < Band::TOTAL_BANDS; ++bandIndex)
    {
        for (Int32 fovIndex = 0; fovIndex < MAX_FOV; ++fovIndex)
        {
            this->correctionTable[fovIndex][bandIndex].serializeCMO(car,
                                               (Band::Wavelength)bandIndex);
        }
    }

    //2 >>----> Add padding between CMO and ILS params
    Int32 padding = 0;
    car & padding;

    //3 >>----> write the ILS parameters data
    this->theILSParametersRecord.serialize(car);

    //2 >>----> Add padding that is in the struct after the ILS params
    car & padding;
}

////////////////////////////////////////////////////////////////////////////////
// NAME
//     ScienceDataProcessor::archiveCmo
//
// DESCRIPTION
//     This method streams the CMO data to the CMO output buffer.
//
// ARGUMENTS
//     CrISDataStream& ss -- stringstream archive container
//
////////////////////////////////////////////////////////////////////////////////
void ScienceDataProcessor::archiveCmo(CrISDataStream& ss)
{
    size_t dataSize = ss.str().size();

    if(dataSize == sizeof(CrISCMOStruct))
    {
        // Initialize Correction Matrix Memory
        if(this->DMS_Data_Inputs->CMOPtr_out == 0)
        {
            // Deleted in ProSdrCris::doProcessing after copied to
            // DMS shared memory
            this->DMS_Data_Inputs->CMOPtr_out = new CrISCMOStruct;
        }

        // Write stream data to CMOPtr_out
        char *pSink = reinterpret_cast<char*>(DMS_Data_Inputs->CMOPtr_out);
        ss.read(pSink, dataSize);
    }
    else
    {
       char logMessage[EVENT_LOG_LENGTH];
       sprintf(logMessage, "ERROR searialzed archiveCmo dataSize (%zu) "
          "differs from cmoSize (%lu)", dataSize,sizeof(CrISCMOStruct));
       EventLogEngine::append(logMessage);
    }

}

void ScienceDataProcessor::saveSerializedEngPkt()
{
    std::stringstream oss(std::ios_base::out|std::ios_base::in|std::ios_base::binary);
    CrISDataStream ss;

    // write the ILS parameters data
    cris_oarchive car(ss);
    this->theEngineeringCalibrationRecord.serialize(car);

    // Extract CrISCMOEngStruct from archive for DMS
    size_t dataSize = ss.str().size();

    if(dataSize == sizeof(CrISEngStruct))
    {
        // Initialize Correction Matrix Memory
        if (this->DMS_Data_Inputs->EngPtr_out == 0)
        {
            // Deleted in ProSdrCris::doProcessing after copied to
            // DMS shared memory
            this->DMS_Data_Inputs->EngPtr_out = new CrISEngStruct;
        }

        // Write stream data to EngPtr_out
        char *pSink = reinterpret_cast<char*>(DMS_Data_Inputs->EngPtr_out);
        ss.read(pSink, dataSize);
    }
    else
    {
       char logMessage[EVENT_LOG_LENGTH];
       sprintf(logMessage,
          "ERROR serialzed archive Eng Pkt dataSize (%zu) differs "
          "from the struct size (%lu)", dataSize,sizeof(CrISEngStruct));
       EventLogEngine::append(logMessage);
    }

}


// ##METHOD HEADER-START## ****************************************************
//
// ARGUMENTS:
//  INPUTS:
//
//  OUTPUTS:
//   none
//
// DESCRIPTION:
//  Loads serialized CMO.  Populates correction matrix tables and
//  ILS parameters from the CMO data.
//
//*** ##METHOD HEADER-END## ***************************************************
void ScienceDataProcessor::loadSerializedCorrectionMatrixFromDMS()
{
    // put CMO into archive format
    CrISCmoAr *pCmo = static_cast<CrISCmoAr *>(&(DMS_Data_Inputs->CMOPtr_in->
       theCMO));
    CrISDataStream ssCmo;
    cris_oarchive carCmo(ssCmo);
    pCmo->serializeCMO(carCmo);

    // put CMO archive into de-serializer
    cris_iarchive icarCmo(ssCmo);

    // Strip "pad-version" bytes
    UInt16 twoBytes = 0;
    icarCmo & twoBytes; // Pad Version
    icarCmo & twoBytes; // Version

    for(Int32 bandIndex = 0; bandIndex < Band::TOTAL_BANDS; ++bandIndex)
    {
        for (Int32 fovIndex = 0; fovIndex < MAX_FOV; ++fovIndex)
        {
            correctionTable[fovIndex][bandIndex].
               serializeCMO(icarCmo, (Band::Wavelength)bandIndex);
        }
    }

    // put ILS data into archive format
    CrISILSAr *pILS = static_cast<CrISILSAr *>(&(DMS_Data_Inputs->CMOPtr_in->
       theILS));
    CrISDataStream ssIls;
    cris_oarchive carIls(ssIls);
    pILS->serializeILS(carIls);

    // put ILS archive into de-serializer
    cris_iarchive icarIls(ssIls);

    // store to struct
    theILSParametersRecord.serialize(icarIls);
}


// ##METHOD HEADER-START## ****************************************************
//
// ARGUMENTS:
//  INPUTS:
//   Spectra*
//
//  OUTPUTS:
//   Spectra*
//
// DESCRIPTION:
//  Compute field of views line-of-sight in spacecraft body frame.
//
//*** ##METHOD HEADER-END## ***************************************************
Spectra* ScienceDataProcessor::
         calculateGeometricCalibration(Spectra* newSpectra)
{
    Float64 commandedCrTrk;
    Float64 commandedInTrk;
    Float64 servoErrCrTrk;
    Float64 servoErrInTrk;
    Float64 inTrkAngle;
    Float64 crTrkAngle;
    Float64 dotProdAns;
    FieldOfRegard theScene = newSpectra->getFieldOfRegard();
    SceneElement theFov = newSpectra->getFieldOfView();
    EngCalRec_PitchRollYawInfo geolocationInfo =
       theEngineeringCalibrationRecord.getEngCalRec_PitchRollYawInfo();
    BOOST::matrix<Float64> normalSSMFMatrix(CRIS_TOTAL_BANDS, MATRIX_COL_SIZE);
    BOOST::matrix<Float64> normalRMFMatrix(CRIS_TOTAL_BANDS, MATRIX_COL_SIZE);
    BOOST::matrix<Float64> losInSBFMatrix(CRIS_TOTAL_BANDS, MATRIX_COL_SIZE);
    BOOST::matrix<Float64> losInSSMFRMatrix(CRIS_TOTAL_BANDS, MATRIX_COL_SIZE);
    BOOST::vector<Float64> normalSSMF(CRIS_TOTAL_BANDS);
    BOOST::vector<Float64> losInSSMFR(CRIS_TOTAL_BANDS);
    BOOST::vector<Float64> losInSBF(CRIS_TOTAL_BANDS);

    // Compute normal to SSM mirror in SSMF (fixed coordinate system)
    // by applying rotations from commanded angles (using FOR index)
    // and servo errors (in-track and cross-track)

    // Get commanded angles
    if (theScene == NonStandardDwell)
    {
        //for test support
        commandedCrTrk = geolocationInfo.
           getCommandedCrossTrackAngleNadir() * .000001;

        if (geolocationInfo.
           fourMinutePacketHasInTrackPositioning())  // Only do if new
                                                     // 4 minute packets
        {
           commandedInTrk = geolocationInfo.
              getCommandedInTrackAngleNadir() * .000001;
        }

        servoErrCrTrk = 0; //Nadir does not contain a value in the SciCalRec
        servoErrInTrk = 0; //Nadir does not contain a value in the SciCalRec
    }
    else
    {
        //get commanded CrossTrack angle for an ES and convert to uRad
        commandedCrTrk = geolocationInfo.
           getCommandedCrossTrackAngleES(theScene) * .000001;

        if (geolocationInfo.
           fourMinutePacketHasInTrackPositioning())  // Only do if new
                                                     // 4 minute packets
        {
           commandedInTrk = geolocationInfo.
              getCommandedInTrackAngleES(theScene) * .000001;
        }

        // Get converted servo errors
        servoErrCrTrk = theScienceCalibrationRecord.
           getConvertedCrossTrackServoErr(theScene);
        servoErrInTrk = theScienceCalibrationRecord.
           getConvertedInTrackServoErr(theScene);
    }

    //get commanded InTrack angle for an ES and convert to uRad
    if (!geolocationInfo.
       fourMinutePacketHasInTrackPositioning())  // Backwards compatible to old
                                                 // 4 minute packet
    {
        commandedInTrk = geolocationInfo.getCommandedIntrackAngle() * .000001;
    }

    // Compute commanded angles with servo errors
    crTrkAngle = commandedCrTrk + servoErrCrTrk;
    inTrkAngle = commandedInTrk + servoErrInTrk;

    //get normalRMF vector and place into a matrix in order to do matrix
    //matrix multiplication
    BOOST::matrix_column<BOOST::matrix<Float64> > (normalRMFMatrix, 0) =
        theEngineeringCalibrationRecord.getNormalRMF();

    // Apply rotations to mirror normal
    normalSSMFMatrix = BOOST::prod(
        theEngineeringCalibrationRecord.rotationMatrixX(crTrkAngle),
        normalRMFMatrix);
    normalSSMFMatrix = BOOST::prod(
        theEngineeringCalibrationRecord.rotationMatrixY(inTrkAngle),
        normalSSMFMatrix);
    normalSSMF =
        BOOST::matrix_column<BOOST::matrix<Float64> > (normalSSMFMatrix, 0);

    // Compute LOS in SBF

    // Compute reflected FOV LOS using mirror normal in SSMF
    dotProdAns = BOOST::inner_prod(
        theEngineeringCalibrationRecord.getLOSInSSMF((SceneElement)theFov),
        normalSSMF);

    //setting losInSSMFR = normalSSMFMatrix col 1
    losInSSMFR =
        BOOST::matrix_column<BOOST::matrix<Float64> > (normalSSMFMatrix, 0);
    losInSSMFR *= (2 * dotProdAns);
    losInSSMFR = theEngineeringCalibrationRecord.
       getLOSInSSMF((SceneElement)theFov) - losInSSMFR;

    // Compute CrIS FOV LOS in SBF coordinate system
    BOOST::matrix_column<BOOST::matrix<Float64> >
        (losInSSMFRMatrix, 0) = losInSSMFR;
    losInSBFMatrix = BOOST::prod(
        theEngineeringCalibrationRecord.getSBFToSSMF(),
        losInSSMFRMatrix);

    // Express LOS as pitch and roll GeoCal Data
    newSpectra->setLosPitchAngle(-asin(losInSBFMatrix(0,0)));
    newSpectra->
       setLosRollAngle(atan(losInSBFMatrix(1,0) / losInSBFMatrix(2,0)));

    return newSpectra;
}

void ScienceDataProcessor::refreshTime(CCSDSFormat* theTelemetryFormat)
{
    Int16 days = theTelemetryFormat->getSecondaryHeader()->days();
    Int32 milliseconds =
       theTelemetryFormat->getSecondaryHeader()->milliseconds();
    Int16 microseconds =
       theTelemetryFormat->getSecondaryHeader()->microseconds();

    Int64 ietTime = 0;
    UInt64 cdsTime = (UInt64)days << 48 | (UInt64)milliseconds << 16 |
        (UInt64)microseconds;

    ProCmnTimeUtil* timeUtil = ProCmnTimeUtil::getInstance();
    (void)timeUtil->cds2Iet(cdsTime,&ietTime);

    theEngineeringCalibrationRecord.setIETTime(ietTime);
}

// ##METHOD HEADER-START## ****************************************************
//
// ARGUMENTS:
//  INPUTS:
//   none
//
//  OUTPUTS:
//   none
//
// DESCRIPTION:
//  Synchronizes all windows based on reference spectra sequence counts,and
//  update sequences counts
//
//*** ##METHOD HEADER-END## ***************************************************
void ScienceDataProcessor::syncWindows()
{
    for(UInt32 fov = 0; fov < MAX_FOV; fov++)
    {
        for(UInt32 band = 0; band < Band::TOTAL_BANDS; band++)
        {
            for(UInt32 psd = 0; psd < BOTH_DIRECTIONS; psd++)
            {
                while (sequenceCount[fov][band][InstrumentCharacteristic.
                   hotReferenceIdentifer][psd] <
                      sequenceCount[fov][band][InstrumentCharacteristic.
                         coldReferenceIdentifer][psd])
                {
                    Spectra* fakeSpectra =
                       new CalibratedSpectra(
                          (SceneElement)fov,
                          (Band::Wavelength)band,
                          (FieldOfRegard)InstrumentCharacteristic.
                             hotReferenceIdentifer,
                          (SweepDirection)psd);

                    fakeSpectra->setSDR_InvalidData(true);
                    fakeSpectra->setSize((processingPeriod.
                       getMeasuredSpectra(
                          processingPeriod.getWindowDepth(
                           (SceneElement)fov,
                           (Band::Wavelength)band,
                           (FieldOfRegard)InstrumentCharacteristic.
                              hotReferenceIdentifer,
                           (SweepDirection)psd),

                       (SceneElement)fov,
                       (Band::Wavelength)band,
                       (FieldOfRegard)InstrumentCharacteristic.
                          hotReferenceIdentifer,
                       (SweepDirection)psd))->getReal().size());
                    fakeSpectra->setFrameTime(TelemetryProcessor::instance()->
                       getLastFrameTime());

                    // set values so fake spectra are added to the window
                    fakeSpectra->setSDR_CalibrationType(
                       GeneralParameter.spectralCalMethod);
                    fakeSpectra->setSDR_ApodType(
                       CorrectionParameter.apodizationType);
                    fakeSpectra->setSDR_CalOrder(CorrectionParameter.
                       calibrationOrder);

                    delete fakeSpectra;
                    fakeSpectra = 0;

                    sequenceCount[fov][band][InstrumentCharacteristic.
                       hotReferenceIdentifer][psd]++;
                    char testMsg[EVENT_LOG_LENGTH];
                    sprintf(testMsg,
                       "SYNC: %s %s %s %s Sequence count out of sync, adding "
                       "place holder spectra",
                       BAND[band].c_str(), FOV[fov].c_str(),
                       FOR[InstrumentCharacteristic.
                          hotReferenceIdentifer].c_str(),
                       PSD[psd].c_str());
                    EventLogEngine::append(testMsg);

                }

                while (sequenceCount[fov][band][InstrumentCharacteristic.
                   coldReferenceIdentifer][psd] <
                      sequenceCount[fov][band][InstrumentCharacteristic.
                         hotReferenceIdentifer][psd])
                {
                    Spectra* fakeSpectra =
                       new CalibratedSpectra((SceneElement)fov,
                          (Band::Wavelength)band,
                             (FieldOfRegard)InstrumentCharacteristic.
                                coldReferenceIdentifer,
                             (SweepDirection)psd);

                    fakeSpectra->setSDR_InvalidData(true);
                    fakeSpectra->setSize((processingPeriod.
                       getMeasuredSpectra(
                          processingPeriod.getWindowDepth(
                             (SceneElement)fov,
                             (Band::Wavelength)band,
                             (FieldOfRegard)InstrumentCharacteristic.
                                coldReferenceIdentifer,
                             (SweepDirection)psd),

                          (SceneElement)fov,
                          (Band::Wavelength)band,
                          (FieldOfRegard)InstrumentCharacteristic.
                             coldReferenceIdentifer,
                          (SweepDirection)psd))->getReal().size());
                    fakeSpectra->setFrameTime(
                       TelemetryProcessor::instance()->getLastFrameTime());

                    // set values so fake spectra are added to the window
                    fakeSpectra->setSDR_CalibrationType(
                       GeneralParameter.spectralCalMethod);
                    fakeSpectra->setSDR_ApodType(
                       CorrectionParameter.apodizationType);
                    fakeSpectra->setSDR_CalOrder(CorrectionParameter.
                       calibrationOrder);

                    delete fakeSpectra;
                    fakeSpectra = 0;

                    sequenceCount[fov][band][InstrumentCharacteristic.
                                             coldReferenceIdentifer][psd]++;
                    char testMsg[EVENT_LOG_LENGTH];
                    sprintf(testMsg,
                       "SYNC: %s %s %s %s Sequence count out of sync, adding "
                       "place holder spectra",
                          BAND[band].c_str(), FOV[fov].c_str(),
                          FOR[InstrumentCharacteristic.
                             coldReferenceIdentifer].c_str(), PSD[psd].c_str());
                    EventLogEngine::append(testMsg);
                }

                for(Int32 scene = 0; scene < TOTAL_SCENES; scene++)
                {

                    if (sequenceCount[fov][band][scene][psd] != 0)
                    {
                        while (sequenceCount[fov][band][scene][psd] <
                           sequenceCount[fov][band][InstrumentCharacteristic.
                              hotReferenceIdentifer][psd])
                        {
                            //insert placeholder spectra
                            Spectra* fakeSpectra =
                               new CalibratedSpectra(
                                  (SceneElement)fov, (Band::Wavelength)band,
                                  (FieldOfRegard)scene, (SweepDirection)psd);

                            fakeSpectra->setSDR_InvalidData(true);
                            fakeSpectra->setSize((processingPeriod.
                               getMeasuredSpectra(
                                  processingPeriod.getWindowDepth(
                                     (SceneElement)fov,
                                     (Band::Wavelength)band,
                                     (FieldOfRegard)scene,
                                     (SweepDirection)psd),

                                   (SceneElement)fov,
                                   (Band::Wavelength)band,
                                   (FieldOfRegard)scene,
                                   (SweepDirection)psd))->getReal().size());
                            fakeSpectra->setFrameTime(
                               TelemetryProcessor::instance()->
                                  getLastFrameTime());

                            // set values so fake spectra are added to the
                            // window
                            fakeSpectra->setSDR_CalibrationType(
                               GeneralParameter.spectralCalMethod);
                            fakeSpectra->setSDR_ApodType(
                               CorrectionParameter.apodizationType);
                            fakeSpectra->setSDR_CalOrder(CorrectionParameter.
                               calibrationOrder);

                            delete fakeSpectra;
                            fakeSpectra = 0;

                            sequenceCount[fov][band][scene][psd] =
                               sequenceCount[fov][band]
                                  [InstrumentCharacteristic.
                                     hotReferenceIdentifer][psd];

                            char testMsg[EVENT_LOG_LENGTH];
                            sprintf(testMsg,
                                    "SYNC: %s %s %s %s Sequence count out "
                                    "of sync, adding place holder spectra",
                                    BAND[band].c_str(),
                                    FOV[fov].c_str(),
                                    FOR[scene].c_str(),
                                    PSD[psd].c_str());
                            EventLogEngine::append(testMsg);
                        }
                    }
                }
            }
        }
    }
}


// ##METHOD HEADER-START## ****************************************************
//
// ARGUMENTS:
//  INPUTS:
//   SpectraManager::ReferenceSpectra spectraType
//   SweepDirection thePSD
//
//  OUTPUTS:
//   none
//
// DESCRIPTION:
//  Calls the Spectra Manager's batchReferenceFCE which does that fringe count
//  detection and correction. This method ensures that all FOVs and Bands of
//  a certain Reference scene/PSD have been received before performing FCE
//  detection and correction.
//
//*** ##METHOD HEADER-END## ***************************************************
void ScienceDataProcessor::launchBatchReferenceDetect(
   SpectraManager::ReferenceSpectra spectraType, SweepDirection thePSD)
{

    if (referenceSceneReceived[SpectraManager::coldReference][FORWARD] &&
        !(spectraType == SpectraManager::coldReference &&
          thePSD == FORWARD))
    {
        processingPeriod.batchReferenceFCE(
           SpectraManager::coldReference, FORWARD);

        //Set flag to no longer detect for DeepSpace FWD until next DeepSpace
        //FWD has been received
        referenceSceneReceived[SpectraManager::coldReference][FORWARD] = false;
    }

    if (referenceSceneReceived[SpectraManager::coldReference][REVERSE] &&
        !(spectraType == SpectraManager::coldReference &&
          thePSD == REVERSE))
    {
        processingPeriod.batchReferenceFCE(
           SpectraManager::coldReference, REVERSE);

        //Set flag to no longer detect for DeepSpace REV until next DeepSpace
        //REV has been received
        referenceSceneReceived[SpectraManager::coldReference][REVERSE] = false;
    }

    if (referenceSceneReceived[SpectraManager::hotReference][FORWARD] &&
        !(spectraType == SpectraManager::hotReference &&
          thePSD == FORWARD))
    {
        processingPeriod.batchReferenceFCE(
           SpectraManager::hotReference, FORWARD);

        //Set flag to no longer detect for ICT FWD until next ICT FWD has
        //been received
        referenceSceneReceived[SpectraManager::hotReference][FORWARD] = false;
    }

    if (referenceSceneReceived[SpectraManager::hotReference][REVERSE] &&
        !(spectraType == SpectraManager::hotReference &&
          thePSD == REVERSE))
    {
        processingPeriod.batchReferenceFCE(
           SpectraManager::hotReference, REVERSE);

        //Set flag to no longer detect for ICT REV until next ICT REV has
        //been received
        referenceSceneReceived[SpectraManager::hotReference][REVERSE] = false;
    }
}



// ##METHOD HEADER-START## ****************************************************
//
// ARGUMENTS:
//  INPUTS:
//   CNiReal32Vector&   profileSubset
//
//  OUTPUTS:
//   none
//
// DESCRIPTION:
//  This function interpolates a subset such that individual offset values are
//  available for each 8 seconds scan period.  The result is stored for
//  later use.
//
//*** ##METHOD HEADER-END## ***************************************************
void ScienceDataProcessor::buildBaffleTemperatureOffset(
     const BOOST::vector<Float32> & source, Int32 orbitDuration)
{
   Int32 srcSize   = source.size();
   Int32 totalSize = srcSize * 3;

   BOOST::vector<Float32> xPosition;
   BOOST::vector<Float32> yPosition(totalSize);

   xPosition.resize(totalSize);

   //triple input data set to aid in interpolation of end points of circular
   //data set
   for(Int32 xPos = 0; xPos < totalSize; xPos ++)
   {
      yPosition[xPos] = source[xPos%srcSize];
      xPosition[xPos] = (orbitDuration * 3.0 / totalSize) * xPos;
   }

   BOOST::vector<Float32> output;
   BOOST::vector<Float64> derivative;
   output.resize(orbitDuration);

   Int32 StartAt = orbitDuration;
   totalSize   = orbitDuration;

   //select the middle orbit period for interpolation of edge values
   Int32 StopAt  = (totalSize * 2);

   // It is enough to take linear interpolation
   Int32 i_coarse = 0;
   while(xPosition[i_coarse] < StartAt) i_coarse++;

   for(Int32 i = StartAt; i < StopAt; i++)
   {
      if(i > xPosition[i_coarse]) i_coarse++;
      output[i-StartAt] = yPosition[i_coarse-1] + (yPosition[i_coarse] -
                          yPosition[i_coarse-1]) / (xPosition[i_coarse] -
                          xPosition[i_coarse-1]) * (i - xPosition[i_coarse-1]);
   }

   setBaffleTemperatureOffset(output);
}

// ##METHOD HEADER-START## ****************************************************
//
// ARGUMENTS:
//  INPUTS:
//   Wavelength   band     requested band
//
//  OUTPUTS:
//   none
//
// DESCRIPTION:
//  This function calculates the gain introduced by the FIR Filter
//
//*** ##METHOD HEADER-END## ***************************************************
void ScienceDataProcessor::buildFirFilterGainTable(
     Band::Wavelength requestedBand)
{
   Float64 decimationFactor  =
      InstrumentCharacteristic.decimationFactor[requestedBand];
   Int32   numIFMpoints      =
      InstrumentCharacteristic.dataPointsDecimatedInterferogram[requestedBand];
   Float64 serialNumberScale =
      (InstrumentCharacteristic.LinearityCorrectionParameter.
         linearityCorrectionControlParam[requestedBand] == LINEARITY_CONFIG) ?
            InstrumentCharacteristic.LinearityCorrectionParameter.
               firFilterScale[requestedBand] :
            theEngineeringCalibrationRecord.
               getEngCalRec_LinearityErrorParameters().getParameters()->
                  firFilterScale[requestedBand];

   Int32    startBit         =
      (TelemetryStateConverter::getFIRAccumulatorStartBit(requestedBand) !=
         TelemetryProcessor::NOT_FOUND ||
            theEngineeringCalibrationRecord.
               getEngCalRec_LinearityErrorParameters().
                  getFirStartBit(requestedBand) > 0) ?
            theEngineeringCalibrationRecord.
               getEngCalRec_LinearityErrorParameters().
                  getFirStartBit(requestedBand) :
            InstrumentCharacteristic.firAccumulatorStartBit[requestedBand];


   Float64 radBinSize = (M_PI * 2) / (decimationFactor * numIFMpoints);

   Float64 gain       = serialNumberScale / (1 << startBit);

   BOOST::vector<std::complex<Float64> > FIR_COEFFs = InstrumentCharacteristic.
      LinearityCorrectionParameter.FIRresponse[requestedBand];
   std::complex<Float64>  i(0., -1.);
   BOOST::vector<Float64> FIR_Response(numIFMpoints);

   Int32 numCoeffs = FIR_COEFFs.size();

   Int32 Nstart_n = ROUND(lowestWavenumber[requestedBand] /
                          deltaSigma[requestedBand]);

   Int32 Nstop_n = Nstart_n + numIFMpoints - 1;
   for(Int32 index = Nstart_n; index <= Nstop_n; index++)
   {
      std::complex<Float64> left(0., 0.);
      for(Int32 filter = 0; filter < numCoeffs; filter++)
      {
         std::complex<Float64> tmpVar = i * (filter * index * radBinSize);
         left += FIR_COEFFs[filter] * exp(tmpVar.real()) *
            std::complex<Float64>( cos(tmpVar.imag()), sin(tmpVar.imag()) );
      }

      FIR_Response[index - Nstart_n] = gain *
         sqrt(left.real()*left.real() + left.imag()*left.imag()) ;
   }

   //Unfold
   BOOST::vector<Float64> *theGain =
      &(FIR_GAIN[requestedBand].bindReal()); // get pointer address

   theGain->resize(FIR_Response.size());
   for(UInt32 idx=0; idx<FIR_Response.size(); idx++)
   {
      (*theGain)[idx] = FIR_Response[idx];  // assign values for every element
   }

   // insert Spectral spacing metadata
   getFirGain(requestedBand)->
      setFirstWavenumber(lowestWavenumber[requestedBand]);
   getFirGain(requestedBand)->
      setWavenumberSpacing(deltaSigma[requestedBand]);

   //inform SpectraManager of new scaling
   processingPeriod.setFIR_GAIN(requestedBand, &FIR_GAIN[requestedBand]);
}

// ##METHOD HEADER-START## ****************************************************
//
// ARGUMENTS:
//  INPUTS:
//   CrisSdrAlgDataType*    ScanIndex
//
//  OUTPUTS:
//   none
//
// DESCRIPTION:
//  This function checks if there is a missing sciCal pkt in the last scan of
//  moving window (scan 34). If it were, ICT temperature parameters would
//  be updated
//
//*** ##METHOD HEADER-END## ***************************************************
void ScienceDataProcessor::checkICT_Status(
     Int32 scanIdx, CrisSdrAlgDataType* algDataPtr)
{
   bool flag = true;
   for(UInt32 thisFOV = 0; thisFOV < MAX_FOV; thisFOV++)
   {
      for(UInt32 thisBand = 0; thisBand < Band::TOTAL_BANDS; thisBand++)
      {
         UInt32 thisFOR = InstrumentCharacteristic.hotReferenceIdentifer;
         for(UInt32 thisPSD = 0; thisPSD < TOTAL_DIRECTIONS; thisPSD++)
         {
            //make sure reference scenes are in sync with earth scene content
            if(scienceCount[thisFOV][thisBand][thisFOR][thisPSD] <
               sequenceCount[thisFOV][thisBand][thisFOR][thisPSD])
            {
               char logMessage[EVENT_LOG_LENGTH];
               sprintf(logMessage,
                  "SYNC:  %s %s %d %d Missing Science Calibration record "
                  "detected in the last scan of moving window.",
               PSD[thisPSD].c_str(), BAND[thisBand].c_str(),
                  thisFOR, thisFOV + 1);
               (void)EventLogEngine::append(logMessage);

               //insert placeholder measurement
               if(flag)
               {
                  theScienceCalibrationRecord.markMissingCal();
                  // To guarantee markMissing once within the scan
                  theScienceCalibrationRecord.setSciCalMissing(true);
                  theScienceCalibrationRecord.
                     updateICT_Status(algDataPtr, scanIdx);
                  flag = false;
               }

               // At the end of moving window, scienceCount has to line up
               // with sequenceCount
               scienceCount[thisFOV][thisBand][thisFOR][thisPSD] =
                  sequenceCount[thisFOV][thisBand][thisFOR][thisPSD];
            }
         }
      }
   }
}

// ##METHOD HEADER-START## ****************************************************
//
// ARGUMENTS:
//  INPUTS:
//   Spectra* hotReference   - hot reference spectra
//   Spectra* coldReference  - cold reference spectra
//   Spectra* newSpectra     - The exiting spectra that has been fully calibrated
//   Spectra* rawSpectra     - The raw spectra corresponding to the above
//                             newSpectra
//   CrisSdrCoefficientsStruct* cfgParmsPtr
//
//  OUTPUTS:
//   a bool value
//
// DESCRIPTION:
//  Handling possible eartch scence FCE detection and correction.
//
//*** ##METHOD HEADER-END## ***************************************************

// FCE update
bool ScienceDataProcessor::batchEarthSceneFCE(Spectra* newSpectra,
              Spectra* rawSpectra, CrisSdrCoefficients* cfgParmsPtr)
{
    bool var = false;

    Band::Wavelength theBand = newSpectra->getBand();;
    SceneElement theFOV = newSpectra->getFieldOfView();
    FieldOfRegard theFOR = newSpectra->getFieldOfRegard();
    SweepDirection thePSD = newSpectra->getSweepDirection();
    BOOST::vector<std::complex<Float64> > linearPhaseShift;

    UncalibratedSpectra hotReference(*(processingPeriod.
               getReferenceSpectra(SpectraManager::hotReference,
                                   theFOV, theBand, thePSD)));
    UncalibratedSpectra coldReference(*(processingPeriod.
               getReferenceSpectra(SpectraManager::coldReference,
                                   theFOV, theBand, thePSD)));

    hotReference.applyLinearityErrorCorrection();
    coldReference.applyLinearityErrorCorrection();

    Int32 fringeCount = newSpectra->detectESFringeCountError(
                &hotReference, &coldReference, rawSpectra,
                processingPeriod.getRadianceSpectra(
                     SpectraManager::hotRadiance, theFOV, theBand),
                processingPeriod.getRadianceSpectra(
                     SpectraManager::coldRadiance, theFOV, theBand));

    if( fringeCount == 0)
    {
        return var; // fringeCount == 0 means NO FCE detected, so return 0

    }
    else
    {
        // Turn FCE detection flag on
        rawSpectra->setSDR_FCEDetect(true);
        rawSpectra->setSdrFringeCount(fringeCount);

    }

    // Up to here, FCE is detected. So need to validate detected FCE by
    // checking earth scence imaginary radiance
    // Copy esSpectra, adjust phase and check if the
    // newSpectra is a fully calibrated spectra,
    // so copy rawSpectra to phaseAdjustSpectra
    CalibratedSpectra phaseAdjustSpectra = *rawSpectra;

    // Calculate linearPhaseShift based on fringe count fceCorrectionIndex
    phaseAdjustSpectra.correctFringeCountError(linearPhaseShift, 1.*fringeCount);
    // Shift earth scence phase for the validation,
    // so the reference spectra are not affected at this time
    phaseAdjustSpectra.adjustPhase(linearPhaseShift);

    // Calculate Vdc
    processingPeriod.calculateVdc(&phaseAdjustSpectra);

    // Apply nonlinear correction
    phaseAdjustSpectra.applyLinearityErrorCorrection();

    // Apply radiometric calibration
    processingPeriod.calibrate(&phaseAdjustSpectra, &hotReference,
                               &coldReference, theFOV, theBand);

    // Check the imaginary radiance of fully calibrated earth spectra
    Float32 minRad = std::numeric_limits<Float32>::max();
    Float32 maxRad = std::numeric_limits<Float32>::min();

    const boost::numeric::ublas::vector<double>
      imagBin = phaseAdjustSpectra.getImaginary();
    for(UInt32 point = cfgParmsPtr->lwImagRadCheckStart;
        point < cfgParmsPtr->lwImagRadCheckEnd; point++)
    {
        if (imagBin[point] > maxRad)
        {
            maxRad = imagBin[point];
        }
        if (((imagBin[point] > FLOAT32_FILL_TEST) ||
             (imagBin[point] < (FLOAT32_FILL_TEST - 1.0))) &&
             (imagBin[point] < minRad))
        {
            minRad = imagBin[point];
        }
    }

    if(maxRad > cfgParmsPtr->lwImagRadUpperThresh ||
       minRad < cfgParmsPtr->lwImagRadLowerThresh)
    {
        // Turn FCE coorection flag on
        rawSpectra->setSDR_FCECorrectFail(true);

        //Set invalid data flag
        char buffer[EVENT_LOG_LENGTH];
        sprintf(buffer,
        "FCE: Longwave Imaginary Radiance Value Outside of Range. "
        " (%f  %f  %f  %f)",
        maxRad, cfgParmsPtr->lwImagRadUpperThresh,
        minRad, cfgParmsPtr->lwImagRadLowerThresh);
        (void)EventLogEngine::append(buffer);

        return var; // The large imaginary radiance cannot be corrected
                    // by FCE correction, so return  0

    }
    else
    {
        var = true;
    }


    // Up to here, FCE dection is validated. So performing phase shift to
    // all ICT/DS spectra
    // adjust hot or cold reference average for each detector(FOV, Band, PSD)
    // to first FC that passed
    for(Int32 band = 0; band < Band::TOTAL_BANDS; band++)
    {
        for(Int32 fov = 0; fov < MAX_FOV; fov++)
        {
            if (processingPeriod.getMeasuredSpectraValidSize(
               (SceneElement)fov, (Band::Wavelength)band, theFOR, thePSD) > 0)
            {

               processingPeriod.getFirstValidMeasuredSpectra(
                 (SceneElement)fov,
                 (Band::Wavelength)band,
                 (FieldOfRegard)InternalCalTarget,
                 (SweepDirection)thePSD)->correctFringeCountError(
                                  linearPhaseShift, 1.*fringeCount);

                // FCE update -- 
                // Since adjustWindowPhase function will recalculate Vdc
                // for ICT spectra, it needs to make phase shift to DS first
                Int32 coldAdjust = processingPeriod.adjustWindowPhase(
                       DeepSpace,
                       (SceneElement)fov,
                       (Band::Wavelength)band,
                       thePSD,
                       &linearPhaseShift);

                Int32 hotAdjust = processingPeriod.adjustWindowPhase(
                       InternalCalTarget,
                       (SceneElement)fov,
                       (Band::Wavelength)band,
                       thePSD,
                       &linearPhaseShift);

                char textOut[EVENT_LOG_LENGTH];
                sprintf(textOut,"FCE detected in ES%d %s%d.  %d DS "
                        " and %d ICT items corrected",
                        theFOR,
                        BAND[band].c_str(),
                        fov + 1,
                        coldAdjust,
                        hotAdjust);

                EventLogEngine::append(textOut);
            }
        }
    }

    // Recalculate Vdc for newSpectra. Don't need to recalculate Vdc for ICT
    // spectra because ICT and DS spectra have the same phase shifted
    processingPeriod.calculateVdc(rawSpectra);

    // Apply nonlinear correction if needed
    if(InstrumentCharacteristic.performLinearityCorrectionControl[theBand])
    {
        hotReference = *(processingPeriod.getReferenceSpectra(
                 SpectraManager::hotReference, theFOV, theBand, thePSD));
        coldReference = *(processingPeriod.getReferenceSpectra(
                 SpectraManager::coldReference, theFOV, theBand, thePSD));

        hotReference.applyLinearityErrorCorrection();
        coldReference.applyLinearityErrorCorrection();

        rawSpectra->applyLinearityErrorCorrection();
    }

    // FCE update --- original code, replaced by STAR updated segment
    // Apply radiometric calibration
    processingPeriod.calibrate(rawSpectra, &hotReference,
                               &coldReference, theFOV, theBand);


    // lhw -- I think, the setSDR_FCEExcess is obsolet, so ignore it right now
    // Need to discuss if still like to set this flag. NPP code calculates
    // the fringeCount by curve fitting
    // But here, we use trial method, indep of excess fringeCount.
    // If still check the fractional maxFCE_threshold, e.g.,
    // if (abs(fringecount) > 10) fringeCountErrorExcessive = true;
    // do we like to set this flag or just skip it?
    //  rawSpectra->setSDR_FCEExcess(fringeCountErrorExcessive);

    rawSpectra->setSDR_FCEDetect(true);

    // Copy to newSpectra
    *newSpectra = *rawSpectra;

    return var;
}

