/* BEGIN JPSS HEADER */
/*****************************************************************************
*
* Peraton, Inc.  All rights reserved.
*
* GOVERNMENT RIGHTS:
* Per contract 1332KP23CNEEJ0001, CAR 1352.227-70 applies giving full
* copyrights to data, including reports and other copyrighted materials
* to the United States Government.
*
* THIRD PARTY RIGHTS:
* Note: Third party entities may own copyrights in this work.
*
* EXPORT CONTROL:
* Unless otherwise specified beneath this header, this item does not contain
* Technology or Technical Data controlled under either the U.S. International
* Traffic in Arms Regulations or the U.S. Export Administration Regulations.
*
* CLASSIFICATION: B
* Refer to Software Standards and Practices Manual (SSPM) for coding and
* documentation standards.
*
*****************************************************************************/
/* END JPSS HEADER */

/*****************************************************************************
*
*  NAME: Spectra.cpp
*
*  DESCRIPTION:  See description preceeding declaration below
*
*****************************************************************************/

/***********************************************************************
*  HISTORY:
*
*  DATE       PR#     AUTHOR        Build  DESCRIPTION
*  ---------  ------  ------------  -----  ------------------------------------
*  25APR2005          B. Henderson  1.3    Initial Version
*                     D. Elliott
*                     T. Hahn
*  01MAR2006          D. Elliott    1.4    Fringe Count Error
*                     T. Hahn              Follow On Drop
*  02AUG2006  011858  D. Elliott    1.4    Modified ITT header with
*                                          permission from Steven G.
*                                          Minnie from ITT Space Systems, LLC
*  19JUN2007          J. DeLotelle  1.5    Updated copyright and government
*                                          rights info in prologue.
*  29JUN2007          D. Elliott    1.5    Modification based upon TM:
*                     B. Henderson         NP-EMD-2007.510.0027.Rev.A
*  14DEC2007  015906  K. Boswell    1.5x1  ECR A-103, EDRPR 1.8 CP3 Updates
*                                          moved roll, pitch, and yaw to temp
*                                          struct and swapped ev and fov
*                                          dimensions in the geo product;
*                                          added reference to
*                                          ProSdrCrisStruct.h
*  07JAN2008  015748  B. Henderson  1.5x1  ISTN_CRIS_SDR_NGST_1.1
*                                          drop implementation
*  21FEB2008  015748  K. Boswell    1.5x1  ISTN_CRIS_SDR_NGST_1.1
*                                          drop implementation -
*                                          format updates for DFCB compliance.
*                                          ref. NP-EMD-2007.510.0027 RevB
*  19MAR2008  017044  K. Boswell    1.5x1  Added comments.
*  24APR2008  017500  K. Boswell    1.5x1  Moved roll and pitch to
*                                          CrIS-SDR-Exit-Ang-IP
*  12JUN2008  017910  K. Boswell    1.5x1  Made theScienceDataProcessor_
*                                          into a smart pointer.
*  25OCT2008          L. Wang    ALG01342  Implem with ITT update 2.14
*  20JAN2009  019301  K. Boswell    1.5x1  ISTN_CRIS_SDR_UNIX_NGST_2.1
*                                          drop implementation
*                                          per TM NP-EMD.2008.510.0055
*  26MAY2009          L. Wang     ALG1443  Update for nonlinearity correction
*  03FEB2010  021718  K. Boswell           SensChar ISTN_CRIS_SDR_UNIX_NGST_2.2
*                                          drop implementation
*                                          per TM NP-EMD.2009.510.0046
*  19MAY2010  023135  K. Boswell           SensChar ISTN_CRIS_SDR_UNIX_NGST_
*                                          2.2.1 drop implementation per TM
*                                          NP-EMD.2010.510.0006 and
*                                          ISTN_CRIS_SDR_UNIX_NGST_2.2.2 per TM
*                                          NP-EMD.2010.510.0006 Rev A
*                                          and ECR A-275
*  31JAN2012  029362  T. Gardner    Mx6    forTimeStampBias now correctly
*                                          read as Int16.
*  07MAR2012  029651  T. Gardner    Mx7a   Corrected the setting of the
*                                          lunar intrusion flag.
*  12MAR2012  028309  S. Manoj      Mx7    ADR 4389 CrIS SDR:
*                                          ESZPDMagnitude data type
*                                          field inconsistencies
*  12DEC2012  033132  T. Gardner    Mx7    Set RDR to invalid in case of
*                                          missing packet
*  06JUN2013  034187  T. Gardner    Mx8.0  LaserWavelength is updated only when
*             034188                       CMO is updated and saved
*  15MAY2013  033696  J. Schwartz   Bk1.5  Util Time Updates
*  02JUL2013  033702  S. Wilderdyke Bk2.0  Changes to work with
*                     S. Manoj             INF Util Time Updates
*  04NOV2013  035224  S. Joo        Bk2.0  Clone of PCR035543 fixing the
*                                          issue of zero values in the first
*                                          14 scans of Monitored Laser
*                                          Wavelength
*  14MAR2014  037312  T. Fisher     Bk2.0  Clone PCR036444: Correct one scan
*                                          offset between DS/ICT invalid
*                                          spectra and window size
*  19MAY2014  038279  R. Neely      Blk2.0 Clone of PCR035945. Changed type of
*                                          forTimeStampBias from UInt32 to
*                                          Int32.
* 26JUN2014          Y. Chen/Y.Han  Mx8.5  Updates for full spectral resolution
* 24Aug2014          Y. Han         Mx8.5  Pass foldIndex to constructor and
*                                          remove the calculation to consolidate
*                                          multiple calculations into one
*  14OCT2014  042555    SEClark Blk2.0.0.4 CrIS SSPM Compliance.
*  03JAN2015   41268  J. Eberle     Blk2.0 Header Changes
*  11MAY2015   48586  T. Gardner    Blk2.0 Integrate TR/FR updates into
*                                          block 2.0
*  01JUL2015          L. Wang   V2.0_beta  Updates for FCE handling
*  01APR2016          Y. Chen      Blk2.0  Intergrate TR/FR updates into
*                                          block 2.0
*  24JUL2017         H. Xu/Y. Chen Blk2.0  Added PCT coefficients to spectra and
*                                          added quality flag to SDR data quality
*                                          according to the spike correction status
*  26JUL2017          Y. Chen      Blk2.0  Using DEFAULT_LASER_WAVELENGTH
*  06SEP2017          G. Pachl     Blk2.1  CCRs-17-3543, 3541.
*  26APR2018  65628   G. Pachl     Blk2.1  Merged into Baseline CCR 3542.
*******************************************************************************/

/*******************************************************************************
* This document contains information proprietary and confidential to ITT Space
* Systems, LLC ("ITT"), or a third party for whom ITT may have a legal
* obligation to protect such information from unauthorized disclosure, use or
* duplication. Any disclosure, use, duplication or diversion contrary
* to U.S. Law of this document or of any of the information contained herein
* for other than the
* specific purpose for which it was disclosed is expressly prohibited,
* except as ITT has otherwise
* agreed in writing or as the U.S. Government has authorized. All copies of
* this document are the
* sole property of ITT and will be returned promptly upon request
******************************************************************************/

#include <ProCmnSecurity.h>
#include <Spectra.h>
#include <boost/numeric/ublas/vector.hpp>
#include <boost/numeric/ublas/vector_proxy.hpp>

#include <FloatCompare.h>
#include <ScienceDataProcessor.h>
#include <SystemTimeUtilities.h>
#include <ProCmnFillType.h>
#include <ProCmnPhysConst.h>
#include <ProCmnTimeUtil.h>
#include <SDR_AlgorithmnParameter.h>
#include <FixSpikeData.h>

// static const class attributes
/*static*/ const Int32 Spectra::DAYS_TO_MICROSECS_SHIFT_CONV = 48;
/*static*/ const Int32 Spectra::MILISECS_TO_MICROSECS_SHIFT_CONV = 16;
/*static*/ const Float32 Spectra::MICRO_TO_MILI_UNIT_CONV = 0.001;

//externally governed variables that regulate construction
Float64 Spectra::LASER_WAVELENGTH[] = {ScienceDataProcessor::DEFAULT_LASER_WAVELENGTH, 
                                       ScienceDataProcessor::DEFAULT_LASER_WAVELENGTH, 
                                       ScienceDataProcessor::DEFAULT_LASER_WAVELENGTH};
bool Spectra::usingDefaultLW = true;
Float64 Spectra::metrologyWavelengthShift[] = {1.0, 1.0, 1.0};
Float64 Spectra::RESAMPLING_WAVELENGTH[] =
   {Spectra::LASER_WAVELENGTH[Band::LONGWAVE] / 2.0,
    Spectra::LASER_WAVELENGTH[Band::MIDWAVE] / 2.0,
    Spectra::LASER_WAVELENGTH[Band::SHORTWAVE] / 2.0};
bool Spectra::INVALID_NEON_CAL = false;
UInt32 Spectra::IMPULSE_NOISE_COUNT_THRESHOLD = 0;
UInt32 Spectra::NUMBER_OPD_OVERSCAN_SAMPLES = 1;
Int32 Spectra::timestampBiasDifference = -164;
Int32 newBinCount = 0;

Float64 Spectra::DESIRED_BAND_CENTER[Band::TOTAL_BANDS] =
{
    (1095 + 650)/2.0,
    (1210 + 1750)/2.0,
    (2155 + 2550)/2.0
};

Spectra::FCEParam Spectra::fceParameters[Band::TOTAL_BANDS] =
{
    {{0.45, 0.45}, 0, 4, 0.001, 0.1, 0.0001, 200, 790, 76, 1.0, 50, 650, 1075 },
    {{0.45, 0.45}, 0, 4, 0.001, 0.1, 0.0001, 200, 482, 48, 1.0, 50, 650, 1075 },
    {{0.45, 0.45}, 0, 4, 0.001, 0.1, 0.0001, 200, 140, 21, 1.0, 50, 650, 1075 }
};
bool Spectra::scalingEnabled = true;

// ##METHOD HEADER-START## ****************************************************
//
// ARGUMENTS:
//  INPUTS:
//   source: the spectra to copy
//
//  OUTPUTS:
//   new object.
//
// DESCRIPTION:
//  Copy Construction
//
//*** ##METHOD HEADER-END## ***************************************************/
Spectra::Spectra(const Spectra& source) :
    response(source.response),
    realWavenumberBin(source.realWavenumberBin),
    imaginaryWavenumberBin(source.imaginaryWavenumberBin),
    NEdN_Estimate(source.NEdN_Estimate),
    firstWavenumber(source.firstWavenumber),
    wavenumberBinSize(source.wavenumberBinSize),
    foldIndex(source.foldIndex),
    spectralBand(source.spectralBand),
    fieldOfView(source.fieldOfView),
    fieldOfRegard(source.fieldOfRegard),
    sweepDirection(source.sweepDirection),
    userLocks(1),
    channel(source.channel),
    interferogram_DS_reals(source.interferogram_DS_reals),
    interferogram_DS_imags(source.interferogram_DS_imags)
{
   Inf_SecureString::memcpy_s(&RDR_Status,
                              sizeof(RDR_Status),
                              &source.RDR_Status,
                              sizeof(RDR_Status));
   Inf_SecureString::memcpy_s(&SDR_Status,
                              sizeof(SDR_Status),
                              &source.SDR_Status,
                              sizeof(SDR_Status));
}


// ##METHOD HEADER-START## ****************************************************
//
// ARGUMENTS:
//  INPUTS:
//    SceneElement     fovInit      - inital Field Of View
//    Band::Wavelength bandInit     - inital Band
//    FieldOfRegard    forInit      - inital Field Of Regard
//    SweepDirection   psdInit      - inital Porch Swing Direction
//
//  OUTPUTS:
//   new object
//
// DESCRIPTION:
//  Initialization Construction
//
//*** ##METHOD HEADER-END## ***************************************************/
Spectra::Spectra(SceneElement fovInit,
                 Band::Wavelength bandInit,
                 FieldOfRegard forInit,
                 SweepDirection psdInit) :
    response(0.0),
    realWavenumberBin((UInt32)0),
    imaginaryWavenumberBin((UInt32)0),
    NEdN_Estimate((UInt32)0),
    firstWavenumber(0.0),
    wavenumberBinSize(1.0),
    foldIndex((UInt32)0),
    spectralBand(bandInit),
    fieldOfView(fovInit),
    fieldOfRegard(forInit),
    sweepDirection(psdInit),
    userLocks(1),
    channel(0),
    interferogram_DS_reals((UInt32)0),
    interferogram_DS_imags((UInt32)0)
{
    //clear status
    ZeroMemory(&RDR_Status, sizeof(RDR_Status));
    ZeroMemory(&SDR_Status, sizeof(SDR_Status));

    //internal notation for no LinErr correction
    SDR_Status.LinErrCorrectionFactor = 1.0;
}

// ##METHOD HEADER-START## ****************************************************
//
// ARGUMENTS:
//  INPUTS:
//    const Interferogram& interferogram  -  creation interferogram
//    UInt32                 theFoldIdx   -  fold index
//    Float64                deltaSigma   -  wavenumber grid size
//    Float64                theFirstW    - first wavenumber
//    bool                 removeAlias    -  remove diagnostic alias (TRUE)
//    CrisSdrCoefficients*   cfgParmsPtr  -  SDR coefficient
//  OUTPUTS:
//   new object
//
// DESCRIPTION:
//  Initialization Construction
//
//*** ##METHOD HEADER-END## ***************************************************/
Spectra::Spectra(const Interferogram& interferogram,
                 UInt32 theFoldIdx,
                 Float64 theFirstW,
                 Float64 deltaSigma,
                 CalibrationOrder theOrder,
                 CrisSdrCoefficients* cfgParmsPtr,
                 bool removeAlias) :
    response(0.0),
    realWavenumberBin((UInt32)0),
    imaginaryWavenumberBin((UInt32)0),
    NEdN_Estimate((UInt32)0),
    firstWavenumber(theFirstW),
    wavenumberBinSize(deltaSigma),
    foldIndex(theFoldIdx),
    spectralBand(interferogram.getBand()),
    fieldOfView(interferogram.getFieldOfView()),
    fieldOfRegard(interferogram.getFieldOfRegard()),
    sweepDirection(interferogram.getPorchSwingDirection()),
    userLocks(1),
    channel(0),
    interferogram_DS_reals((UInt32)0),
    interferogram_DS_imags((UInt32)0)
{
    Inf_SecureString::memcpy_s(&RDR_Status,
                              sizeof(RawDataRecordStatusRegister),
                              &interferogram.getRDR_Status(),
                              sizeof(RawDataRecordStatusRegister));

    //clear self status
    ZeroMemory(&SDR_Status, sizeof(ScienceDataRecordStatus));

    //immediately invalidate the SDR if the RDR is suspect
    SDR_Status.invalidData = RDR_Status.invalidData  ||
                             RDR_Status.bitTrimFailure ||
                             RDR_Status.fringeCountError ||
                             RDR_Status.missingData;
    SDR_Status.excessiveNEdN = RDR_Status.impulseNoiseCount >
       IMPULSE_NOISE_COUNT_THRESHOLD;
    SDR_Status.laserWavelength = LASER_WAVELENGTH[spectralBand];

    SDR_Status.monitoredLaserWavelength = LASER_WAVELENGTH[spectralBand];

    SDR_Status.spectralResamplingLaserWavelength =
       RESAMPLING_WAVELENGTH[spectralBand];
    SDR_Status.invalidNeonCalibration = INVALID_NEON_CAL;

    SDR_Status.LinErrCorrectionFactor = 1.0;

    //assign the calibration oder
    SDR_Status.calibrationOrder = (UInt8) theOrder;

    // Save deep space interferogram.
    if(interferogram.getFieldOfRegard() == DeepSpace &&
       !RDR_Status.missingData)
    {
        interferogram_DS_reals =
            interferogram.getSamples(Interferogram::RealPart);
        interferogram_DS_imags =
            interferogram.getSamples(Interferogram::ImaginaryPart);
    }

    ////////////////////////////////////////////
    //  Transform interferogram into spectra  //
    ////////////////////////////////////////////

    //Values and calculations used during unfolding filtering
    //Reference document #8195964 if more information is needed.
    static UInt32 sampleCount[Band::TOTAL_BANDS] =
        {
            TelemetryProcessor::instance()->theVideoData.
               getExtractionRecord(VideoData::LW).complexSamples - 
                                   2* NUMBER_OPD_OVERSCAN_SAMPLES,
            TelemetryProcessor::instance()->theVideoData.
               getExtractionRecord(VideoData::MW).complexSamples - 
                                   2* NUMBER_OPD_OVERSCAN_SAMPLES,
            TelemetryProcessor::instance()->theVideoData.
               getExtractionRecord(VideoData::SW).complexSamples - 
                                   2* NUMBER_OPD_OVERSCAN_SAMPLES
        };

    static UInt32 decimationFactor[Band::TOTAL_BANDS]  =
        {
            TelemetryProcessor::instance()->theVideoData.
               getExtractionRecord(VideoData::LW).decimationRate,
            TelemetryProcessor::instance()->theVideoData.
               getExtractionRecord(VideoData::MW).decimationRate,
            TelemetryProcessor::instance()->theVideoData.
               getExtractionRecord(VideoData::SW).decimationRate
        };


    //Do FFT math and populate Wave Number bins with unfolded data
    if (!RDR_Status.missingData)
    {
       if(interferogram.getSamples(Interferogram::ImaginaryPart).size())
       {
          // NORMAL MODE INTERFEROGRAM (FFT SHIFT CONSTRUCTION)

          //get interferogram data while clipping and folding
          UInt32 shiftSize = (sampleCount[spectralBand])/2;

          realWavenumberBin = project(interferogram.
             getSamples(Interferogram::RealPart),
                BOOST::range(NUMBER_OPD_OVERSCAN_SAMPLES,
                   interferogram.getSamples(Interferogram::RealPart).size() -
                      NUMBER_OPD_OVERSCAN_SAMPLES));
          imaginaryWavenumberBin = project(interferogram.
             getSamples(Interferogram::ImaginaryPart),
                BOOST::range(NUMBER_OPD_OVERSCAN_SAMPLES,
                   interferogram.getSamples(Interferogram::RealPart).size() -
                      NUMBER_OPD_OVERSCAN_SAMPLES));
          // Check for radiation spikes and make correction
          RDR_Status.spikeDetectionCorrection = NO_SPIKE;  
          if (cfgParmsPtr != NULL)
          {
             // check whether need igm spike correction
             if (cfgParmsPtr->applySpikeCorrection[spectralBand] == 1)
             {
                Float64 diffrat;
                Int32 ipeak = ChkAsym(&diffrat, realWavenumberBin, 
                      imaginaryWavenumberBin,
                      cfgParmsPtr->spikeBinPoints[spectralBand], 
                      cfgParmsPtr->spikeBinCenterEscape[spectralBand], 
                      cfgParmsPtr->spikeThreshold[spectralBand]);
                if (ipeak > -1)
                {
                   // write the spike info into log file
                   char logMessage[EVENT_LOG_LENGTH];
                   sprintf(logMessage, "FixSpike: band= %d, drat= %f, ipeak= %d, "
                         "FOR= %d, FOV= %d, milsec= %d", 
                         spectralBand, diffrat, ipeak, 
                         fieldOfRegard, fieldOfView + 1, 
                         RDR_Status.milliseconds);
                   EventLogEngine::append(logMessage);
                   RDR_Status.spikeDetectionCorrection = SPIKE_DETECTED;  
                   // define spike correction object
                   FixSpikeData pfcnData;
                   pfcnData.decimationFactor = 
                      cfgParmsPtr->decimationFactor[spectralBand];
                   pfcnData.decimationOffset = 
                      cfgParmsPtr->decimationOffset[spectralBand]; 
                   pfcnData.spikeModelPoints = 
                      cfgParmsPtr->spikeModelPoints[spectralBand]; 
                   // FIR filter
                   pfcnData.FIRresponse.resize(NUM_FIR_FILTER_RESP_PTS);
                   std::complex<Float64>  i(0., 1.);
                   for (Int32 pt = 0; pt < NUM_FIR_FILTER_RESP_PTS; ++pt)
                   {
                      pfcnData.FIRresponse(pt) = double(cfgParmsPtr->
                            firFilterResponse_Real[spectralBand][pt])
                         - i*double(cfgParmsPtr->
                               firFilterResponse_Imag[spectralBand][pt]);
                   }
                   // 5 pole butterworth filter
                   pfcnData.bresponse.resize(NUM_SPIKE_BRESPONSE_PTS);
                   for (Int32 pt = 0; pt < NUM_SPIKE_BRESPONSE_PTS; ++pt)
                      pfcnData.bresponse(pt) = cfgParmsPtr->spikeBresponse[pt];
                   pfcnData.numOfUsedBresponse = cfgParmsPtr->numOfUsedBresponse;

                   // copy the orginal spectrum
                   BOOST::vector<Float64> tstreal(realWavenumberBin.size());
                   BOOST::vector<Float64> tstimag(realWavenumberBin.size());
                   for (int ip = 0; ip < realWavenumberBin.size(); ip++)
                   {
                      tstreal(ip) = realWavenumberBin(ip);
                      tstimag(ip) = imaginaryWavenumberBin(ip);
                   }

                   //Make correction
                   Float64 fitpos, fitamp;
                   CorrectInts(&fitpos, &fitamp, tstreal, tstimag, ipeak, pfcnData);
                   sprintf(logMessage, "FixSpike: fitpos= %f, fitamp= %f", 
                         fitpos, fitamp);
                   EventLogEngine::append(logMessage);

                   //Check if correction worked
                   ipeak = ChkAsym(&diffrat, tstreal, tstimag,
                         cfgParmsPtr->spikeBinPoints[spectralBand], 
                         cfgParmsPtr->spikeBinCenterEscape[spectralBand], 
                         cfgParmsPtr->spikeThreshold[spectralBand]);
                   if (ipeak > -1)
                   {
                      // set current ICT and DS as invalid
                      if (fieldOfRegard == InternalCalTarget || 
                            fieldOfRegard == DeepSpace)
                         SDR_Status.invalidData = true;
                      sprintf(logMessage, "FixSpike: Correction Failed");
                      EventLogEngine::append(logMessage);
                   }
                   else //Use corrected interferogram
                   {
                      for (int ip = 0; ip < realWavenumberBin.size(); ip++)
                      {
                         realWavenumberBin(ip) = tstreal(ip);
                         imaginaryWavenumberBin(ip) = tstimag(ip);
                      }
                      RDR_Status.spikeDetectionCorrection = SPIKE_CORRECTED;  
                   }
                }
             }
          }
          std::rotate(
                realWavenumberBin.begin(),
                realWavenumberBin.begin() + (Int32)shiftSize,
                realWavenumberBin.end());
          std::rotate(
                imaginaryWavenumberBin.begin(),
                imaginaryWavenumberBin.begin() + (Int32)shiftSize,
                imaginaryWavenumberBin.end());

          Int32 status = FFT(&realWavenumberBin,&imaginaryWavenumberBin);

          if(status != 0)
          {
             EventLogEngine::append("Creation of Spectra Failed.");
             SDR_Status.invalidData = true;
          }

          std::rotate(
                realWavenumberBin.begin(),
                realWavenumberBin.begin() + static_cast<Int32>(foldIndex),
                realWavenumberBin.end());
          std::rotate(
                imaginaryWavenumberBin.begin(),
                imaginaryWavenumberBin.begin() + static_cast<Int32>(foldIndex),
                imaginaryWavenumberBin.end());

          //scale spectra
          if(scalingEnabled)
          {
             Float64 factor = (Float64)((Float64)(LASER_WAVELENGTH[spectralBand]
                / 2.0) * decimationFactor[spectralBand] * 0.0000001);
             realWavenumberBin *= factor;
             imaginaryWavenumberBin *= factor;
          }
       }
       else
       {
          // DIAGNOSTIC MODE INTERFEROGRAM
          UInt32 sampleCnt =
                interferogram.getSamples(Interferogram::RealPart).size();

          Float64 maxPathDifferential = (LASER_WAVELENGTH[spectralBand] / 2.0) *
             sampleCnt / 2.0 * 0.0000001;
          Float64 deltaSigmaValue = 1/(2 * maxPathDifferential);

          wavenumberBinSize = deltaSigmaValue;
          firstWavenumber = 0;

          foldIndex = 0;

          UINT shiftSize = sampleCnt / 2;

          realWavenumberBin = project(interferogram.getSamples(
             Interferogram::RealPart),BOOST::range(1,shiftSize*2-1));

          std::rotate(
                realWavenumberBin.begin(),
                realWavenumberBin.begin() + (Int32)shiftSize,
                realWavenumberBin.end());

          //Set imaginary bin size
          imaginaryWavenumberBin.resize(realWavenumberBin.size());

          Int32 status = FFTR2C(&realWavenumberBin, &imaginaryWavenumberBin);

          if(status != 0)
          {
             EventLogEngine::append("Creation of Diagnostic Spectra Failed.");
             SDR_Status.invalidData = true;
          }

          //scale spectra
          if(scalingEnabled)
          {
             Float64 factor = (Float64)((Float64)(
                LASER_WAVELENGTH[spectralBand]/2.0) * 0.0000001);

             realWavenumberBin *= factor;
             imaginaryWavenumberBin *= factor;
          }

          if (removeAlias)
          {
             //no unfolding required and only elements 1->n/2 are desired
             realWavenumberBin =
                   BOOST::vector_slice<BOOST::vector<Float64> >
             (realWavenumberBin,
                   BOOST::slice(0, 1, realWavenumberBin.size()/2));
             imaginaryWavenumberBin =
                   BOOST::vector_slice<BOOST::vector<Float64> >
             (imaginaryWavenumberBin,
                   BOOST::slice(0, 1, imaginaryWavenumberBin.size()/2));
          }
       }

       //size NEdN
       NEdN_Estimate.resize(realWavenumberBin.size());
    }
}


// ##METHOD HEADER-START## *****************************************************
//
// ARGUMENTS:
//  INPUTS:
//   none
//
//  OUTPUTS:
//   reference to the root of the complex squares
//
// DESCRIPTION:
//  "Just In Time" construction of magnitude
//
//*** ##METHOD HEADER-END## ***************************************************/
const BOOST::vector<Float64>& Spectra::getMagnitude() const
{
    static BOOST::vector<Float64> magnitudeWavenumberBin;
    magnitudeWavenumberBin.resize(realWavenumberBin.size());
    for(UInt32 index = 0; index < realWavenumberBin.size(); index++)
    {
        magnitudeWavenumberBin[index] = sqrt((realWavenumberBin[index] *
           realWavenumberBin[index]) + (imaginaryWavenumberBin[index] *
              imaginaryWavenumberBin[index]));
    }

    return magnitudeWavenumberBin;
}

const BOOST::vector<std::complex<Float64> > & Spectra::getComplex()
{
   static BOOST::vector<std::complex<Float64> > complexWavenumberBin;

   complexWavenumberBin.resize(realWavenumberBin.size());

   //build magnitude
   for(UInt32 index = 0; index < realWavenumberBin.size(); index++)
   {
      complexWavenumberBin[index] = (realWavenumberBin[index],
         imaginaryWavenumberBin[index]);
   }

   return complexWavenumberBin;
}

// ##METHOD HEADER-START## *****************************************************
//
// ARGUMENTS:
//  INPUTS:
//   newSize: the number of the spectral bins
//
//  OUTPUTS:
//   None.
//
// DESCRIPTION:
//  This method will resize the bin count
//
//*** ##METHOD HEADER-END## ***************************************************/
void Spectra::setSize(UInt32 newSize)
{
    if (realWavenumberBin.size() != newSize)
    {
        realWavenumberBin.resize(newSize);
        imaginaryWavenumberBin.resize(newSize);
        NEdN_Estimate.resize(newSize);
    }
}



// ##METHOD HEADER-START## ****************************************************
//
// ARGUMENTS:
//  INPUTS:
//   none
//
//  OUTPUTS:
//   None.
//
// DESCRIPTION:
//  clears internal data vectors.
//
//*** ##METHOD HEADER-END## ***************************************************/
void Spectra::clear()
{
    realWavenumberBin.resize(0);
    imaginaryWavenumberBin.resize(0);
    NEdN_Estimate.resize(0);
    interferogram_DS_reals.clear();
    interferogram_DS_imags.clear();
}



// ##METHOD HEADER-START## ****************************************************
//
// ARGUMENTS:
//  INPUTS:
//   Spectra& source         - the source data to operate on
//   bool     complex        - indicates to operate on complex number
//
//  OUTPUTS:
//   None.
//
// DESCRIPTION:
//  when complex
//     increments by (a + bi) * (a - bi) = (a * a) + (b * b)
//  otherwise
//     increments by (a * a)
//
//*** ##METHOD HEADER-END## ***************************************************/
void Spectra::addSquare(const Spectra& source, bool complex)
{
    Int32 binSize = realWavenumberBin.size();
    if(binSize == 0)
    {
        binSize = source.realWavenumberBin.size();
        realWavenumberBin.resize(binSize);
        imaginaryWavenumberBin.resize(binSize);
        NEdN_Estimate.resize(binSize);
    }

    if(complex)
    {
        BOOST::vector<Float64> realSquared(source.realWavenumberBin.size());
        BOOST::vector<Float64> imagSquared(
           source.imaginaryWavenumberBin.size());

        // Square real part.
        std::transform(source.realWavenumberBin.begin(),
                       source.realWavenumberBin.end(),
                       source.realWavenumberBin.begin(),
                       realSquared.begin(),
                       std::multiplies<Float64>());

        // Square imaginary part.
        std::transform(source.imaginaryWavenumberBin.begin(),
                       source.imaginaryWavenumberBin.end(),
                       source.imaginaryWavenumberBin.begin(),
                       imagSquared.begin(),
                       std::multiplies<Float64>());

        // Add and assign.
        realWavenumberBin += realSquared + imagSquared;

        if (source.getSDR_CalibrationType() == CalibrationType_None)
        {
           for (Int32 index = 0; index < binSize; index++)
           {
              NEdN_Estimate[index] += sqrt(source.realWavenumberBin[index] *
                 source.realWavenumberBin[index] +
                    source.imaginaryWavenumberBin[index] *
                       source.imaginaryWavenumberBin[index]);
           }
        }
    }
    else
    {
        BOOST::vector<Float64> realSquared(source.realWavenumberBin.size());

        // Square real part.
        std::transform(source.realWavenumberBin.begin(),
                       source.realWavenumberBin.end(),
                       source.realWavenumberBin.begin(),
                       realSquared.begin(),
                       std::multiplies<Float64>());

        // Add and assign.
        realWavenumberBin += realSquared;

        BOOST::vector<Float64> imagSquared(
           source.imaginaryWavenumberBin.size());
        // Square imag part.
        std::transform(source.imaginaryWavenumberBin.begin(),
                       source.imaginaryWavenumberBin.end(),
                       source.imaginaryWavenumberBin.begin(),
                       imagSquared.begin(),
                       std::multiplies<Float64>());
        imaginaryWavenumberBin += imagSquared;

    }
}



// ##METHOD HEADER-START## ****************************************************
//
// ARGUMENTS:
//  INPUTS:
//   source spectra
//
//  OUTPUTS:
//   None.
//
// DESCRIPTION:
//  when complex
//     decrements by (a + bi) * (a - bi) = (a * a) + (b * b)
//  otherwise
//     decrements by (a * a)
//
//*** ##METHOD HEADER-END## ***************************************************/
void Spectra::removeSquare(const Spectra& source, bool complex)
{
    if(complex)
    {
        BOOST::vector<Float64> realSquared(source.realWavenumberBin.size());
        BOOST::vector<Float64> imagSquared(
           source.imaginaryWavenumberBin.size());

        // Square real part.
        std::transform(source.realWavenumberBin.begin(),
                       source.realWavenumberBin.end(),
                       source.realWavenumberBin.begin(),
                       realSquared.begin(),
                       std::multiplies<Float64>());

        // Square imaginary part.
        std::transform(source.imaginaryWavenumberBin.begin(),
                       source.imaginaryWavenumberBin.end(),
                       source.imaginaryWavenumberBin.begin(),
                       imagSquared.begin(),
                       std::multiplies<Float64>());

        // Subtract and assign.
        realWavenumberBin -= realSquared + imagSquared;

        if (source.getSDR_CalibrationType() == CalibrationType_None)
        {
          int binSize = NEdN_Estimate.size();
          for (int index = 0; index < binSize; index++)
          {
             NEdN_Estimate[index] -= sqrt((source.realWavenumberBin[index] *
                source.realWavenumberBin[index]) +
                   (source.imaginaryWavenumberBin[index] *
                      source.imaginaryWavenumberBin[index]));
          }
        }
    }
    else
    {
        BOOST::vector<Float64> realSquared(source.realWavenumberBin.size());

        // Square real part.
        std::transform(source.realWavenumberBin.begin(),
                       source.realWavenumberBin.end(),
                       source.realWavenumberBin.begin(),
                       realSquared.begin(),
                       std::multiplies<Float64>());

        // Minus and assign.
        realWavenumberBin -= realSquared;
    }
}



// ##METHOD HEADER-START## ****************************************************
//
// ARGUMENTS:
//  INPUTS:
//   factor: the number to divide this spectra by
//
//  OUTPUTS:
//   None.
//
// DESCRIPTION:
//  This method will divide each element of this spectra by a specified number.
//
//*** ##METHOD HEADER-END## ***************************************************/
void Spectra::operator/=(const Float64 factor)
{
    realWavenumberBin *= (1/factor);
    imaginaryWavenumberBin *= (1/factor);
    NEdN_Estimate *= (1/factor);
   SDR_Status.Vdc /= factor;
   SDR_Status.LinErrCorrectionFactor /= factor;
}



// ##METHOD HEADER-START## ****************************************************
//
// ARGUMENTS:
//  INPUTS:
//   factor: the number to multiply this spectra by
//
//  OUTPUTS:
//   None.
//
// DESCRIPTION:
//  This method will multiply each element of this spectra by a specified number.
//
//*** ##METHOD HEADER-END## ***************************************************/
void Spectra::operator*=(const Float64 factor)
{
    realWavenumberBin *= factor;
    imaginaryWavenumberBin *= factor;
    NEdN_Estimate *= factor;
   SDR_Status.Vdc *= factor;
   SDR_Status.LinErrCorrectionFactor *= factor;
}



// ##METHOD HEADER-START## ****************************************************
//
// ARGUMENTS:
//  INPUTS:
//   factor: the number to multiply this spectra by
//
//  OUTPUTS:
//   None.
//
// DESCRIPTION:
//  This method will multiply each element of this spectra bin by bin with the
//  factor
//
//*** ##METHOD HEADER-END## ***************************************************/
void Spectra::operator*=(const Spectra& factor)
{
    std::transform(realWavenumberBin.begin(),
                   realWavenumberBin.end(),
                   factor.realWavenumberBin.begin(),
                   realWavenumberBin.begin(),
                   std::multiplies<Float64>());
    std::transform(imaginaryWavenumberBin.begin(),
                   imaginaryWavenumberBin.end(),
                   factor.imaginaryWavenumberBin.begin(),
                   imaginaryWavenumberBin.begin(),
                   std::multiplies<Float64>());
    std::transform(NEdN_Estimate.begin(),
                   NEdN_Estimate.end(),
                   factor.NEdN_Estimate.begin(),
                   NEdN_Estimate.begin(),
                   std::multiplies<Float64>());
}



// ##METHOD HEADER-START## ****************************************************
//
// ARGUMENTS:
//  INPUTS:
//   dividend: the spectra to divide this spectra by
//
//  OUTPUTS:
//   None.
//
// DESCRIPTION:
//  This method will divide this spectra by a specified spectra.
//
//*** ##METHOD HEADER-END## ***************************************************/
void Spectra::operator/=(const Spectra& divisor)
{
    if( realWavenumberBin.size() == 0)
    {
        realWavenumberBin.resize(divisor.realWavenumberBin.size());
        imaginaryWavenumberBin.resize(divisor.imaginaryWavenumberBin.size());
        NEdN_Estimate.resize(divisor.NEdN_Estimate.size());

        firstWavenumber = divisor.firstWavenumber;
        wavenumberBinSize = divisor.wavenumberBinSize;
    }
    else
    {
        std::transform(realWavenumberBin.begin(),
                       realWavenumberBin.end(),
                       divisor.realWavenumberBin.begin(),
                       realWavenumberBin.begin(),
                       std::divides<Float64>());
        std::transform(imaginaryWavenumberBin.begin(),
                       imaginaryWavenumberBin.end(),
                       divisor.imaginaryWavenumberBin.begin(),
                       imaginaryWavenumberBin.begin(),
                       std::divides<Float64>());
        std::transform(NEdN_Estimate.begin(),
                       NEdN_Estimate.end(),
                       divisor.NEdN_Estimate.begin(),
                       NEdN_Estimate.begin(),
                       std::divides<Float64>());
       SDR_Status.Vdc /= divisor.SDR_Status.Vdc;
       SDR_Status.LinErrCorrectionFactor /= divisor.SDR_Status.
          LinErrCorrectionFactor;
    }
}



// ##METHOD HEADER-START## ****************************************************
//
// ARGUMENTS:
//  INPUTS:
//   source: the spectra to copy
//
//  OUTPUTS:
//   None.
//
// DESCRIPTION:
//  This method will make a copy of source spectra.
//
//*** ##METHOD HEADER-END## ***************************************************/
Spectra& Spectra::operator=(const Spectra& source)
{
    if(&source != this)
    {
        realWavenumberBin = source.realWavenumberBin;
        imaginaryWavenumberBin = source.imaginaryWavenumberBin;
        NEdN_Estimate = source.NEdN_Estimate;

        fieldOfView    = source.fieldOfView;
        spectralBand   = source.spectralBand;
        fieldOfRegard  = source.fieldOfRegard;
        sweepDirection = source.sweepDirection;
        response       = source.response;
        foldIndex      = source.foldIndex;

        channel        = source.channel;

        Inf_SecureString::memcpy_s(&RDR_Status,
                                 sizeof(RDR_Status),
                                 &source.RDR_Status,
                                 sizeof(RDR_Status));
        Inf_SecureString::memcpy_s(&SDR_Status,
                                 sizeof(SDR_Status),
                                 &source.SDR_Status,
                                 sizeof(SDR_Status));

        userLocks = this->userLocks;

        firstWavenumber = source.firstWavenumber;
        wavenumberBinSize = source.wavenumberBinSize;
    }

    return *this;
}



// ##METHOD HEADER-START## ****************************************************
//
// ARGUMENTS:
//  INPUTS:
//   rightOperand: the spectra to add to this spectra
//
//  OUTPUTS:
//   None.
//
// DESCRIPTION:
//  This method will add a spectra to the current spectra.
//
//*** ##METHOD HEADER-END## ***************************************************/
void Spectra::operator+=(const Spectra& rightOperand)
{
    if( realWavenumberBin.size() == 0)
    {
        realWavenumberBin = rightOperand.realWavenumberBin;
        imaginaryWavenumberBin = rightOperand.imaginaryWavenumberBin;
        NEdN_Estimate = rightOperand.NEdN_Estimate;

        fieldOfView    = rightOperand.fieldOfView;
        spectralBand   = rightOperand.spectralBand;
        fieldOfRegard  = rightOperand.fieldOfRegard;
        sweepDirection = rightOperand.sweepDirection;
        response       = rightOperand.response;
        foldIndex      = rightOperand.foldIndex;

        channel        = rightOperand.channel;

      Inf_SecureString::memcpy_s(&RDR_Status,
                                 sizeof(RDR_Status),
                                 &rightOperand.RDR_Status,
                                 sizeof(RDR_Status));
      Inf_SecureString::memcpy_s(&SDR_Status,
                                 sizeof(SDR_Status),
                                 &rightOperand.SDR_Status,
                                 sizeof(SDR_Status));

        userLocks = this->userLocks;

        firstWavenumber = rightOperand.firstWavenumber;
        wavenumberBinSize = rightOperand.wavenumberBinSize;
    }
    else
    {
        realWavenumberBin += rightOperand.realWavenumberBin;
        imaginaryWavenumberBin += rightOperand.imaginaryWavenumberBin;
        NEdN_Estimate += rightOperand.NEdN_Estimate;
        SDR_Status.Vdc += rightOperand.SDR_Status.Vdc;
        SDR_Status.LinErrCorrectionFactor += rightOperand.SDR_Status.
           LinErrCorrectionFactor;
    }
}



// ##METHOD HEADER-START## ****************************************************
//
// ARGUMENTS:
//  INPUTS:
//   rightOperand: the spectra to subtract from this spectra
//
//  OUTPUTS:
//   None.
//
// DESCRIPTION:
//  This method will subtract a spectra from the current spectra.
//
//*** ##METHOD HEADER-END## ***************************************************/
void Spectra::operator-=(const Spectra& rightOperand)
{

    if(rightOperand.realWavenumberBin.size() == 0 ||
       rightOperand.imaginaryWavenumberBin.size() == 0 ||
       rightOperand.NEdN_Estimate.size() == 0)
    {
        realWavenumberBin.resize(0);
        imaginaryWavenumberBin.resize(0);
        NEdN_Estimate.resize(0);
        return;
    }

    if( realWavenumberBin.size() == 0)
    {
        firstWavenumber = rightOperand.firstWavenumber;
        wavenumberBinSize = rightOperand.wavenumberBinSize;
        realWavenumberBin.resize(wavenumberBinSize);
        imaginaryWavenumberBin.resize(wavenumberBinSize);
        NEdN_Estimate.resize(wavenumberBinSize);

        realWavenumberBin -= rightOperand.realWavenumberBin;
        imaginaryWavenumberBin -= rightOperand.imaginaryWavenumberBin;
        NEdN_Estimate -= rightOperand.NEdN_Estimate;
    }
    else
    {
        realWavenumberBin -= rightOperand.realWavenumberBin;
        imaginaryWavenumberBin -= rightOperand.imaginaryWavenumberBin;
        NEdN_Estimate -= rightOperand.NEdN_Estimate;
        SDR_Status.Vdc -= rightOperand.SDR_Status.Vdc;
        SDR_Status.LinErrCorrectionFactor -= rightOperand.SDR_Status.
           LinErrCorrectionFactor;
    }
}

// ##METHOD HEADER-START## ****************************************************
//
// ARGUMENTS:
//  INPUTS:
//   none
//
//  OUTPUTS:
//   binary export of SDR
//
// DESCRIPTION:
//   Returns packed content of the class..
//
//*** ##METHOD HEADER-END## ***************************************************/
std::string Spectra::save() const
{
    static std::string binaryImage;

    switch (getVersionNumber())
    {
    case 0x400:
        saveVersion400(binaryImage);
        break;
    default:
        break;
    }

    return binaryImage;
}



// ##METHOD HEADER-START## ****************************************************
//
// ARGUMENTS:
//  INPUTS:
//   Float64 startWavenumber   -  resultant low clip frequency
//   Float64 stopWavenumber    -  resultant high clip frequency
//   Float64 binSize           -  resultant sampling rate
//
//  OUTPUTS:
//   boolean sucess
//
// DESCRIPTION:
//    This clips and resamples it's self to new resolution
//
//*** ##METHOD HEADER-END## ***************************************************/
bool Spectra::resample(
   Float64 startWavenumber, Float64 stopWavenumber, Float64 binSize)
{
    bool status = true;

    if(startWavenumber > stopWavenumber)
    {
        status = false;
    }
    else
    {
        UInt32 totalBins = realWavenumberBin.size();
      UInt32 finalSize = ((stopWavenumber - startWavenumber) / binSize) + 1;

        BOOST::vector<Float64> wavenumberSpace(totalBins);
        BOOST::vector<Float64> realDerivative;
        BOOST::vector<Float64> imagDerivative;
        for(UInt32 wavenumberBin = 0; wavenumberBin<totalBins; wavenumberBin++)
        {
            wavenumberSpace[wavenumberBin] = firstWavenumber +
               (wavenumberBinSize * wavenumberBin);
        }
        BOOST::vector<Float64> realValue(finalSize);
        BOOST::vector<Float64> imagValue(finalSize);

        Int32 newBinCnt = 0;

        Int32 result = 0;

        result = SplineInterpolation(&wavenumberSpace, &realWavenumberBin,
                                     startWavenumber, stopWavenumber, binSize,
                                     &realValue);
        result = result | SplineInterpolation(&wavenumberSpace,
                                              &imaginaryWavenumberBin,
                                              startWavenumber, stopWavenumber,
                                              binSize, &imagValue);

        if(result != 0)
        {
            EventLogEngine::append("Spectra Resampling Failed.");
            SDR_Status.invalidData = true;
            status = false;
        }

        //assign new content
        realWavenumberBin = BOOST::vector_slice<BOOST::vector<Float64> >
            (realValue, BOOST::slice(0, 1, newBinCnt - 1));
        imaginaryWavenumberBin = BOOST::vector_slice<BOOST::vector<Float64> >
            (imagValue, BOOST::slice(0, 1, newBinCnt - 1));

        firstWavenumber = startWavenumber;
        wavenumberBinSize = binSize;
    }

    return status;
}

// ##METHOD HEADER-START## ****************************************************
//
// ARGUMENTS:
//  INPUTS:
//   Float64 startWavenumber   -  resultant low clip frequency
//   Float64 stopWavenumber    -  resultant high clip frequency
//
//  OUTPUTS:
//   boolean sucess
//
// DESCRIPTION:
//    This clips and resamples it's self to new resolution
//
//*** ##METHOD HEADER-END## ***************************************************/
bool Spectra::clipGuardBands(Float64 startWavenumber, Float64 stopWavenumber)
{
    bool clipStatus = false;

    if(startWavenumber < stopWavenumber)
    {
        //calculate clip points
        UInt32 originalSize = realWavenumberBin.size();
        UInt32 lowBinIndex = (UInt32)floor((startWavenumber - firstWavenumber) /
           wavenumberBinSize);
        UInt32 highBinIndex = (UInt32)ceil(((wavenumberBinSize * originalSize +
           firstWavenumber) - stopWavenumber) / wavenumberBinSize) - 1;
        UInt32 resultantSize = originalSize - (highBinIndex + lowBinIndex);

        //validate clip points
        if((resultantSize + lowBinIndex) <= originalSize)
        {
            //adjust metadata
            firstWavenumber = firstWavenumber + (lowBinIndex*wavenumberBinSize);

            //adjust content
            realWavenumberBin = BOOST::vector_slice<BOOST::vector<Float64> >
                (realWavenumberBin,
                 BOOST::slice(lowBinIndex, 1, resultantSize));
            imaginaryWavenumberBin =
               BOOST::vector_slice<BOOST::vector<Float64> >
                (imaginaryWavenumberBin,
                 BOOST::slice(lowBinIndex, 1, resultantSize));
            NEdN_Estimate = BOOST::vector_slice<BOOST::vector<Float64> >
                (NEdN_Estimate,
                 BOOST::slice(lowBinIndex, 1, resultantSize));

            clipStatus = true;
        }
    }

    return clipStatus;
}

// ##METHOD HEADER-START## ****************************************************
//
// ARGUMENTS:
//  INPUTS:
//   none
//
//  OUTPUTS:
//   binary export of SDR
//
// DESCRIPTION:
//   Returns packed content of the class.
//   VERSION 400 SDR Algorithm
//
//*** ##METHOD HEADER-END## ***************************************************/
void Spectra::saveVersion400(std::string& binaryImage) const
{
    UInt32 index;

    //number of bytes of meta data before the Realwavenumber bin data
    static Int32 fixedOffset = 168;

    //three sets of spectra bins are saved: real, imaginary, and NEdN
    static Int32 numSpectra = 3;

    //Temporary variable to aide in swapping bytes
    Float32 floatTemp;
    UInt32 uintTemp;
    UInt32 *binary = new UInt32[2];
    UInt32 packetSize =
        ((realWavenumberBin.size() * sizeof(Float32)) * numSpectra) +
           fixedOffset;

    //build packet dereferencers
    binaryImage.resize((Int32)packetSize);
    UInt16* wordIndex = (UInt16*)binaryImage.c_str();

    ZeroMemory(wordIndex, packetSize);
    UInt16  wordValue = 0;

    //////////////////////////////
    //Add Algorithm Version Number
    //////////////////////////////
    *wordIndex = htons(getVersionNumber());
    wordIndex ++;

    ///////////////////////////////////
    //Add RDR metadata (Flight Version)
    ///////////////////////////////////
    wordValue = RDR_Status.flightSoftwareVersion << 5;
    //Append RDR metadata (Sensor ID)
    wordValue |= RDR_Status.sensorID;

    *wordIndex = htons(wordValue);
    wordIndex ++;

    ////////////////////////////////////
    //Add RDR metadata (Field Of Regard)
    ////////////////////////////////////
    wordValue = fieldOfRegard << SIZEOF(UInt8);

    //Append RDR metadata (Spectral Band)
    wordValue |= spectralBand;
    *wordIndex = htons(wordValue);
    wordIndex ++;
    ////////////////////////////////////
    //Add RDR metadata (Sweep Direction)
    ////////////////////////////////////
    wordValue = sweepDirection << 12;

    //Append RDR metadata (Field Of View)
    wordValue |= fieldOfView;
    *wordIndex = htons(wordValue);
    wordIndex ++;

    ///////////////
    //Add Time Code
    ///////////////
    static const UInt32 mns_per_hr   = 60;
    static const UInt32 secs_per_mn  = 60;
    static const UInt32 mils_per_sec = 1000;

    UInt32 millisecondsOfDay = RDR_Status.frameTime.wMilliseconds +
        mils_per_sec * (RDR_Status.frameTime.wSecond +
                        secs_per_mn * (RDR_Status.frameTime.wMinute +
                                     mns_per_hr * RDR_Status.frameTime.wHour));

    //////////
    //Add Days
    //////////
    *wordIndex = htons(
       CcsdsSecondaryHeader::numDaysSinceJanuary1_1958(&RDR_Status.frameTime));
    wordIndex++;

    //////////////////
    //Add Milleseconds
    //////////////////
    //convert UInt32 with host to network long
    millisecondsOfDay = htonl(millisecondsOfDay);

    //copy converted UInt32 to the output buffer
    Inf_SecureString::memcpy_s(wordIndex,
                              sizeof(Float32),
                              &millisecondsOfDay,
                              sizeof(Float32));
    wordIndex +=2;

    /////////////////////////////////////////
    //Add Line of Sight Elevation Angle
    /////////////////////////////////////////
    Inf_SecureString::memcpy_s(binary,
                              sizeof(Float64),
                              &SDR_Status.losElevationAngle,
                              sizeof(Float64));
   binary[0] = htonl(binary[0]);
   binary[1] = htonl(binary[1]);

   Inf_SecureString::memcpy_s(wordIndex,
                              sizeof(Int32),
                              &binary[1],
                              sizeof(Int32));
   wordIndex += 2;

   Inf_SecureString::memcpy_s(wordIndex,
                              sizeof(Int32),
                              &binary[0],
                              sizeof(Int32));
   wordIndex += 2;

    /////////////////////////////////////////
    //Add Line of Sight Roll Angle
    /////////////////////////////////////////
    Inf_SecureString::memcpy_s(binary,
                              sizeof(Float64),
                              &SDR_Status.losRollAngle,
                              sizeof(Float64));
   binary[0] = htonl(binary[0]);
   binary[1] = htonl(binary[1]);

   Inf_SecureString::memcpy_s(wordIndex,
                              sizeof(Int32),
                              &binary[1],
                              sizeof(Int32));
   wordIndex += 2;

   Inf_SecureString::memcpy_s(wordIndex,
                              sizeof(Int32),
                              &binary[0],
                              sizeof(Int32));
   wordIndex += 2;

    /////////////////////////////////////////
    //Add Line of Sight Pitch Angle
    /////////////////////////////////////////
    Inf_SecureString::memcpy_s(binary,
                              sizeof(Float64),
                              &SDR_Status.losPitchAngle,
                              sizeof(Float64));
   binary[0] = htonl(binary[0]);
   binary[1] = htonl(binary[1]);

   Inf_SecureString::memcpy_s(wordIndex,
                              sizeof(Int32),
                              &binary[1],
                              sizeof(Int32));
   wordIndex += 2;

   Inf_SecureString::memcpy_s(wordIndex,
                              sizeof(Int32),
                              &binary[0],
                              sizeof(Int32));
   wordIndex += 2;

    /////////////////////////////////////////
    //Add Line of Sight Yaw Angle
    /////////////////////////////////////////
    Inf_SecureString::memcpy_s(binary,
                              sizeof(Float64),
                              &SDR_Status.losYawAngle,
                              sizeof(Float64));
   binary[0] = htonl(binary[0]);
   binary[1] = htonl(binary[1]);

   Inf_SecureString::memcpy_s(wordIndex,
                              sizeof(Int32),
                              &binary[1],
                              sizeof(Int32));
   wordIndex += 2;

   Inf_SecureString::memcpy_s(wordIndex,
                              sizeof(Int32),
                              &binary[0],
                              sizeof(Int32));
   wordIndex += 2;

    /////////////////////////////////////////
    //Add Satelite Zenith Angle
    /////////////////////////////////////////
    Inf_SecureString::memcpy_s(binary,
                              sizeof(Float64),
                              &SDR_Status.sateliteZenithAngle,
                              sizeof(Float64));
   binary[0] = htonl(binary[0]);
   binary[1] = htonl(binary[1]);

   Inf_SecureString::memcpy_s(wordIndex,
                              sizeof(Int32),
                              &binary[1],
                              sizeof(Int32));
   wordIndex += 2;

   Inf_SecureString::memcpy_s(wordIndex,
                              sizeof(Int32),
                              &binary[0],
                              sizeof(Int32));
   wordIndex += 2;

    /////////////////////////////////////////
    //Add Satelite Azimuth Angle
    /////////////////////////////////////////
    Inf_SecureString::memcpy_s(binary,
                              sizeof(Float64),
                              &SDR_Status.sateliteAzimuthAngle,
                              sizeof(Float64));
   binary[0] = htonl(binary[0]);
   binary[1] = htonl(binary[1]);

   Inf_SecureString::memcpy_s(wordIndex,
                              sizeof(Int32),
                              &binary[1],
                              sizeof(Int32));
   wordIndex += 2;

   Inf_SecureString::memcpy_s(wordIndex,
                              sizeof(Int32),
                              &binary[0],
                              sizeof(Int32));
   wordIndex += 2;

    /////////////////////////////////////////
    //Add Sun Zenith Angle
    /////////////////////////////////////////
    Inf_SecureString::memcpy_s(binary,
                              sizeof(Float64),
                              &SDR_Status.sunZenithAngle,
                              sizeof(Float64));
   binary[0] = htonl(binary[0]);
   binary[1] = htonl(binary[1]);

   Inf_SecureString::memcpy_s(wordIndex,
                              sizeof(Int32),
                              &binary[1],
                              sizeof(Int32));
   wordIndex += 2;

   Inf_SecureString::memcpy_s(wordIndex,
                              sizeof(Int32),
                              &binary[0],
                              sizeof(Int32));
   wordIndex += 2;

    /////////////////////////////////////////
    //Add Sun Azimuth Angle
    /////////////////////////////////////////
    Inf_SecureString::memcpy_s(binary,
                              sizeof(Float64),
                              &SDR_Status.sunAzimuthAngle,
                              sizeof(Float64));
   binary[0] = htonl(binary[0]);
   binary[1] = htonl(binary[1]);

   Inf_SecureString::memcpy_s(wordIndex,
                              sizeof(Int32),
                              &binary[1],
                              sizeof(Int32));
   wordIndex += 2;

   Inf_SecureString::memcpy_s(wordIndex,
                              sizeof(Int32),
                              &binary[0],
                              sizeof(Int32));
   wordIndex += 2;

    /////////////////////////////////////////
    //Add Satelite Altitude
    /////////////////////////////////////////
    Inf_SecureString::memcpy_s(binary,
                              sizeof(Float64),
                              &SDR_Status.sateliteAltitude,
                              sizeof(Float64));
   binary[0] = htonl(binary[0]);
   binary[1] = htonl(binary[1]);

   Inf_SecureString::memcpy_s(wordIndex,
                              sizeof(Int32),
                              &binary[1],
                              sizeof(Int32));
   wordIndex += 2;

   Inf_SecureString::memcpy_s(wordIndex,
                              sizeof(Int32),
                              &binary[0],
                              sizeof(Int32));
   wordIndex += 2;

    /////////////////////////////////////////
    //Add Range
    /////////////////////////////////////////
    Inf_SecureString::memcpy_s(binary,
                              sizeof(Float64),
                              &SDR_Status.range,
                              sizeof(Float64));
   binary[0] = htonl(binary[0]);
   binary[1] = htonl(binary[1]);

   Inf_SecureString::memcpy_s(wordIndex,
                              sizeof(Int32),
                              &binary[1],
                              sizeof(Int32));
   wordIndex += 2;

   Inf_SecureString::memcpy_s(wordIndex,
                              sizeof(Int32),
                              &binary[0],
                              sizeof(Int32));
   wordIndex += 2;

    /////////////////////////////////////////
    //Add Longitude
    /////////////////////////////////////////
    Inf_SecureString::memcpy_s(binary,
                              sizeof(Float64),
                              &SDR_Status.longitude,
                              sizeof(Float64));
   binary[0] = htonl(binary[0]);
   binary[1] = htonl(binary[1]);

   Inf_SecureString::memcpy_s(wordIndex,
                              sizeof(Int32),
                              &binary[1],
                              sizeof(Int32));
   wordIndex += 2;

   Inf_SecureString::memcpy_s(wordIndex,
                              sizeof(Int32),
                              &binary[0],
                              sizeof(Int32));
   wordIndex += 2;

    /////////////////////////////////////////
    //Add Latitude
    /////////////////////////////////////////
    Inf_SecureString::memcpy_s(binary,
                              sizeof(Float64),
                              &SDR_Status.latitude,
                              sizeof(Float64));
   binary[0] = htonl(binary[0]);
   binary[1] = htonl(binary[1]);

   Inf_SecureString::memcpy_s(wordIndex,
                              sizeof(Int32),
                              &binary[1],
                              sizeof(Int32));
   wordIndex += 2;

   Inf_SecureString::memcpy_s(wordIndex,
                              sizeof(Int32),
                              &binary[0],
                              sizeof(Int32));
   wordIndex += 2;

    ////////////////////////////
    //Add Major X Axis Footprint
    ////////////////////////////

    //copy the majorXAxisFootprint Float32 into an UInt32
   Inf_SecureString::memcpy_s(&uintTemp,
                              sizeof(Float32),
                              &SDR_Status.majorXAxisFootprint,
                              sizeof(Float32));

   //convert UInt32 with host to network long
   uintTemp = htonl(uintTemp);

   //copy converted UInt32 to the output buffer
   Inf_SecureString::memcpy_s(wordIndex,
                              sizeof(Float32),
                              &uintTemp,
                              sizeof(Float32));
   wordIndex +=2;

    ////////////////////////////
    //Add Minor X Axis Footprint
    ////////////////////////////

    //copy the minorXAxisFootprint Float32 into an UInt32
    Inf_SecureString::memcpy_s(&uintTemp,
                              sizeof(Float32),
                              &SDR_Status.minorXAxisFootprint,
                              sizeof(Float32));

   //convert UInt32 with host to network long
   uintTemp = htonl(uintTemp);

   //copy converted UInt32 to the output buffer
   Inf_SecureString::memcpy_s(wordIndex,
                              sizeof(Float32),
                              &uintTemp,
                              sizeof(Float32));
   wordIndex +=2;

    /////////////////////
    //Add RDR status word
    /////////////////////
    wordValue = (UInt8)RDR_Status.impulseNoiseCount << 8;
    wordValue |= RDR_Status.invalidData << 7;
    wordValue |= RDR_Status.fringeCountError << 6;
    wordValue |= RDR_Status.bitTrimFailure << 5;

    *wordIndex = htons(wordValue);
    wordIndex++;

    ///////////////////
    //Add ZPD Amplitude
    ///////////////////
    *wordIndex = htons(RDR_Status.zpdAmplitude);
    wordIndex++;

    /////////////////////
    //Add ZPD Finge Count
    /////////////////////
    *wordIndex = htons(RDR_Status.zpdLocation);
    wordIndex++;

    ///////////////////////////
    //Add SDR Status Upper word
    ///////////////////////////
    wordValue = 0;
    *wordIndex = htons(wordValue);
    wordIndex++;

    ///////////////////////////
    //Add SDR Status Lower word
    ///////////////////////////
    wordValue |= SDR_Status.invalidData << 15;
    wordValue |= SDR_Status.fringeCountErrorDetected << 14;
    wordValue |= SDR_Status.fringeCountErrorCorrected << 13;
    wordValue |= SDR_Status.fringeCountErrorCorrectionFailure << 12;
    wordValue |= SDR_Status.fringeCountErrorExcessive << 11;
    wordValue |= SDR_Status.invalidNeonCalibration << 10;
    wordValue |= SDR_Status.thermalDriftExcessive << 9;
    wordValue |= SDR_Status.excessiveNEdN << 8;
    wordValue |= SDR_Status.invalidSpectralCalibration << 7;
    wordValue |= SDR_Status.calibrationType << 6;
    wordValue |= SDR_Status.invalidGeolocation << 5;
    wordValue |= SDR_Status.apodizationType << 2;
    wordValue |= SDR_Status.lunarIntrusionDetected << 1;
    wordValue |= SDR_Status.spare << 0;

    *wordIndex = htons(wordValue);
    wordIndex++;

    //////////////////////
    //Add SDR Fringe Count
    //////////////////////
    *wordIndex = htons((UInt16)SDR_Status.sdrFringeCount);
    wordIndex++;

    ////////////////////////////////
    //Add DS Measurement Window Size
    ////////////////////////////////
    *wordIndex = htons(SDR_Status.coldCalTargetSignificance);
    wordIndex++;

    //Append Internal Calibration Measurement Window Size
    *wordIndex = htons(SDR_Status.hotCalTargetSignificance);
    wordIndex++;

    ///////////////////////////////
    //Add Measured Laser Wavelength
    ///////////////////////////////
    Inf_SecureString::memcpy_s(binary,
                              sizeof(Float64),
                              &SDR_Status.laserWavelength,
                              sizeof(Float64));
   binary[0] = htonl(binary[0]);
   binary[1] = htonl(binary[1]);

   Inf_SecureString::memcpy_s(wordIndex,
                              sizeof(Int32),
                              &binary[1],
                              sizeof(Int32));
   wordIndex += 2;

   Inf_SecureString::memcpy_s(wordIndex,
                              sizeof(Int32),
                              &binary[0],
                              sizeof(Int32));
   wordIndex += 2;

    /////////////////////////////////////////
    //Add Spectral Resampling Laserwavelength
    /////////////////////////////////////////
    Inf_SecureString::memcpy_s(binary,
                              sizeof(Float64),
                              &SDR_Status.spectralResamplingLaserWavelength,
                              sizeof(Float64));
   binary[0] = htonl(binary[0]);
   binary[1] = htonl(binary[1]);

   Inf_SecureString::memcpy_s(wordIndex,
                              sizeof(Int32),
                              &binary[1],
                              sizeof(Int32));
   wordIndex += 2;

   Inf_SecureString::memcpy_s(wordIndex,
                              sizeof(Int32),
                              &binary[0],
                              sizeof(Int32));
   wordIndex += 2;

    ////////////////////////////////
    //Add Monitored Laser Wavelength
    ////////////////////////////////
    Inf_SecureString::memcpy_s(binary,
                              sizeof(Float64),
                              &SDR_Status.monitoredLaserWavelength,
                              sizeof(Float64));
   binary[0] = htonl(binary[0]);
   binary[1] = htonl(binary[1]);

   Inf_SecureString::memcpy_s(wordIndex,
                              sizeof(Int32),
                              &binary[1],
                              sizeof(Int32));
   wordIndex += 2;

   Inf_SecureString::memcpy_s(wordIndex,
                              sizeof(Int32),
                              &binary[0],
                              sizeof(Int32));
   wordIndex += 2;

    ///////////////////////////////
    //Add Spectral Wavenumber Start
    ///////////////////////////////

    //assign Float64 to a Float32
    floatTemp = (Float32)firstWavenumber;

   //copy the wavenumberVal Float32 into an UInt32
   Inf_SecureString::memcpy_s(&uintTemp,
                              sizeof(Float32),
                              &floatTemp,
                              sizeof(Float32));

   //convert UInt32 with host to network long
   uintTemp = htonl(uintTemp);

   //copy converted UInt32 to the output buffer
   Inf_SecureString::memcpy_s(wordIndex,
                              sizeof(Float32),
                              &uintTemp,
                              sizeof(Float32));
   wordIndex +=2;

    /////////////////////////////////
    //Add Spectral Wavenumber Spacing
    /////////////////////////////////

    //assign Float64 to a Float32
    floatTemp = (Float32)wavenumberBinSize;

   //copy the wavenumberVal Float32 into an UInt32
   Inf_SecureString::memcpy_s(&uintTemp,
                              sizeof(Float32),
                              &floatTemp,
                              sizeof(Float32));

   //convert UInt32 with host to network long
   uintTemp = htonl(uintTemp);

   //copy converted UInt32 to the output buffer
   Inf_SecureString::memcpy_s(wordIndex,
                              sizeof(Float32),
                              &uintTemp,
                              sizeof(Float32));
   wordIndex +=2;

    //////////////////////////////////
    //Add Spectra Wavenumber Bin Count
    //////////////////////////////////
    wordValue = htons((UInt16)realWavenumberBin.size());
    *wordIndex = wordValue;
    wordIndex++;

    ////////////////////////////
    //Add Real Wavenumber Values
    ////////////////////////////
    for (index = 0; index < realWavenumberBin.size(); index++)
    {
        //assign Float64 to a Float32
        floatTemp = (Float32)realWavenumberBin[index];

        //copy the wavenumberVal Float32 into an UInt32
        Inf_SecureString::memcpy_s(&uintTemp,
                                 sizeof(Float32),
                                 &floatTemp,
                                 sizeof(Float32));

        //convert UInt32 with host to network long
        uintTemp = htonl(uintTemp);

        //copy converted UInt32 to the output buffer
        Inf_SecureString::memcpy_s(wordIndex,
                                 sizeof(Float32),
                                 &uintTemp,
                                 sizeof(Float32));
        wordIndex += 2;
    }

    /////////////////////////////////
    //Add Imaginary Wavenumber Values
    /////////////////////////////////
    for (index = 0; index < imaginaryWavenumberBin.size(); index++)
    {
        //assign Float64 to a Float32
        floatTemp = (Float32)imaginaryWavenumberBin[index];

        //copy the wavenumberVal Float32 into an UInt32
        Inf_SecureString::memcpy_s(&uintTemp,
                                 sizeof(Float32),
                                 &floatTemp,
                                 sizeof(Float32));

        //convert UInt32 with host to network long
        uintTemp = htonl(uintTemp);

        //copy converted UInt32 to the output buffer
        Inf_SecureString::memcpy_s(wordIndex,
                                 sizeof(Float32),
                                 &uintTemp,
                                 sizeof(Float32));
        wordIndex += 2;
    }

    ////////////////////////////
    //Add NEdN Wavenumber Values
    ////////////////////////////
    for (index = 0; index < NEdN_Estimate.size(); index++)
    {
        //assign Float64 to a Float32
        floatTemp = (Float32)NEdN_Estimate[index];

        //copy the wavenumberVal Float32 into an UInt32
        Inf_SecureString::memcpy_s(&uintTemp,
                                 sizeof(Float32),
                                 &floatTemp,
                                 sizeof(Float32));

        //convert UInt32 with host to network long
        uintTemp = htonl(uintTemp);

        //copy converted UInt32 to the output buffer
        Inf_SecureString::memcpy_s(wordIndex,
                                 sizeof(Float32),
                                 &uintTemp,
                                 sizeof(Float32));
        wordIndex += 2;
    }

    delete [] binary;
    binary = 0;
}


void Spectra::generateUSN(Spectra*, //rawSum
                          Spectra* squaredSum,
                          Int32 windowSize)
{
   const BOOST::vector<Float64>* squareMagBin = &squaredSum->getReal();
   const BOOST::vector<Float64>* sumMagBin = &squaredSum->getNEdN();
   Float64 tempSumMagnitude;
   Float64 tempSumMagSquared;
   Float64 tempCalculation;
   Int32    numPoints = realWavenumberBin.size();
   if (NEdN_Estimate.size() != numPoints)
   {
       NEdN_Estimate.resize(numPoints);
   }
   if (squareMagBin->size() < numPoints)
   {
       windowSize = 0;
   }
   for (Int32 index = 0; index < numPoints; index++)
   {
       if (windowSize > 2)  // Don't want a divide by zero (if 1), need at
                            // least 3 points for an SD
       {
           tempSumMagnitude     = (*squareMagBin)[index];
         tempSumMagSquared    = (*sumMagBin)[index] / (Float64) windowSize;
         tempSumMagSquared   *= tempSumMagSquared;
         tempSumMagSquared   *= windowSize;
         tempCalculation      = tempSumMagnitude - tempSumMagSquared;
           NEdN_Estimate[index] = sqrt(tempCalculation / ((Float64) windowSize - 1.0));
       }
       else
       {
           NEdN_Estimate[index] = 0.0;
       }
   }
}


void Spectra::smooth(Int32 windowSize)
{
   (windowSize % 2) ? windowSize : windowSize += 1;
   Int32 filterDelay = windowSize / 2;
   Int32 dataSize    = realWavenumberBin.size();
   for (Int32 index = filterDelay; index < dataSize - filterDelay; index++)
   {
      Float64 avgRealSample = 0;
      Float64 avgImagSample = 0;
      for(Int32 sum = -filterDelay; sum <= filterDelay; sum++)
      {
         avgRealSample     += realWavenumberBin[index + sum];
         avgImagSample     += imaginaryWavenumberBin[index + sum];
      }
      realWavenumberBin[index - filterDelay] = avgRealSample / windowSize;
      imaginaryWavenumberBin[index - filterDelay] = avgImagSample / windowSize;
   }
   realWavenumberBin.resize(dataSize - windowSize);
   imaginaryWavenumberBin.resize(dataSize - windowSize);
   NEdN_Estimate = imaginaryWavenumberBin;
   firstWavenumber = firstWavenumber + (filterDelay * wavenumberBinSize);
}


void Spectra::applyLinearityErrorCorrection()
{
   if(!SDR_Status.linearityCorrMode)
   {
      if(!SDR_Status.invalidData && FloatCompare<Float64>::equal(
          SDR_Status.LinErrCorrectionFactor,0.0) == false)
      {
         realWavenumberBin *= SDR_Status.LinErrCorrectionFactor;
         imaginaryWavenumberBin *= SDR_Status.LinErrCorrectionFactor;


         SDR_Status.linearityCorrMode = true;
      }
      else
      {
         SDR_Status.invalidData = true;
      }
   }
   else
   {
      //already corrected
      EventLogEngine::append("Design Error: Repeated LinErr correction.");
   }
}

void Spectra::updateCrisSdrGeolocation(CrisSdrAlgDataType* algDataPtr,
                                       UInt32 scanIdx, Int32 forTimeStampBias)
{


   if(fieldOfRegard == InternalCalTarget || fieldOfRegard == DeepSpace ||
      fieldOfRegard == NonStandardDwell || fieldOfRegard == TOTAL_SCENES)
    {
        return;
    }
    else
    {
        algDataPtr->geoTcPtr->angData.roll[scanIdx][fieldOfRegard-1]
                                          [fieldOfView]
            = getLosRollAngle();
        algDataPtr->geoTcPtr->angData.pitch[scanIdx][fieldOfRegard-1]
                                           [fieldOfView]
            = getLosPitchAngle();
    }

    UInt64 tmpTime = 0;

    tmpTime = (UInt64)RDR_Status.days << DAYS_TO_MICROSECS_SHIFT_CONV |
        (UInt64)RDR_Status.milliseconds
                << MILISECS_TO_MICROSECS_SHIFT_CONV |
        (UInt64)RDR_Status.microseconds;

    if (ProCmnTimeUtil::getInstance()->cds2Iet(
            tmpTime,
            &algDataPtr->geoPtr->FOR_IETusec[scanIdx][fieldOfRegard-1]) != 0)
    {
        (void)EventLogEngine::append("cds2Iet returned nonzero value.");
    }

    algDataPtr->geoPtr->FOR_IETusec[scanIdx][fieldOfRegard-1] +=
            forTimeStampBias / MILLI;

    InfUtil_TimeGdt gdttime;

    InfUtil_TimApi::getInstance()->iet2Gdt(
            algDataPtr->geoPtr->FOR_IETusec[scanIdx][fieldOfRegard-1],
            &gdttime);

}

void Spectra::updateCrisSdrData(CrisSdrData *sdrPtr, CrisSdrHdrDataType *hdrPtr,
                                UInt32 scanIdx,
                   std::auto_ptr<ScienceDataProcessor>& ScienceDataProcessorPtr)
{

    if(fieldOfRegard == NonStandardDwell || fieldOfRegard == TOTAL_SCENES)
    {
        return;
    }
    else if(fieldOfRegard == InternalCalTarget)
    {
       // update ICT spike correction status in QF2 flag                                                                       
       if (RDR_Status.spikeDetectionCorrection != NO_SPIKE)                                                                    
       {                                                                                                                       
          UInt8 oldQF2Flag = sdrPtr->QF2[scanIdx][fieldOfView][spectralBand]                                                  
             & ICT_SPIKE_STATUS;                                                                              
          if (oldQF2Flag == 0) // only 1 direction detected spike                                                               
          {                                                                                                                     
             // write status                                                                                                     
             if (RDR_Status.spikeDetectionCorrection == 1)                                                                       
                sdrPtr->QF2[scanIdx][fieldOfView][spectralBand] |= 
                   ICT_SPIKE_DETECTED;                                      
             else                                                                                                                
                sdrPtr->QF2[scanIdx][fieldOfView][spectralBand] |= 
                   ICT_SPIKE_CORRECTED;                                     
             // write direction                                                                                                  
             if (sweepDirection == REVERSE)                                                                                      
                sdrPtr->QF2[scanIdx][fieldOfView][spectralBand] |= 
                   ICT_SPIKE_IN_REVERSE;                                    
          }                                                                                                                     
          else // both direction has spike                                                                                      
          {                                                                                                                     
             oldQF2Flag = sdrPtr->QF2[scanIdx][fieldOfView][spectralBand] 
                & ICT_SPIKE_CORRECTION_STATUS;                         
             if (oldQF2Flag == ICT_SPIKE_DETECTED && 
                   RDR_Status.spikeDetectionCorrection == 1)                                   
             {                                                                                                                   
                sdrPtr->QF2[scanIdx][fieldOfView][spectralBand] &= 
                   ICT_DEFAULT_SPIKE_STATUS;                                      
                sdrPtr->QF2[scanIdx][fieldOfView][spectralBand] |= 
                   ICT_SPIKE_BOTH_DETECTED;                                       
             }                                                                                                                   
             else if (oldQF2Flag == ICT_SPIKE_CORRECTED 
                   && RDR_Status.spikeDetectionCorrection == 2)                             
             {                                                                                                                   
                sdrPtr->QF2[scanIdx][fieldOfView][spectralBand] &= 
                   ICT_DEFAULT_SPIKE_STATUS;                                      
                sdrPtr->QF2[scanIdx][fieldOfView][spectralBand] |= 
                   ICT_SPIKE_BOTH_CORRECTED;                                      
             }                                                                                                                   
             else                                                                                                                
             {                                                                                                                   
                sdrPtr->QF2[scanIdx][fieldOfView][spectralBand] &= 
                   ICT_DEFAULT_SPIKE_STATUS;                                      
                sdrPtr->QF2[scanIdx][fieldOfView][spectralBand] |= 
                   ICT_SPIKE_ONE_DETECTED_ONE_CORRECTED;                          
             }                                                                                                                   
          }                                                                                                                     
       }                                                                                                                       
       if(sweepDirection == REVERSE)
       {
          // If FringCountErrorHandling flag is on, when the last spectra
          // ICT Spectra[REVERSE][SHORTWAVE][FOV9]
          // of each scan is pulled in, all ICT Spectra
          // [REVERSE][All bands][All FOVs] are saved then, So output
          // ICT_SpectraStability[REVERSE][All bands][All FOVs] here 
          // to align with FringCountErrorHandling off
          if(spectralBand == Band::SHORTWAVE && fieldOfView == FOV9)
          {
             for(Int32 theFOV = 0; theFOV < MAX_FOV; theFOV++)
             {
                for(Int32 theBand = 0; theBand < Band::TOTAL_BANDS; theBand++)
                {
                   sdrPtr->ICT_SpectraStability[scanIdx][REVERSE][theFOV][theBand] =
                        ScienceDataProcessorPtr->getIctSpectraStab
                           (theFOV, theBand, REVERSE);
                }
             }
          }

// ADD --- ICT spectra[REVERSE is the last spectra of each scan, so set outputs here
          sdrPtr->ICT_SpectraStability
                    [scanIdx][FORWARD][fieldOfView][spectralBand] =
                        ScienceDataProcessorPtr->
                          getIctSpectraStab(fieldOfView, spectralBand, FORWARD);
          sdrPtr->DS_SpectralStability
                    [scanIdx][REVERSE][fieldOfView][spectralBand] =
                        ScienceDataProcessorPtr->
                           getDsSpectraStab(fieldOfView, spectralBand, REVERSE);
          sdrPtr->DS_SpectralStability
                    [scanIdx][FORWARD][fieldOfView][spectralBand] =
                        ScienceDataProcessorPtr->
                           getDsSpectraStab(fieldOfView, spectralBand, FORWARD);

       }
    }
    else if(fieldOfRegard == DeepSpace)
    {
       // update DS spike correction status in QF2 flag (added by huixu)                                                      
       if (RDR_Status.spikeDetectionCorrection != NO_SPIKE)                                                                   
       {                                                                                                                      
          UInt8 oldQF2Flag = sdrPtr->QF2[scanIdx][fieldOfView][spectralBand] 
             & DS_SPIKE_STATUS;                                
          if (oldQF2Flag == 0) // only 1 direction detected spike                                                              
          {                                                                                                                    
             // write status                                                                                                    
             if (RDR_Status.spikeDetectionCorrection == 1)                                                                      
                sdrPtr->QF2[scanIdx][fieldOfView][spectralBand] |= 
                   DS_SPIKE_DETECTED;                                      
             else                                                                                                               
                sdrPtr->QF2[scanIdx][fieldOfView][spectralBand] |= 
                   DS_SPIKE_CORRECTED;                                     
             // write direction                                                                                                 
             if (sweepDirection == REVERSE)                                                                                     
                sdrPtr->QF2[scanIdx][fieldOfView][spectralBand] |= 
                   DS_SPIKE_IN_REVERSE;                                    
          }                                                                                                                    
          else // both direction has spike                                                                                     
          {                                                                                                                    
             oldQF2Flag = sdrPtr->QF2[scanIdx][fieldOfView][spectralBand] 
                & DS_SPIKE_CORRECTION_STATUS;                         
             if (oldQF2Flag == DS_SPIKE_DETECTED && 
                   RDR_Status.spikeDetectionCorrection == 1)                                   
             {                                                                                                                  
                sdrPtr->QF2[scanIdx][fieldOfView][spectralBand] &= 
                   DS_DEFAULT_SPIKE_STATUS;                                     
                sdrPtr->QF2[scanIdx][fieldOfView][spectralBand] |= 
                   DS_SPIKE_BOTH_DETECTED;                                      
             }                                                                                                                  
             else if (oldQF2Flag == DS_SPIKE_CORRECTED && 
                   RDR_Status.spikeDetectionCorrection == 2)                             
             {                                                                                                                  
                sdrPtr->QF2[scanIdx][fieldOfView][spectralBand] &= 
                   DS_DEFAULT_SPIKE_STATUS;                                     
                sdrPtr->QF2[scanIdx][fieldOfView][spectralBand] |= 
                   DS_SPIKE_BOTH_CORRECTED;                                     
             }                                                                                                                  
             else                                                                                                               
             {                                                                                                                  
                sdrPtr->QF2[scanIdx][fieldOfView][spectralBand] &= 
                   DS_DEFAULT_SPIKE_STATUS;                                     
                sdrPtr->QF2[scanIdx][fieldOfView][spectralBand] |= 
                   DS_SPIKE_ONE_DETECTED_ONE_CORRECTED;                         
             }                                                                                                                  
          }                                                                                                                    
       }                                                                                                                      
       // Get DS interferogram
       Int32 spectra_indx = ScienceDataProcessorPtr->
                     getSpectraCount(fieldOfView, spectralBand, 
                        fieldOfRegard, sweepDirection)/2 + 1;

       if(ScienceDataProcessorPtr->getPerformFringeCountErrorHandling())
       {
          spectra_indx++; // With the fringeCountErrorHanding flag turned on,
                          // DS spectra is delayed
                          // to push into the container through calling 
                          // launchBatchReferenceDetect(),
                          // So need to increase the spectra_indx by 1
       }

       Spectra *spectraDS = ScienceDataProcessorPtr->getMeasuredSpectra(
                               fieldOfView, spectralBand, fieldOfRegard, 
                               sweepDirection, spectra_indx);

       if(sweepDirection == REVERSE)
       {
          ScienceDataProcessorPtr->set_revDS_reals(scanIdx, fieldOfView,
                       spectralBand, spectraDS->get_interferogram_DS_reals());
          ScienceDataProcessorPtr->set_revDS_imags(scanIdx, fieldOfView,
                       spectralBand, spectraDS->get_interferogram_DS_imags());

       }
       else if(sweepDirection == FORWARD)
       {
          ScienceDataProcessorPtr->set_forDS_reals(scanIdx, fieldOfView,
                       spectralBand, spectraDS->get_interferogram_DS_reals());
          ScienceDataProcessorPtr->set_forDS_imags(scanIdx, fieldOfView,
                       spectralBand, spectraDS->get_interferogram_DS_imags());

       }
    }
    else
    {

        // Do not set the header fields from "missing" packets
        if (!getRDR_MissingData())
        {
          hdrPtr->flightSoftwareVersion = RDR_Status.flightSoftwareVersion;
          sprintf(hdrPtr->sensorID, "%d", RDR_Status.sensorID);
          sprintf(hdrPtr->softwareVersion, "%d", getVersionNumber());
        }


        // Set laser wavelengths only if the packet contains actual data
        // If it's a missing packet, let the values default to fill
        if (!getRDR_MissingData())
        {
           sdrPtr->resamplingLaserWavelength[scanIdx] =
                 SDR_Status.spectralResamplingLaserWavelength;
           sdrPtr->monitoredLaserWavelength[scanIdx] =
                 SDR_Status.monitoredLaserWavelength;
        }

        // Update DS and ICT Window size.  Don't need to update on every ES FOR
        // so only update if the window size is at it's initialized value.
        if(sweepDirection == REVERSE)
        {
          if (ProCmnFillType<UInt16>::isFill(sdrPtr->
                 DS_WindowSize[scanIdx][REVERSE][fieldOfView][spectralBand]))
          {
             sdrPtr->
                DS_WindowSize[scanIdx][REVERSE][fieldOfView][spectralBand] =
                ScienceDataProcessorPtr->getMeasuredSpectraValidSize(fieldOfView,
                   spectralBand, DeepSpace, REVERSE);
          }
          if (ProCmnFillType<UInt16>::isFill(sdrPtr->
                  ICT_WindowSize[scanIdx][REVERSE][fieldOfView][spectralBand]))
          {
             sdrPtr->ICT_WindowSize[scanIdx][REVERSE][fieldOfView][spectralBand] =
                ScienceDataProcessorPtr->getMeasuredSpectraValidSize(fieldOfView,
                   spectralBand, InternalCalTarget, REVERSE);
          }
        }
        else if(sweepDirection == FORWARD)
        {
           if (ProCmnFillType<UInt16>::isFill(sdrPtr->
                   DS_WindowSize[scanIdx][FORWARD][fieldOfView][spectralBand]))
           {
              sdrPtr->
                 DS_WindowSize[scanIdx][FORWARD][fieldOfView][spectralBand] =
                 ScienceDataProcessorPtr->getMeasuredSpectraValidSize(fieldOfView,
                    spectralBand, DeepSpace, FORWARD);
           }
           if (ProCmnFillType<UInt16>::isFill(sdrPtr->
                   ICT_WindowSize[scanIdx][FORWARD][fieldOfView][spectralBand]))
           {
              sdrPtr->
                 ICT_WindowSize[scanIdx][FORWARD][fieldOfView][spectralBand] =
                 ScienceDataProcessorPtr->getMeasuredSpectraValidSize(fieldOfView,
                    spectralBand, InternalCalTarget, FORWARD);
           }
        }

        if(spectralBand == LONGWAVE)
        {

#ifdef PSEUDO_TEST_DATA
            for(Int32 index = 0; index < realWavenumberBin.size() &&
                    index < LWNUMBEROFPOINTS; index++)
#else
            for(Int32 index = 0; index < LWNUMBEROFPOINTS; index++)
#endif

            {
               if (!getRDR_MissingData())
               {
                  sdrPtr->ESRealLW[scanIdx][fieldOfRegard-1]
                     [fieldOfView][index] = realWavenumberBin[index];
                  sdrPtr->ESImaginaryLW[scanIdx][fieldOfRegard-1]
                     [fieldOfView][index] = imaginaryWavenumberBin[index];
                  sdrPtr->ESNEdNLW[scanIdx][fieldOfRegard-1]
                     [fieldOfView][index] = NEdN_Estimate[index];
               }
               else // If it's a missing packet, set the values to fill
               {
                  sdrPtr->ESRealLW[scanIdx][fieldOfRegard-1][fieldOfView]
                                  [index] = MISS_FLOAT32_FILL;
                  sdrPtr->ESImaginaryLW[scanIdx][fieldOfRegard-1][fieldOfView]
                                       [index] = MISS_FLOAT32_FILL;
                  sdrPtr->ESNEdNLW[scanIdx][fieldOfRegard-1][fieldOfView]
                                  [index] = MISS_FLOAT32_FILL;
               }
            }
        }
        else if(spectralBand == MIDWAVE)
        {
#ifdef PSEUDO_TEST_DATA
            for(Int32 index = 0; index < realWavenumberBin.size() &&
                    index < MW_NUMBER_OF_POINTS; index++)
#else
            for(Int32 index = 0; index < MW_NUMBER_OF_POINTS; index++)
#endif
            {
               if (!getRDR_MissingData())
               {
                  sdrPtr->ESRealMW[scanIdx][fieldOfRegard-1]
                     [fieldOfView][index] = realWavenumberBin[index];
                  sdrPtr->ESImaginaryMW[scanIdx][fieldOfRegard-1]
                     [fieldOfView][index] = imaginaryWavenumberBin[index];
                  sdrPtr->ESNEdNMW[scanIdx][fieldOfRegard-1]
                     [fieldOfView][index] = NEdN_Estimate[index];
               }
               else // If it's a missing packet, set the values to fill
               {
                  sdrPtr->ESRealMW[scanIdx][fieldOfRegard-1][fieldOfView]
                                  [index] = MISS_FLOAT32_FILL;
                  sdrPtr->ESImaginaryMW[scanIdx][fieldOfRegard-1][fieldOfView]
                                       [index] = MISS_FLOAT32_FILL;
                  sdrPtr->ESNEdNMW[scanIdx][fieldOfRegard-1][fieldOfView]
                                  [index] = MISS_FLOAT32_FILL;
               }
            }
        }
        else if(spectralBand == SHORTWAVE)
        {
#ifdef PSEUDO_TEST_DATA
            for(Int32 index = 0; index < realWavenumberBin.size() &&
                    index < SW_NUMBER_OF_POINTS; index++)
#else
            for(Int32 index = 0; index < SW_NUMBER_OF_POINTS; index++)
#endif
            {
               if (!getRDR_MissingData())
               {
                  sdrPtr->ESRealSW[scanIdx][fieldOfRegard-1]
                     [fieldOfView][index] = realWavenumberBin[index];
                  sdrPtr->ESImaginarySW[scanIdx][fieldOfRegard-1]
                     [fieldOfView][index] = imaginaryWavenumberBin[index];
                  sdrPtr->ESNEdNSW[scanIdx][fieldOfRegard-1]
                     [fieldOfView][index] = NEdN_Estimate[index];
               }
               else // If it's a missing packet, set the values to fill
               {
                  sdrPtr->ESRealSW[scanIdx][fieldOfRegard-1][fieldOfView]
                                  [index] = MISS_FLOAT32_FILL;
                  sdrPtr->ESImaginarySW[scanIdx][fieldOfRegard-1][fieldOfView]
                                       [index] = MISS_FLOAT32_FILL;
                  sdrPtr->ESNEdNSW[scanIdx][fieldOfRegard-1][fieldOfView]
                                  [index] = MISS_FLOAT32_FILL;
               }
            }
        }

        // None of these should be set for missing packets.
        if (!getRDR_MissingData())
        {
           if (SDR_Status.fringeCountErrorCorrectionFailure)
           {
              sdrPtr->QF3[scanIdx][fieldOfRegard-1][fieldOfView][spectralBand]
                    |= FRINGE_COUNT_ERROR_CORRECT_FAIL;
           }

           if (SDR_Status.lunarIntrusionDetected)
           {
              if(sweepDirection == REVERSE)
              {
                 sdrPtr->QF2[scanIdx][fieldOfView][spectralBand] |=
                    LUNAR_INTRUSION_REV;
              }
              else if(sweepDirection == FORWARD)
              {
                 sdrPtr->QF2[scanIdx][fieldOfView][spectralBand] |=
                    LUNAR_INTRUSION_FWD;
              }
           }

           if (SDR_Status.lambda_monitored_quality)
           {
              sdrPtr->QF1[scanIdx] |= LAMBDA_MONITORED_QUALITY;
           }

           if (RDR_Status.fringeCountError)
           {
              sdrPtr->QF4[scanIdx][fieldOfRegard-1][fieldOfView][spectralBand]
                    |= FRINGE_COUNT_ERROR_DETECTED;
           }

           if (RDR_Status.bitTrimFailure)
           {
              sdrPtr->QF4[scanIdx][fieldOfRegard-1][fieldOfView][spectralBand]
                    |= BIT_TRIM_FAILED;
           }

           // update ES spike correction status in QF4 and QF3 flag                                                         
           if (RDR_Status.spikeDetectionCorrection != NO_SPIKE)                                                             
           {                                                                                                                
              if (RDR_Status.spikeDetectionCorrection == SPIKE_DETECTED)                                               
              {                                                                                                        
                 sdrPtr->QF4[scanIdx][fieldOfRegard - 1]                  
                    [fieldOfView][spectralBand] |= ES_SPIKE_DETECTED;              
                 sdrPtr->QF3[scanIdx][fieldOfRegard - 1]                  
                    [fieldOfView][spectralBand] |= SDR_QUALITY_DEGRADED;                                                                      
              }                                                                                                        
              else                                                                                                     
              {                                                                                                        
                 sdrPtr->QF4[scanIdx][fieldOfRegard - 1]
                    [fieldOfView][spectralBand] |= ES_SPIKE_CORRECTED;             
                 sdrPtr->QF3[scanIdx][fieldOfRegard - 1]
                    [fieldOfView][spectralBand] |= SDR_QUALITY_GOOD;                                                                          
              }                                                                                                        
           }

	   // K. Zhang, ADR-11194, set QF3 Invalid Spectral Cal Flag to 3 (0x60)
	   if (SDR_Status.useConfigLaserWavelength)
           {
              sdrPtr->QF3[scanIdx][fieldOfRegard - 1]                  
                 [fieldOfView][spectralBand] |= CONFIG_SPECTRAL_CALIBRATION;
	   }	  
        }

        if (!getRDR_MissingData())
        {
           sdrPtr->ESRdrImpulseNoise[scanIdx][fieldOfRegard-1][fieldOfView]
                                [spectralBand] = RDR_Status.impulseNoiseCount;
           sdrPtr->ESZPDAmplitude[scanIdx][fieldOfRegard-1][fieldOfView]
                                 [spectralBand] = RDR_Status.zpdAmplitude;
           sdrPtr->ESZPDFringeCount[scanIdx][fieldOfRegard-1][fieldOfView]
                                   [spectralBand] = RDR_Status.zpdLocation;
           sdrPtr->sdrFringeCount[scanIdx][fieldOfRegard-1][fieldOfView]
                                 [spectralBand] +=
                                    (UInt16)SDR_Status.sdrFringeCount;

        }
        else // If it's a missing packet, set the values to fill
        {
           sdrPtr->ESRdrImpulseNoise[scanIdx][fieldOfRegard-1][fieldOfView]
                                    [spectralBand] = MISS_UINT8_FILL;
           sdrPtr->ESZPDAmplitude[scanIdx][fieldOfRegard-1][fieldOfView]
                                 [spectralBand] = MISS_INT16_FILL;
           sdrPtr->ESZPDFringeCount[scanIdx][fieldOfRegard-1][fieldOfView]
                                   [spectralBand] = MISS_UINT16_FILL;
           sdrPtr->sdrFringeCount[scanIdx][fieldOfRegard-1][fieldOfView]
                                 [spectralBand] = MISS_UINT16_FILL;
        }

        if (RDR_Status.invalidData)
        {
           sdrPtr->QF4[scanIdx][fieldOfRegard-1][fieldOfView][spectralBand]
                 |= INVALID_RDR_DATA;
        }

        sdrPtr->measuredLaserWavelength[scanIdx] =
           ScienceDataProcessorPtr->getEngineeringCalibrationRecord().
              getAverageMetrologyWavelength();
    }
}

// FCE update --- After spectra phase shifted, need to 
// re-generate interferograms accordingly
void Spectra::reGenerateInterferogram()
{
   if( !(fieldOfRegard == DeepSpace && !RDR_Status.missingData) )
   {
      return;
   }

   BOOST::vector<Float64 > realWavenumberBin_tmp = realWavenumberBin;
   BOOST::vector<Float64 > imagWavenumberBin_tmp = imaginaryWavenumberBin;

   UInt32 sampleCount[Band::TOTAL_BANDS] =
        {
            TelemetryProcessor::instance()->theVideoData.
               getExtractionRecord(VideoData::LW).complexSamples - 
               2 * NUMBER_OPD_OVERSCAN_SAMPLES,
            TelemetryProcessor::instance()->theVideoData.
               getExtractionRecord(VideoData::MW).complexSamples -
               2 * NUMBER_OPD_OVERSCAN_SAMPLES,
            TelemetryProcessor::instance()->theVideoData.
               getExtractionRecord(VideoData::SW).complexSamples - 
               2 * NUMBER_OPD_OVERSCAN_SAMPLES
        };

   UInt32 decimationFactor[Band::TOTAL_BANDS]  =
        {
            TelemetryProcessor::instance()->theVideoData.
               getExtractionRecord(VideoData::LW).decimationRate,
            TelemetryProcessor::instance()->theVideoData.
               getExtractionRecord(VideoData::MW).decimationRate,
            TelemetryProcessor::instance()->theVideoData.
               getExtractionRecord(VideoData::SW).decimationRate
        };

   //scale spectra
   if(scalingEnabled)
   {
      Float64 factor = (Float64)((Float64)
             (LASER_WAVELENGTH[spectralBand] / 2.0) * 
             decimationFactor[spectralBand] * 0.0000001);
      realWavenumberBin_tmp /= factor;
      imagWavenumberBin_tmp /= factor;
   }

   std::rotate(
         realWavenumberBin_tmp.begin(),
         realWavenumberBin_tmp.begin() + realWavenumberBin_tmp.size() - 
             static_cast<Int32>(foldIndex),
         realWavenumberBin_tmp.end());
   std::rotate(
         imagWavenumberBin_tmp.begin(),
         imagWavenumberBin_tmp.begin() + imagWavenumberBin_tmp.size() - 
              static_cast<Int32>(foldIndex),
         imagWavenumberBin_tmp.end());
   
   // Inverse  FFT to get interferograms
   UInt32 shiftSize = (sampleCount[spectralBand])/2;

   std::rotate(
           realWavenumberBin_tmp.begin(),
           realWavenumberBin_tmp.begin() + realWavenumberBin_tmp.size() - 
               (Int32)shiftSize,
           realWavenumberBin_tmp.end());
   std::rotate(
           imagWavenumberBin_tmp.begin(),
           imagWavenumberBin_tmp.begin() + imagWavenumberBin_tmp.size() - 
               (Int32)shiftSize,
           imagWavenumberBin_tmp.end());

   UInt32 N = interferogram_DS_reals.size();
   for(Int32 i = NUMBER_OPD_OVERSCAN_SAMPLES; 
       i < N-NUMBER_OPD_OVERSCAN_SAMPLES; i++)
   {
      interferogram_DS_reals[i] = 
         realWavenumberBin_tmp[i-NUMBER_OPD_OVERSCAN_SAMPLES];
      interferogram_DS_imags[i] = 
         imagWavenumberBin_tmp[i-NUMBER_OPD_OVERSCAN_SAMPLES];
   }

   return;
}

 
