/* BEGIN JPSS HEADER */
/*****************************************************************************
*
* Peraton, Inc.  All rights reserved.
*
* GOVERNMENT RIGHTS:
* Per contract 1332KP23CNEEJ0001, CAR 1352.227-70 applies giving full
* copyrights to data, including reports and other copyrighted materials
* to the United States Government.
*
* THIRD PARTY RIGHTS:
* Note: Third party entities may own copyrights in this work.
*
* EXPORT CONTROL:
* Unless otherwise specified beneath this header, this item does not contain
* Technology or Technical Data controlled under either the U.S. International
* Traffic in Arms Regulations or the U.S. Export Administration Regulations.
*
* CLASSIFICATION: B
* Refer to Software Standards and Practices Manual (SSPM) for coding and
* documentation standards.
*
*****************************************************************************/
/* END JPSS HEADER */

/*****************************************************************************
*
*  NAME: ScienceDataProcessor.h
*
*  DESCRIPTION:   See description preceeding declaration below
*
*****************************************************************************/

/***********************************************************************
*  HISTORY:
*  
* DATE        PR#      AUTHOR            Build    DESCRIPTION
* ---------   ---      ------            -----    -----------
* 25Apr2005            Byan Henderson      1.3    Initial Version
*                      Dan Elliott
*                      Tim Hahn
* 01Mar2006            Dan Elliott         1.4    Fringe Count Error
*                      Tim Hahn                   Follow On Drop
* 02Aug2006  011858    Dan Elliott         1.4    Modified ITT header with
*                                                 permission from Steven G.
*                                                 Minnie from ITT Space
*                                                 Systems, LLC
* 12Sep2006   12216    Tim Hahn            1.4    Combine Correction Matrix and
*                                                 Engineering Packet items
* 19Jun2007            Jeff DeLotelle      1.5    Updated copyright and govern-
*                                                 ment rights info in prologue.
* 29Jun2007            D.Elliott           1.5    Deep Space symmetry QF 
*                                                 addition.
* 07Jan2008   15748    Bryan Henderson    1.5.x.1 ISTN_CRIS_SDR_NGST_1.1
*                                                 drop implementation
* 04Feb2008   15748    Kelly Boswell      1.5.x.1 ISTN_CRIS_SDR_NGST_1.1
*                                                 crop implementation: removed
*                                                 unused public method 
*                                                 disgorgeCorrectionMatrices -
*                                                 implemented format updates
*                                                 for DFCB compliance
*                                                 ref. NP-EMD-2007.510.0027
*                                                 RevB
* 19MAR2008   17044    K. Boswell         1.5.x.1 Fixed dereferencing null 
*                                                 algorithm data 
*                                                 pointer from temporary 
*                                                 variable.
* 25Oct2008            L. Wang           ALG01342 Implem with ITT update 2.14
* 20Jan2009   19301    K. Boswell        1.5x1    ISTN_CRIS_SDR_UNIX_NGST_2.1
*                                                 drop implementation
*                                                 per TM NP-EMD.2008.510.0055
* 27May2009            L. Wang           ALG1443  Update for nonlinearity 
*                                                 correction 
* 03Feb2010   21718    K. Boswell        SensChar ISTN_CRIS_SDR_UNIX_NGST_2.2
*                                                 drop implementation
*                                                 per TM NP-EMD.2009.510.0046
* 10MAR2010   21464    K. Boswell        SensChar Changes in support of
*                                                 NP-EMD.2009.510.0046
*                                                 CrIS_Scan Baffle Temperature
*                                                 Correction
* 19MAY2010   23135    K. Boswell        SensChar ISTN_CRIS_SDR_UNIX_NGST_2.2.1
*                                                 drop implementation
*                                                 per TM
*                                                 NP-EMD.2010.510.0006
*                                                 and 
*                                                 ISTN_CRIS_SDR_UNIX_NGST_2.2.2
*                                                 per TM
*                                                 NP-EMD.2010.510.0006 Rev A
*                                                 and ECR A-275
* 31JUL2012   31562    T. Gardner        Mx6.3    Add set sciCalMissing flag
* 04JUN2012   30126    J. Schwartz       Blk1.5   Messaging/TaskData/TK updates
* 12Nov2012   31266    S. Wilderdyke     Blk1.5   Replaced methods that used
*                                                 BOOST for serialization
* 04MAY2013   34609    T. Gardner        Mx8.0    Added checkICT_Status and 
*                                                 setScienceCalibrationRecord_
*                                                 ICT_Status methods
* 25JUN2014            Y. Chen/Y.Han     Mx8.5    Adding different calibration 
*                                                 apporaches for full and normal 
*                                                 SDRs 
* 26AUG2014            Y. Chen/Y.Han     Mx8.5    Adding function to access 
*                                                 calibrationMatrix for NEDN
* 14OCT2014  042555    SEClark         Blk2.0.0.4 CrIS SSPM Compliance. 
* 03JAN2015   41268    J. Eberle         Blk2.0   Header Changes
* 11MAY2015   48586    T. Gardner        Blk2.0   Integrate TR/FR updates into
*                                                 block 2.0
* 15May2015            L. Wang        V2.0_beta   Fringe count error correction              
*                                                 implementation                             
* 01JUL2015            L. Wang        V2.0_beta   Updates for FCE handling                   
* 02JUL2015            Y. Chen        V2.0_beta   DR7982 change maxLunarRadiance
*                                                 to an array and check all 
*                                                 bands for lunar intrusion    
* 20AUG2015            Y. Chen        V2.0_beta   Add variable 
*                                                 usePostFilterOrCosineFilter   
* 04APR2016            Y. Chen           Blk2.0   Integrate TR/FR updates into
*                                                 block 2.0 
* 6May2015             Y. Chen           Blk2.0   Adding variable
*                                                 dataPointsTruncatedInterferogram    
* 27MAY2016            Y. Han            Blk2.0   Add functionality 
*                                                 for apodizing interferogram
* 26JUL2017            Y. Chen           Blk2.0   Add const DEFAULT_LASER_WAVELENGTH
* 06SEP2017            G. Pachl          Blk2.1   CCRs-17-3543, 3541.
* 13NOV2017   64232    G. Pachl          Blk2.1   Rename of engPktOutputTimeStep
*                                                 to spare in pct.
* 31AUG2017   63167    J. Gibbs          Blk2.1   fixed compile warnings
* 11APR2018            Y. Chen           Blk2.1   DR8631 add min and max
*                                                 frequency for lunar intrusion
*                                                 detection
* 17JUL2018   66227    G. Pachl          Blk2.1   Merge CCR 3922 Lunar intrusion
* 17APR2019            Y. Chen           Blk2.1   CCR 4469 Implemented Polarization
*                                                 Correction
* 13JUN2019            Pachl             Blk2.1   Baselined CCR 4469
* 19JAN2021            Pachl             Blk2.3   Made saveSerializedCorrectionMatrix public.
***********************************************************************/

/*******************************************************************************
* This document contains information proprietary and confidential to ITT Space 
* Systems, LLC ("ITT"), or a third party for whom ITT may have a legal 
* obligation to protect such information from unauthorized disclosure, use or
* duplication. Any disclosure, use, duplication or diversion contrary
* to U.S. Law of this document or of any of the information contained herein 
* for other than the
* specific purpose for which it was disclosed is expressly prohibited, 
* except as ITT has otherwise
* agreed in writing or as the U.S. Government has authorized. All copies of 
* this document are the
* sole property of ITT and will be returned promptly upon request
******************************************************************************/

#ifndef _INC_SCIENCEDATAPROCESSOR_3F86B3F60212_INCLUDED
#define _INC_SCIENCEDATAPROCESSOR_3F86B3F60212_INCLUDED

#include <Band.h>
#include <BoostLibrary.h>
#include <CalibratedSpectra.h>
#include <CalibrationParam.h>
#include <CorrectionMatrix.h>
#include <CrIS_InstrumentCharacteristics.h>
#include <CrIS_SDR.h>
#include <CrIS_TestEnvironmentParameters.h>
#include <EngineeringCalibrationRecord.h>
#include <EventLogEngine.h>
#include <Interferogram.h>
#include <LibraryReplacements.h>
#include <map>
#include <Octets.h>
#include <ProSdrCrisStruct.h>
#include <ScienceCalibrationRecord.h>
#include <SDR_AlgorithmnParameter.h>
#include <SDR_GeneralParameters.h>
#include <Spectra.h>
#include <SpectraManager.h>
#include <SystemTimeUtilities.h>
#include <TargetTempCalibrationRecord.h>
#include <TelemetryProcessor.h>
#include <TelemetryStateConverter.h>
#include <TimeRange.h>
#include <UncalibratedSpectra.h>
#include <WinDef.h>
#include <ProSdrCrisGbl.h>
#include <CrISDataStream.h>
#include <ILSParametersRecord.h>
#include <SpectralResDefs.h>

class ProSdrCris;

class ScienceDataProcessor 
{

   static const Int32 MATRIX_COL_SIZE = 1;

 public:

   static const Float64 DEFAULT_LASER_WAVELENGTH;
   static Float64 radianceModelOffset[MAX_FOV][Band::TOTAL_BANDS];
   static void configureRadianceModelOffset(SceneElement theFOV, 
                                            Band::Wavelength theBand, 
                                            Float64 newOffset)
   {
      radianceModelOffset[theFOV][theBand] = newOffset;
   } 

   enum CalibrationTypes 
      {
         Engineering,
         Science,
         MAX_TYPES
      };
   ScienceDataProcessor();
   virtual ~ScienceDataProcessor();
   void cleanup();
   void refreshTime(CCSDSFormat* theTelemetryFormat);
   void setWindowSize(Int32 newSize);
   void saveSerializedCorrectionMatrix();
   inline Int32  getWindowSize() const  { return windowSize; }
   inline Int32  getReferenceWindowSize() const 
      { return processingPeriod.getReferenceWindowSize();}
   inline Int32  getCalibrationWindowSize() const     
      { return processingPeriod.getCalibrationWindowSize();}
   inline Int32  getValidWindowSize() const     
      { return CalibrationParameter.validWindowSize;}

   inline const BOOST::matrix<Float64>&  getCalibrationMatrix
                    (SceneElement theFOV, Band::Wavelength theBand) const
      {
         return calibrationMatrix[theFOV][theBand];
      }
   inline void setCalibrationMatrix(SceneElement theFOV, 
                                    Band::Wavelength theBand, UInt32 rowIndx, 
                                    UInt32 colIndx, Float64 data)
      {
         calibrationMatrix[theFOV][theBand](rowIndx, colIndx)  = data;
      }

   inline Float64 getIctSpectraStab(Int32 fov, Int32 band, Int32 direction) 
      const { return processingPeriod.getIctSpectraStab(fov, band, direction);}; 
   inline Float64 getDsSpectraStab(Int32 fov, Int32 band, Int32 direction) const 
      { return processingPeriod.getDsSpectraStab(fov, band, direction);};
   inline UInt32 getMeasuredSpectraValidSize(SceneElement fov, 
                                            Band::Wavelength band, 
                                            FieldOfRegard scene, 
                                            SweepDirection psd) 
      { return processingPeriod.getMeasuredSpectraValidSize(fov,band,scene,psd);}

   inline Spectra* getMeasuredSpectra(SceneElement fov, Band::Wavelength band, 
                                      FieldOfRegard scene, SweepDirection psd, 
                                      Int32 index) 
      { return processingPeriod.getMeasuredSpectra(index,fov,band,scene,psd);}
   inline const Spectra* getReferenceSpectra(
                         SpectraManager::ReferenceSpectra spectraType, 
                         SceneElement fov, Band::Wavelength band, 
                         SweepDirection psd) 
     { return processingPeriod.getReferenceSpectra(spectraType,fov,band,psd);}
   inline const Spectra* getRadianceSpectra(
                         SpectraManager::RadianceSpectra spectraType, 
                         SceneElement fov, Band::Wavelength band)
   { 
      return processingPeriod.getRadianceSpectra(spectraType,fov,band);
   }
   inline const std::string getCorrectionFilename(Int32 temporalItem, 
                                             SceneElement fov, 
                                             Band::Wavelength band) 
      { return processingPeriod.getCorrectionFilename(temporalItem, fov, band);}
   inline Int32 getSpectraCount(SceneElement fov, Band::Wavelength band, 
                                FieldOfRegard scene, SweepDirection psd) const;
   Spectra* flushWindow(SceneElement fov, Band::Wavelength band, 
                        FieldOfRegard scene, SweepDirection psd);

   /**
    * Process the most recent Interferometer packet into Spectra and associated
    * reference/calibration data.
    *
    * @param packetType         Application Packet ID
    * @param missingPacket      true if the current packet is a placeholder for
    *                           a missing packet
    * @param theTelemetryFormat Handle to the "telemetry" formatter
    * @param cfgParmsPtr        Handle to the algorithm config parameters
    * @param algDataPtr         (Optional) Handle to the algorithm data struct
    * @param scanIdx            (Optional) scan number (relative to the start
    *                           of the current granule, not the sliding window
    * @return Pointer to a spectra object, if enough scans have been processed
    *         push data off the sliding window.
    * @throws ProSdrCrisException
    */
   Spectra* processRawDataRecord(UInt16 packetType,
                                 bool missingPacket,
                                 CCSDSFormat* theTelemetryFormat,
                                 CrisSdrCoefficients* cfgParmsPtr,
                                 CrisSdrAlgDataType* algDataPtr = 0,
                                 UInt32 scanIdx = -1);

   inline BOOST::vector  < Float64>& get_forDS_reals(UInt32 scan, UInt32 fov, 
                                                     UInt32 band) 
      { return forDS_reals[scan][fov][band];}
   inline BOOST::vector  < Float64>& get_forDS_imags(UInt32 scan, UInt32 fov, 
                                                     UInt32 band) 
      { return forDS_imags[scan][fov][band];}
   inline BOOST::vector  < Float64>& get_revDS_reals(UInt32 scan, UInt32 fov, 
                                                     UInt32 band) 
      { return revDS_reals[scan][fov][band];} 
   inline BOOST::vector  < Float64>& get_revDS_imags(UInt32 scan, UInt32 fov, 
                                                     UInt32 band) 
      { return revDS_imags[scan][fov][band];}

   inline void set_forDS_reals(
           UInt32 scan, 
           UInt32 fov, 
           UInt32 band, 
           BOOST::vector < Float64>& interferogram_forDS_reals)
   {
       forDS_reals[scan][fov][band] = interferogram_forDS_reals;
   }
   inline void set_forDS_imags(
           UInt32 scan, 
           UInt32 fov, 
           UInt32 band, 
           BOOST::vector < Float64>& interferogram_forDS_imags) 
   {
       forDS_imags[scan][fov][band] = interferogram_forDS_imags;
   }
   inline void set_revDS_reals(
           UInt32 scan, 
           UInt32 fov, 
           UInt32 band, 
           BOOST::vector < Float64>& interferogram_revDS_reals) 
   {
       revDS_reals[scan][fov][band] = interferogram_revDS_reals;
   }
   inline void set_revDS_imags(
           UInt32 scan, 
           UInt32 fov, 
           UInt32 band, 
           BOOST::vector < Float64>& interferogram_revDS_imags) 
   {
       revDS_imags[scan][fov][band] = interferogram_revDS_imags;
   }

   bool processEnvironmentalData(const SYSTEMTIME& theSpectraFrameTime);
   Int32 setCalibrationPeriod(Int32 newWindowSize);
   void buildPolarizationCurve(Spectra* newSpectra, FieldOfRegard theScene, 
                               Band::Wavelength theBand);
   Spectra* calculateHotRadiance(SceneElement theFOV, Band::Wavelength theBand);
   Spectra* calculateColdRadiance(SceneElement theFOV, Band::Wavelength theBand);
   void calculateCalibrationMatrix();
   inline const CorrectionMatrix& getCorrectionTable(
      SceneElement theFov, Band::Wavelength theBand) const 
         {return correctionTable[theFov][theBand];}
   inline void 
      setAllowCalibrationTargetDataMissing(
         Int32 theAllowCalibrationTargetDataMissing) 
            {CalibrationParameter.allowCalibrationTargetDataMissing = 
               theAllowCalibrationTargetDataMissing;}
   inline void 
      setAllowEngineeringDataPacketsMissing(
         Int32 theAllowEngineeringDataPacketsMissing) 
            {CalibrationParameter.allowEngineeringDataPacketsMissing = 
               theAllowEngineeringDataPacketsMissing;}
   inline void 
      setAllowSpaceTargetTemperatureDataMissing(
         Int32 theAllowSpaceTargetTemperatureDataMissing) 
            {CalibrationParameter.allowSpaceTargetTemperatureDataMissing = 
               theAllowSpaceTargetTemperatureDataMissing;}
   inline void 
      setDisableTimeStampBasedMovingWindow(
         Int32 theDisableTimeStampBasedMovingWindow);
   inline void setPerformRadiometricCalibration(
      Int32 thePerformRadiometricCalibration) 
         {CalibrationParameter.performRadiometricCalibration = 
            thePerformRadiometricCalibration;}
   inline void setSkipIctDsPhaseSynchronization(
      Int32 theSkipIctDsPhaseSynchronization) 
         {CalibrationParameter.skipIctDsPhaseSynchronization = 
            theSkipIctDsPhaseSynchronization;}
   inline void setUseDeepSpaceRadiance(Int32 theUseDeepSpaceRadiance) 
      {CalibrationParameter.useDeepSpaceRadiance = theUseDeepSpaceRadiance;}
   inline void setUseIctEnvironmentalCorrectionModel(
      Int32 theUseIctEnvironmentalCorrectionModel) 
         {CalibrationParameter.useIctEnvironmentalCorrectionModel = 
            theUseIctEnvironmentalCorrectionModel;}
   inline void setUseWavenumberDependentDsEmissivity(
      Int32 theUseWavenumberDependentDsEmissivity) 
         {CalibrationParameter.useWavenumberDependentDsEmissivity = 
            theUseWavenumberDependentDsEmissivity;}
   inline void setDsTemperatureOrigin(
      DsTemperatureOriginSource theDsTemperatureOrigin);
   inline const DsTemperatureOriginSource& getDsTemperatureOrigin() 
      { return CalibrationParameter.dsTemperatureOrigin;};
   inline void 
      setIctEmissivityOrigin(IctEmissivityOriginSource theIctEmissivityOrigin) 
         {CalibrationParameter.ictEmissivityOrigin = theIctEmissivityOrigin;}
   inline void 
      setUseWavenumberDependentIctEmissivity(
         Int32 theUseWavenumberDependentIctEmissivity) 
            {CalibrationParameter.useWavenumberDependentIctEmissivity = 
               theUseWavenumberDependentIctEmissivity;}
   inline void 
      setCalibrationTargetDataValidityDuration(
         Float64 theCalibrationTargetDataValidityDuration); 
   inline void 
      setCalibrationTargetDataValidityDurationTolerance(
         Float64 theCalibrationTargetDataValidityDurationTolerance);
   inline void 
      setElapsedTimeForValidScienceTlmData(
         Float64 theElapsedTimeForValidScienceTlmData);
   inline void 
      setElapsedTimeForValidSpaceTargetTemperature(
         Float64 theElapsedTimeForValidSpaceTargetTemperature);
   inline void 
      setSpaceTargetTemperatureTimeDifferenceTolerance(
         Float64 theSpaceTargetTemperatureTimeDifferenceTolerance);
   inline void 
      setScienceTlmTimeDifferenceTolerance(
         Float64 theScienceTlmTimeDifferenceTolerance);
   inline void 
      setMovingAverageWindowSize(Int32 theMovingAverageWindowSize) 
         {CalibrationParameter.movingAverageWindowSize = 
            theMovingAverageWindowSize;}
   inline void 
      setInstrumentTemperatureOrigin(
         InstrumentTemperatureOriginSource theInstrumentTemperatureOrigin);
   inline void 
      setAllowScienceTlmDataMissing(Int32 theAllowScienceTlmDataMissing) 
         {CalibrationParameter.allowScienceTlmDataMissing = 
            theAllowScienceTlmDataMissing;}
   inline void 
      setMaximumNumberOfFceTriesDuringIctDsSynchronization(
         Int32 theMaximumNumberOfFceTriesDuringIctDsSynchronization) 
            {CalibrationParameter.
               maximumNumberOfFceTriesDuringIctDsSynchronization = 
                  theMaximumNumberOfFceTriesDuringIctDsSynchronization;}
   inline void 
      setMaximumNumberOfIctDsSynchronizationTries(
         Int32 theMaximumNumberOfIctDsSynchronizationTries) 
            {CalibrationParameter.maximumNumberOfIctDsSynchronizationTries = 
               theMaximumNumberOfIctDsSynchronizationTries;}
   inline void 
      setApplyPolarizationCorrections(Int32 theApplyPolarizationCorrections);
   inline void 
      setApplyPostCalibrationFilterMatrixCorrection(
         Int32 theApplyPostCalibrationFilterMatrixCorrection);
   inline void 
      setApplyIlsFovEffectsCorrection(Int32 theApplyIlsFovEffectsCorrection);
   inline void 
      setApplyIlsResidualEffectCorrection(
         Int32 theApplyIlsResidualEffectCorrection);
   inline void setApplyResamplingMatrix(Int32 theApplyResamplingMatrix);
   inline void setDisableLaserMonitoring(Int32 theDisableLaserMonitoring);
   inline void 
      setPerformFringeCountErrorHandling(
         Int32 thePerformFringeCountErrorHandling);
   inline void 
      setPerformPolarizationCorrection(Int32 thePerformPolarizationCorrection);
   inline void 
      setPerformSpectralAndSpatialCorrection(
         Int32 thePerformSpectralAndSpatialCorrection);
   inline void setUseSavedMatrices(Int32 theUseSavedMatrices);
   inline void setUsePostFilterOrCosineFilter(
                  Int32 theUsePostFilterOrCosineFilter);
   inline void setApodizationType(ApodizationType theApodizationType);
   inline void setCalibrationOrder(CalibrationOrder theCalibrationOrder);
   inline void setSincFlag(CalibrationOrder theCalibrationOrder);
   inline void 
      setLaserDiodeWavelengthOrigin(
         LaserWavelengthSource theLaserDiodeWavelengthOrigin);
   inline void 
      setImpulseNoiseCountThreshold(UInt32 theImpulseNoiseCountThreshold);
   inline void 
      setEdrNumberOfPoints(
         Band::Wavelength theBand, UInt32 theEdrNumberOfPoints);
   inline void 
      setEdrDeltaSigma(
         Band::Wavelength theBand, Float64 theEdrDeltaSigma);
   inline void 
      setEdrMinimumWavenumber(
         Band::Wavelength theBand, Float64 theEdrMinimumWavenumber);
   inline void 
      setEdrMaximumWavenumber(
         Band::Wavelength theBand, Float64 theEdrMaximumWavenumber);
   inline void 
      setUserClippingSelection(bool clip) 
         { CorrectionParameter.userSelectedClipping = clip;}
   inline void 
      setBenchMeanIctEmissivity(
         Band::Wavelength theBand, Float64 theBenchMeanIctEmissivity) 
            {InstrumentCharacteristic.benchMeanIctEmissivity[theBand] = 
               theBenchMeanIctEmissivity;}
   inline void 
      setChamberMeanIctEmissivity(
         Band::Wavelength theBand, Float64 theChamberMeanIctEmissivity) 
            {InstrumentCharacteristic.chamberMeanIctEmissivity[theBand] = 
               theChamberMeanIctEmissivity;}
   inline void 
      setDefaultLaserWavelength(Float64 newWavelength) 
         { InstrumentCharacteristic.defaultLaserWavelength = newWavelength;}
   inline void 
      setHotReferenceFieldOfRegard(FieldOfRegard newFOR);
   inline void 
      setColdReferenceFieldOfRegard(FieldOfRegard newFOR);
   inline void 
      setNumberFOR(Int32 theNumberFOR) 
         {InstrumentCharacteristic.numberFOR = theNumberFOR;}
   inline void 
      setNumberFOV(Int32 theNumberFOV) 
         {InstrumentCharacteristic.numberFOV = theNumberFOV;}
   inline void 
      setNumberSpectralBands(Int32 theNumberSpectralBands) 
         {InstrumentCharacteristic.numberSpectralBands = 
            theNumberSpectralBands;}
   inline void 
      setNumberSamplesPerLaserWavelength(
         Int32 theNumberSamplesPerLaserWavelength) 
            {InstrumentCharacteristic.numberSamplesPerLaserWavelength = 
               theNumberSamplesPerLaserWavelength;}
   inline void 
      setSpaceTargetTemperatureDriftLimit(
         Float64 theSpaceTargetTemperatureDriftLimit);
   inline void 
      setReverseSweepDirectionIdentifier(
         Int32 theReverseSweepDirectionIdentifier) 
            {InstrumentCharacteristic.reverseSweepDirectionIdentifier = 
               theReverseSweepDirectionIdentifier;}
   inline void 
      setForwardSweepDirectionIdentifier(
         Int32 theForwardSweepDirectionIdentifier) 
            {InstrumentCharacteristic.forwardSweepDirectionIdentifier = 
               theForwardSweepDirectionIdentifier;}
   inline void 
      setDecimationFactor(
         Band::Wavelength theBand, UInt32 theDecimationFactor);
   inline void 
      setDataPointsDecimatedInterferogram(
         Band::Wavelength theBand, UInt32 theDataPointsDecimatedInterferogram);
   inline void 
      setDataPointsUndecimatedInterferogram(
         Band::Wavelength theBand, 
         UInt32 theDataPointsUndecimatedInterferogram);
   inline void 
      setLWDsEffectiveEmissivity(
         const BOOST::vector<Float64>& theLWDsEffectiveEmissivity)
      {
         TestEnvironmentParameter.
            dsEffectiveEmissivity[Band::LONGWAVE].
               resize(theLWDsEffectiveEmissivity.size());
         TestEnvironmentParameter.dsEffectiveEmissivity[Band::LONGWAVE] =
            theLWDsEffectiveEmissivity;
      }
   inline void setMWDsEffectiveEmissivity(
      const BOOST::vector<Float64>& theMWDsEffectiveEmissivity)
      {
         TestEnvironmentParameter.
            dsEffectiveEmissivity[Band::MIDWAVE].
               resize(theMWDsEffectiveEmissivity.size());
         TestEnvironmentParameter.dsEffectiveEmissivity[Band::MIDWAVE] =
            theMWDsEffectiveEmissivity;
      }
   inline void setSWDsEffectiveEmissivity(
      const BOOST::vector<Float64>& theSWDsEffectiveEmissivity)
      {
         TestEnvironmentParameter.
            dsEffectiveEmissivity[Band::SHORTWAVE].
               resize(theSWDsEffectiveEmissivity.size());
         TestEnvironmentParameter.dsEffectiveEmissivity[Band::SHORTWAVE] =
            theSWDsEffectiveEmissivity;
      }
   inline void setBeamsplitterTempBench(Float64 theBeamsplitterTempBench);
   inline void setBeamsplitterTempChamber(Float64 theBeamsplitterTempChamber);
   inline void setDsTempBench(Float64 theDsTempBench);
   inline void setDsTempChamber(Float64 theDsTempChamber);
   inline void setIctTempBench(Float64 theIctTempBench);
   inline void setIctTempChamber(Float64 theIctTempChamber);
   inline void setMeanDsEmissivityBench(Float64 theMeanDsEmissivityBench);
   inline void setMeanDsEmissivityChamber(Float64 theMeanDsEmissivityChamber);
   inline void setOmaTempBench(Float64 theOmaTempBench);
   inline void setOmaTempChamber(Float64 theOmaTempChamber);
   inline void 
      setScanBaffleTempOffset(
         const BOOST::vector<Float32> & theScanBaffleTempOffset)
   {
      TestEnvironmentParameter.scanBaffleTempOffset = theScanBaffleTempOffset;
   }
   inline void setDurationOfOrbit(Int32 theDurationOfOrbit) 
      {TestEnvironmentParameter.durationOfOrbit = theDurationOfOrbit;}
   inline void setScanBaffleTempBench(Float64 theScanBaffleTempBench);
   inline void setScanBaffleTempChamber(Float64 theScanBaffleTempChamber);
   inline void setColdBeamsplitterViewFactor(Float64 theViewFactor) 
      {TestEnvironmentParameter.coldBeamSplitterViewFactor = theViewFactor;}
   inline void setWarmBeamsplitterViewFactor(Float64 theViewFactor) 
      {TestEnvironmentParameter.warmBeamSplitterViewFactor = theViewFactor;}
   inline void setScanBaffleViewFactor(Float64 theViewFactor) 
      {TestEnvironmentParameter.scanBaffleViewFactor = theViewFactor;}
   inline void setScanBaffleEmissivity(
      Band::Wavelength theBand, Float64 emissivity) 
         {TestEnvironmentParameter.scanBaffleEmissivity[theBand] = emissivity;}
   inline void setOmaViewFactor(Float64 theViewFactor) 
      {TestEnvironmentParameter.omaViewFactor = theViewFactor;}
   inline void setOmaEmissivity(Band::Wavelength theBand, Float64 emissivity) 
      {TestEnvironmentParameter.omaEmissivity[theBand] = emissivity;}
   inline void setIctBaffleViewFactor(Float64 theViewFactor) 
      {TestEnvironmentParameter.ictBaffleViewFactor = theViewFactor;}
   inline void setIctBaffleEmissivity(
      Band::Wavelength theBand, Float64 emissivity) 
         {TestEnvironmentParameter.ictBaffleEmissivity[theBand] = emissivity;}
   inline void setSpaceViewFactor(Float64 theViewFactor) 
      {TestEnvironmentParameter.spaceViewFactor = theViewFactor;}
   inline void setEarthTargetOverride(bool override) 
      {TestEnvironmentParameter.earthTargetOverride = override;}
   inline void setEarthTargetTempBench(Float64 temp) 
      {TestEnvironmentParameter.ssmTargetTempBench = temp + CONVERT_C_TO_K;}
   inline void setEarthTargetTempChamber(Float64 temp) 
      {TestEnvironmentParameter.ssmTargetTempChamber = temp;}
   inline void setEarthTargetEmissivity(
      Band::Wavelength theBand, Float64 emissivity) 
         {TestEnvironmentParameter.ssmTargetEmissivity[theBand] = emissivity;}
   inline void setIctPrt1Bias(Float64 theIctPrt1Bias);
   inline void setIctPrt2Bias(Float64 theIctPrt2Bias);
   inline void 
      setPostCalibrationA1(
         Band::Wavelength theBand, Int32 thePostCalibrationA1);
   inline void 
      setPostCalibrationA2(
         Band::Wavelength theBand, Float64 thePostCalibrationA2);
   inline void 
      setPostCalibrationA3(
         Band::Wavelength theBand, Int32 thePostCalibrationA3);
   inline void 
      setPostCalibrationA4(
         Band::Wavelength theBand, Float64 thePostCalibrationA4);
   inline void 
      setPostCalibrationK(Band::Wavelength theBand, Int32 thePostCalibrationK);
   inline void 
      setPostCalibrationK0(
         Band::Wavelength theBand, Int32 thePostCalibrationK0);
   inline void 
      setPostCalibrationK1(
         Band::Wavelength theBand, Int32 thePostCalibrationK1);
   inline void 
      setFceParamAmpThreshRejectLimit(
         Spectra::FCE_LIMIT type, 
         Band::Wavelength theBand, 
         Float64 theFceParamAmpThreshRejectLimit);
   inline void 
      setFceParamDimensionThresholdLimit(
         Band::Wavelength theBand, 
         Float64 theFceParamDimensionThresholdLimit);
   inline void 
      setFceParamFractionalFceThresholdLimit(
         Band::Wavelength theBand, 
         Float64 theFceParamFractionalFceThresholdLimit);
   inline void 
     setFceParamGoodLinearFittingThreshLimit(
        Band::Wavelength theBand, 
        Float64 theFceParamGoodLinearFittingThreshLimit);
   inline void 
      setFceParamMaxIndex(Band::Wavelength theBand, Int32 theFceParamMaxIndex);
   inline void 
      setFceParamMaxFceThreshLimit(
         Band::Wavelength theBand, Float64 theFceParamMaxFceThreshLimit);
   inline void 
      setFceParamMinIndex(Band::Wavelength theBand, Int32 theFceParamMinIndex);
   inline void 
      setMaximumFractionRejections(
         Float64 theMaximumFractionRejections) 
            {AlgorithmParameter.maximumFractionRejections = 
               theMaximumFractionRejections;}
   inline void 
      setNumberOpdOverscanSamples(
         Int32 theNumberOpdOverscanSamples) 
            {AlgorithmParameter.numberOpdOverscanSamples = 
               theNumberOpdOverscanSamples;}
   inline void 
      setComputedWavelengthRejectionThreshold(
         Int32 theComputedWavelengthRejectionThreshold) 
   {
     AlgorithmParameter.computedWavelengthRejectionThreshold = 
        theComputedWavelengthRejectionThreshold;
     theEngineeringCalibrationRecord.
        setComputedWavelengthRejectionThreshold(
           theComputedWavelengthRejectionThreshold);
   }
   inline void 
      setLaserWavelengthDriftTolerance(Float64 theLaserWavelengthDriftTolerance)
         {AlgorithmParameter.laserWavelengthDriftTolerance = 
            theLaserWavelengthDriftTolerance;}
   inline void 
      setFceParamDefaultDetectorBand(Int32 theFceParamDefaultDetectorBand);
   inline void 
      setFceParamDefaultDetectorFOV(Int32 theFceParamDefaultDetectorFOV);
   inline void 
      setPolarizationCorrectionFitOrder(Int32 thePolarizationCorrectionFitOrder)
         {AlgorithmParameter.polarizationCorrectionFitOrder = 
            thePolarizationCorrectionFitOrder;}
   inline void setSurfaceEmissivityCoeff(
           const SurfaceEmissivityCoeffIdxEnum c,
           const Float64 val)
   {
      if (c >= 0 && c < SURF_EMISS_COEFF_CMAX)
      {
         InstrumentCharacteristic.surfaceEmissivityCoeff[c] = val;
      }
      else
      {
         // Should throw an exception here.
      }
   }

   inline void setSAExpansionFactor(Band::Wavelength theBand, 
                                    Float64 saExpansionFactor)
      {CorrectionParameter.saExpansionFactor[theBand] = saExpansionFactor;} 
   inline void setFoldIndexOffset(Band::Wavelength theBand, 
                                  Int32 foldIndexOffset)
      {InstrumentCharacteristic.foldIndexOffset[theBand] = foldIndexOffset;} 
   inline void setDataPointsTruncatedInterferogram(Band::Wavelength theBand, 
                                  Int32 dataPointsTruncatedInterferogram)
      {InstrumentCharacteristic.dataPointsTruncatedInterferogram[theBand] = 
                                dataPointsTruncatedInterferogram;} 
   inline void setFirAccumulatorStartBit(
      Band::Wavelength theBand, Int32 firAccumStartBit)
         {InstrumentCharacteristic.
            firAccumulatorStartBit[theBand] = firAccumStartBit;} 
   inline void setFirFilterScale(Band::Wavelength theBand, Float64 scaleFactor)
       {InstrumentCharacteristic.
           LinearityCorrectionParameter.firFilterScale[theBand] = scaleFactor;}
   inline void setPassBandStart(Band::Wavelength theBand, Float64 passBandStart)
       {InstrumentCharacteristic.passBandStart[theBand] = passBandStart;}
   inline void setPassBandStop(Band::Wavelength theBand, Float64 passBandStop)
       {InstrumentCharacteristic.passBandStop[theBand] = passBandStop;}
   inline void setEngCalInstrumentCharacteristics(
               const CrIS_InstrumentCharacteristics& newValue);
   inline void setFirFilterResponse(
               Band::Wavelength band, 
               const BOOST::vector<std::complex<Float64> > & response);
   inline void setLinearityCorrectionParameter_a2(
               Band::Wavelength band, BOOST::vector<Float64> a2Value);
   inline void setLinearityCorrectionParameter_Vinst(
               Band::Wavelength band, BOOST::vector<Float64> vInstValue);
   inline void setLinearityCorrectionParameter_ModEff(
               Band::Wavelength band, BOOST::vector<Float64> modEff);
   inline void setPerformLinearityCorrectionControl(
               Band::Wavelength theBand, bool value);
   inline void setLinearityModeSelectControl(
               Band::Wavelength band, 
               LinearityCorrectionControlOriginSource theSource);
   inline void setLaserDiodeWavelengthOffsetMW(Float64 offset);
   inline void setLaserDiodeWavelengthOffsetSW(Float64 offset);
   inline void setEffGainSetting(
               Band::Wavelength theBand, 
               BOOST::vector<Int32> effectiveGain);
   inline void setEffectiveGainMap(
               Band::Wavelength theBand, 
               GAIN_MAP effectiveGainMap);
   inline Int32 getInvalidInterferogramCount(
                SceneElement fov, 
                Band::Wavelength band) const 
       {return invalidInterferogramCount[fov][band];}
   inline Int32 getInvalidSpectraCount(
                SceneElement fov, Band::Wavelength band) const 
       {return invalidSpectraCount[fov][band];}
   inline void resetCounters() 
       { ZeroMemory(invalidInterferogramCount, 
            sizeof(Int32) * MAX_FOV * Band::TOTAL_BANDS);
   ZeroMemory(invalidSpectraCount, 
                 sizeof(Int32) * MAX_FOV * Band::TOTAL_BANDS);}
   inline const SYSTEMTIME& getWindowTimeSpan(
                               SceneElement fov, 
                               Band::Wavelength band, 
                               FieldOfRegard scene, 
                               SweepDirection psd)  
       { return processingPeriod.getWindowTimeSpan(fov,band,scene,psd);}
   inline void setInstrumentLocation(InstrumentLocation theInstrumentLocation);
   inline const InstrumentLocation& getInstrumentLocation() 
       { return GeneralParameter.instrumentLocation;}
   void setSpectralCalibrationMethod(SpectralCalibrationMethod newMethod);
   inline void enableNEdN(bool state);
   inline void setBoxCarWidth(Int32 theWidth);

   bool buildPostCalibrationTable(Band::Wavelength theBand);
   bool buildPostCalibrationTable_raisedCos(Band::Wavelength theBand);
   bool buildResamplingTable(Band::Wavelength theBand);
   bool buildResamplingTable(Float64 lowestWavenumberFrom, Float64 deltaSigmaFrom,  
                             Float64 lowestWavenumberTo,   Float64 deltaSigmaTo,   UInt32 bigNto,
                             BOOST::matrix<Float64> &rMatrix);
   bool buildUserApodizationTable(Band::Wavelength theBand);

   Float64 computeMaxPathDifference(Band::Wavelength theBand, UInt32 interferogramSize);
   
   void calculateWavenumberBinSpacing(Band::Wavelength theBand);
   void updateCorrectionMatrixParameters();
   void checkICT_Status(Int32 scanIdx, CrisSdrAlgDataType* algDataPtr = 0);
   inline void setScienceCalibrationRecord_ICT_Status(
                  Int32 scanIdx, CrisSdrAlgDataType* algDataPtr = 0)
   {
      theScienceCalibrationRecord.updateICT_Status(algDataPtr, scanIdx);
   }
   inline void setTableSize(SceneElement theFOV, 
                            Band::Wavelength theBand, 
                            UInt32 newSize) 
       { correctionTable[theFOV][theBand].setSize(newSize);}
   inline const EngineeringCalibrationRecord& 
     getEngineeringCalibrationRecord() const 
       {return theEngineeringCalibrationRecord;}
   inline const TargetTempCalibrationRecord& 
     getTargetTempCalibrationRecord() const 
       {return theTargetTempCalibrationRecord;}
   inline const ScienceCalibrationRecord& getScienceCalibrationRecord() const 
       {return theScienceCalibrationRecord;}
   inline UInt32 getBufferedSpectraCount(bool includeReferenceCount = true) 
       { return processingPeriod.
                   getBufferedSpectraCount(includeReferenceCount);}
   void declareWindowSize(UInt32 referenceSize, 
                          UInt32 targetTempSize, 
                          UInt32 calibrationSize);
   bool loadCMOs();
   bool loadEngPkt();
   void saveSerializedEngPkt();
   Int32 refreshCorrectionMatrix(bool isLoading = true);
   inline void assignCalibrationRecordIdentifiers(
     UInt32 newScienceCalibrationFrameType, 
     UInt32 newEngineeringCalibrationFrameType) 
       { SCIENCE_CALIBRATION_FRAME_TYPE = newScienceCalibrationFrameType;
   ENGINEERING_CALIBRATION_FRAME_TYPE = newEngineeringCalibrationFrameType;}
   inline UInt32 getScienceCalibrationRecordIdentifier() const 
       { return SCIENCE_CALIBRATION_FRAME_TYPE;}
   inline UInt32 getEngineeringCalibrationRecordIdentifier() const 
       { return ENGINEERING_CALIBRATION_FRAME_TYPE;}
   inline const Spectra* getPolarizationSpectra(
     Int32 temporalItem, FieldOfRegard scene, Band::Wavelength band) 
       { return processingPeriod.
                   getPolarizationSpectra(temporalItem, scene, band);}
   inline const Spectra* getGeolocationSpectra(
     Int32 temporalItem, FieldOfRegard scene, SceneElement theFov) 
       { return processingPeriod.
                   getGeolocationSpectra(temporalItem, scene, theFov);}
   inline Int32 getPerformFringeCountErrorHandling() { 
         return  CorrectionParameter.performFringeCountErrorHandling; }

   Spectra* calculateGeometricCalibration(Spectra* newSpectra);
   inline void enableLunarIntrusionTracking(bool state, 
                  Band::Wavelength theBand, Float32 ratio, 
                  Float32 minFreq, Float32 maxFreq) 
   { 
     CalibrationParameter.trackLunarIntrusion = state;
     CalibrationParameter.maxLunarIntrusionRatio[theBand] = ratio;
     CalibrationParameter.minFreqMicroWindow[theBand] = minFreq;
     CalibrationParameter.maxFreqMicroWindow[theBand] = maxFreq;
     processingPeriod.setLunarIntrusionTracking(state, theBand, ratio, 
                                                minFreq, maxFreq);
   }
   inline CrisSdrAlgDataType* getDMS_Data_Inputs(){return DMS_Data_Inputs;}
   inline void setDMS_DATA_Inputs(CrisSdrAlgDataType* DMS_Data_Inputs_param)
     {this->DMS_Data_Inputs = DMS_Data_Inputs_param;}
   inline UInt32 getScanIdx(){return scanIdx_;}
   inline void setScanIdx(UInt32 a){
      if(a !=  scanIdx_){
         currentDSrev_p = false;
         currentDSfwd_p = false;
      }
      scanIdx_ = a;
      theScienceCalibrationRecord.setSciCalMissing(false);

      memset(earthScenefceDetectionComplete, false, sizeof(bool)*TOTAL_SCENES);

   }

   inline bool get_IlsCorrection_Loaded() {return ilsCorrection_Loaded;}
   inline void set_IlsCorrection_Loaded(bool a) { ilsCorrection_Loaded = a;}
   inline bool get_EngPkt_Loaded() {return engPkt_Loaded;}
   inline void set_EngPkt_Loaded(bool a) {engPkt_Loaded = a;}
   inline bool get_firstEngPkt() {return firstEngPkt;}
   inline void set_firstEngPkt(bool a) {firstEngPkt = a;}
 
   inline UInt16 getNumCMOBuilds() {return numCMOBuilds_;}
   inline void setNumCMOBuilds(UInt16 a) {numCMOBuilds_ = a;}
   inline UInt16 getNumEngBuilds() {return numEngBuilds_;}
   inline void setNumEngBuilds(UInt16 a) {numEngBuilds_ = a;}
   inline bool getTaskedSequentially() {return taskedSequentially_;}
   inline void setTaskedSequentially(bool a) {taskedSequentially_ = a;}
   inline void setScanWhichCausedCMORebuild(Int16 a)
      {scanWhichCausedCMORebuild_ = a;}
   inline Int16 getScanWhichCausedCMORebuild() 
      {return scanWhichCausedCMORebuild_;}
   inline void setScanWhichCausedEngRebuild(Int16 a)
      {scanWhichCausedEngRebuild_ = a;}
   inline Int16 getScanWhichCausedEngRebuild() 
      {return scanWhichCausedEngRebuild_;}

   inline void setReferenceWindowSize(Int32 theReferenceWindowSize)   
     { CalibrationParameter.referenceWindowSize = theReferenceWindowSize;}
   inline void setTargetTempWindowSize(Int32 theTargetTempWindowSize) 
     { CalibrationParameter.targetTempWindowSize = theTargetTempWindowSize;}
   inline void setCalibratedWindowSize(Int32 theCalibratedWindowSize) 
     { CalibrationParameter.calibratedWindowSize = theCalibratedWindowSize;}
   inline void setValidWindowSize(Int32 theValidWindowSize) 
     { CalibrationParameter.validWindowSize = theValidWindowSize;}

   inline void setSuppressBaffleProfile(bool mode) 
     { CalibrationParameter.suppressSsmBaffleProfile = mode;}
   inline const Spectra* getLastReferenceSpectra(
                            SpectraManager::ReferenceSpectra spectraType, 
                            SceneElement fov, 
                            Band::Wavelength band,
                            SweepDirection psd) 
     { return processingPeriod.
                 getLastReferenceSpectra(spectraType,fov,band,psd);}
   inline const SDR_CorrectionParam& getCorrectionParameter() 
     {return CorrectionParameter;}
   inline void setPlankConstantC1(Float64 newC1Constant) 
     { plankConstC1 = newC1Constant; }
   inline void setPlankConstantC2(Float64 newC2Constant) 
     { plankConstC2 = newC2Constant; }
   inline Float64 getPlankConstantC1() { return (Float64)plankConstC1; }
   inline Float64 getPlankConstantC2() { return (Float64)plankConstC2; }

   void syncWindows();
   void launchBatchReferenceDetect(SpectraManager::ReferenceSpectra spectraType,
                                   SweepDirection thePSD);

   inline void sendIlsInfoToEngCalRec(IlsOriginSource theSource, 
                                      ILS_Parameters theParams);
   inline const CrIS_InstrumentCharacteristics& getInstrumentCharacteristic() 
     { return InstrumentCharacteristic;}

   inline const CrIS_TestEnvironmentParameters& getTestEnvironmentParams() 
     { return TestEnvironmentParameter;}
   inline void setBaffleTemperatureOffset(BOOST::vector<Float32> & newProfile) 
     { theEngineeringCalibrationRecord.setEngCalBaffleTempOffset(newProfile); }
   void buildBaffleTemperatureOffset(
           const  BOOST::vector<Float32> & profileSubset, 
           Int32 orbitDuration);
   void buildFirFilterGainTable(Band::Wavelength requestedBand); 
   inline Spectra* getFirGain(Band::Wavelength band) 
     { return &FIR_GAIN[band]; }
   inline Int32 getSequenceCount(SceneElement fov, 
                                 Band::Wavelength band, 
                                 FieldOfRegard scene, 
                                 SweepDirection psd) 
     {return sequenceCount[fov][band][scene][psd];}
   inline void setAppShiftFactorFlag(bool shiftFactorFlag) 
     {appShiftFactorFlag = shiftFactorFlag;}
   inline void setShiftFactor(UInt32 fov, Float64 newShiftFactor)
     {shiftFactor[fov] = newShiftFactor;}
   inline void setMinNpCrossingTime(const SYSTEMTIME minTime)
   {
       minNpCrossingTime_ = minTime;
   }
   inline void setMaxNpCrossingTime(const SYSTEMTIME maxTime)
   {
       maxNpCrossingTime_ = maxTime;
   }
   inline void clearMsgThrottles()
   {
       orbitMsgThrottle_.clear();
   }
 
   inline void setFCE_esImagThreshold(Float64 esImagThreshold);
   inline void setFCE_esFceNumErrorThreshold(UInt32 esFceNumErrorThreshold);
   inline void setStartWavenumber(UInt32 startWavenumber);
   inline void setEndWavenumber(UInt32 endWavenumber);

   // polarization parameters                                                      
   inline void setPolarizationAlpha(                         
        Band::Wavelength theBand,                                                  
        const BOOST::vector<Float64>& thePolarizationAlpha)                        
   {                                                                               
      CorrectionParameter.polarizationAlpha[theBand].        
           resize(thePolarizationAlpha.size());                                                                    
      CorrectionParameter.polarizationAlpha[theBand] = thePolarizationAlpha;       
   }                                                                               
   inline void setDsPolarizationDelta(                       
        SceneElement fov, Float64 theDsPolarizationDelta )                         
   {                                                                               
      CorrectionParameter.dsPolarizationDelta[fov] = theDsPolarizationDelta;       
   }                                                                               
   inline void setIctPolarizationDelta(                      
        SceneElement fov, Float64 theIctPolarizationDelta )                        
   {                                                                               
      CorrectionParameter.ictPolarizationDelta[fov] = theIctPolarizationDelta;     
   }                                                                               
   inline void setESPolarizationDelta(                       
      SceneElement theFov,                                                         
      const BOOST::vector<Float64>& theESPolarizationDelta)                        
   {                                                                               
         CorrectionParameter.esPolarizationDelta[theFov].                          
           resize(theESPolarizationDelta.size());                                                                    
         CorrectionParameter.esPolarizationDelta[theFov]=                          
           theESPolarizationDelta;                                                 
   }                                                                               
   inline void setLWPolarizationReflTrans(                   
      SceneElement theFov,                                                         
      const BOOST::vector<Float64>& theLWPolarizationReflTrans)                    
   {                                                                               
         CorrectionParameter.polarizationReflTrans[Band::LONGWAVE][theFov].        
           resize(theLWPolarizationReflTrans.size());                                                                    
         CorrectionParameter.polarizationReflTrans[Band::LONGWAVE][theFov]=        
           theLWPolarizationReflTrans;                                             
   }                                                                               
   inline void setMWPolarizationReflTrans(                   
      SceneElement theFov,                                                         
      const BOOST::vector<Float64>& theMWPolarizationReflTrans)                    
   {                                                                               
         CorrectionParameter.polarizationReflTrans[Band::MIDWAVE][theFov].         
           resize(theMWPolarizationReflTrans.size());                                                                    
         CorrectionParameter.polarizationReflTrans[Band::MIDWAVE][theFov]=         
           theMWPolarizationReflTrans;                                             
   }                                                                               
   inline void setSWPolarizationReflTrans(                   
      SceneElement theFov,                                                         
      const BOOST::vector<Float64>& theSWPolarizationReflTrans)                    
   {                                                                               
         CorrectionParameter.polarizationReflTrans[Band::SHORTWAVE][theFov].       
           resize(theSWPolarizationReflTrans.size());                                                                    
         CorrectionParameter.polarizationReflTrans[Band::SHORTWAVE][theFov]=       
           theSWPolarizationReflTrans;                                             
   }                                                                               

 protected:
   CorrectionMatrix correctionTable[MAX_FOV][Band::TOTAL_BANDS];
   BOOST::matrix<Float64> resamplingMatrix[Band::TOTAL_BANDS];
   Float64 resampling_laserwavelength;
   boost::numeric::ublas::banded_matrix<
                  Float64,boost::numeric::ublas::column_major> 
                     postFilter[Band::TOTAL_BANDS];
   BOOST::matrix<Float64> apodizationMatrix[Band::TOTAL_BANDS];
   BOOST::matrix<Float64> calibrationMatrix[MAX_FOV][Band::TOTAL_BANDS];
    
 private:
   Int32 engPacketCount;
   CrisSdrAlgDataType *DMS_Data_Inputs;
   Spectra* processSpectra(Spectra* newSpectra);

   Spectra* processSpectra(Spectra* newSpectra, 
                           CrisSdrCoefficients* cfgParmsPtr);
   bool batchEarthSceneFCE(Spectra* newSpectra, Spectra* rawSpectra, 
                           CrisSdrCoefficients* cfgParmsPtr);
   void calibrate(Spectra* newSpectra, Spectra* hotRefSpectra, 
        Spectra* coldRefSpectra, SceneElement oldFOV, Band::Wavelength oldBand);   

   Int32 windowSize;
   SpectraManager processingPeriod;
   Spectra* polarizationCorrectionCurve;
 
   Spectra* addSpectra(Spectra* newSpectra, 
                       CrisSdrCoefficients* cfgParmsPtr);

   Int32 refresh_Invsa_CorrectionMatrix(bool isLoading = false);
   void compute_ICT_Temperatures();
   CalibrationParam CalibrationParameter;
   SDR_CorrectionParam CorrectionParameter;
   CrIS_InstrumentCharacteristics InstrumentCharacteristic;
   CrIS_TestEnvironmentParameters TestEnvironmentParameter;
   SDR_GeneralParameters GeneralParameter;
   SDR_AlgorithmnParameter AlgorithmParameter;
   Float64 maxPathDifferential[Band::TOTAL_BANDS];
   Float64 deltaSigma[Band::TOTAL_BANDS];
   Float64 lowestWavenumber[Band::TOTAL_BANDS];
   TimeRange validCalTargetDuration;
   SYSTEMTIME validCalTargetTolerence;
   TimeRange validColdTargetDuration;
   SYSTEMTIME validColdTargetTolerence;
   TimeRange validHotTargetDuration;
   SYSTEMTIME validHotTargetTolerence;
   Int32 sequenceCount
      [MAX_FOV][Band::TOTAL_BANDS][TOTAL_SCENES][TOTAL_DIRECTIONS];
   Int32 scienceCount
      [MAX_FOV][Band::TOTAL_BANDS][TOTAL_SCENES][TOTAL_DIRECTIONS];
   Int32 targetCount
      [MAX_FOV][Band::TOTAL_BANDS][TOTAL_SCENES][TOTAL_DIRECTIONS];
   Int32 invalidInterferogramCount[MAX_FOV][Band::TOTAL_BANDS];
   Int32 invalidSpectraCount[MAX_FOV][Band::TOTAL_BANDS];
   std::string currentExternalCalFileLine;
   bool monitorLaserDiode();
   bool monitorLaserWavelength();
   EngineeringCalibrationRecord theEngineeringCalibrationRecord;
   ILSParametersRecord theILSParametersRecord;
   TargetTempCalibrationRecord theTargetTempCalibrationRecord;
   ScienceCalibrationRecord theScienceCalibrationRecord;
   Float64 plankConstC1;
   Float64 plankConstC2;
   std::string appDirectory;
   void calculateDesiredBandCenter();
   UInt32 SCIENCE_CALIBRATION_FRAME_TYPE;
   UInt32 ENGINEERING_CALIBRATION_FRAME_TYPE;
   void applySpectralCorrection(Spectra* newSpectra, SceneElement theFov, Band::Wavelength theBand);

   // Use cris_(i/o)archive to Serialize and archive to stringstream
   void serializeCmo(CrISDataStream& ss);

   // Copy archive to DMS struct
   void archiveCmo(CrISDataStream& ss);
   void loadSerializedCorrectionMatrixFromDMS();
   bool requiresPhaseSync[MAX_FOV][Band::TOTAL_BANDS][TOTAL_DIRECTIONS];
   Int32 syncAttemptCount[MAX_FOV][Band::TOTAL_BANDS][TOTAL_DIRECTIONS];
   //thread management
   Int32 totalActiveThreads;
   Int32 threadFOVmap[MAX_FOV];
   UInt32 combineRequestedCorrections(UInt32 self);

   // build interferogram apodization table
   void buildInterferogramApodizationTable();
   BOOST::vector < Float64> interferogramApodizationTable[Band::TOTAL_BANDS];

   bool earthScenefceDetectionComplete[TOTAL_SCENES];
   Int32 referenceHotSceneReceived;

   bool referenceSceneReceived[SpectraManager::TOTAL_REF_SPECTRA]
                              [TOTAL_DIRECTIONS];
   bool referenceSceneSync[MAX_FOV][Band::TOTAL_BANDS][TOTAL_DIRECTIONS];
   bool initialCorrectionReady;
   UncalibratedSpectra FIR_GAIN[Band::TOTAL_BANDS];
   Float64 shiftFactor[MAX_FOV];
   bool appShiftFactorFlag;
   UInt32 scanIdx_;
   UInt16 numCMOBuilds_;
   UInt16 numEngBuilds_;
   bool taskedSequentially_;
   Int16 scanWhichCausedCMORebuild_;
   Int16 scanWhichCausedEngRebuild_;
   SYSTEMTIME minNpCrossingTime_;   
   SYSTEMTIME maxNpCrossingTime_;   
   std::map<Int32, bool> orbitMsgThrottle_;

   Interferogram currentDSfwd;
   Interferogram currentDSrev;
   bool currentDSfwd_p;
   bool currentDSrev_p;

   bool engPkt_Loaded;
   bool ilsCorrection_Loaded;
   bool firstEngPkt;
     
   BOOST::vector  < Float64> forDS_reals
                             [CRIS_SCAN_PER_GRAN][MAX_FOV][Band::TOTAL_BANDS];
   BOOST::vector  < Float64> forDS_imags
                             [CRIS_SCAN_PER_GRAN][MAX_FOV][Band::TOTAL_BANDS];
   BOOST::vector  < Float64> revDS_reals
                             [CRIS_SCAN_PER_GRAN][MAX_FOV][Band::TOTAL_BANDS];
   BOOST::vector  < Float64> revDS_imags
                             [CRIS_SCAN_PER_GRAN][MAX_FOV][Band::TOTAL_BANDS];
};
inline void ScienceDataProcessor::setSpaceTargetTemperatureDriftLimit(
            Float64 theSpaceTargetTemperatureDriftLimit) 
{
   InstrumentCharacteristic.spaceTargetTemperatureDriftLimit = 
      theSpaceTargetTemperatureDriftLimit;
   theTargetTempCalibrationRecord.setSpaceTargetTemperatureDriftLimit(
      theSpaceTargetTemperatureDriftLimit);
}
inline Int32 ScienceDataProcessor::getSpectraCount(
             SceneElement fov, 
             Band::Wavelength band, 
             FieldOfRegard scene, 
             SweepDirection psd) const
{
   return processingPeriod.getWindowDepth(fov, band, scene, psd);
}
inline void ScienceDataProcessor::setDsTemperatureOrigin(
            DsTemperatureOriginSource theDsTemperatureOrigin) 
{
   CalibrationParameter.dsTemperatureOrigin = 
      theDsTemperatureOrigin;
   theTargetTempCalibrationRecord.setDsTemperatureOrigin(
      theDsTemperatureOrigin);
}
inline void ScienceDataProcessor::setIctPrt2Bias(Float64 theIctPrt2Bias) 
{
   AlgorithmParameter.ictPrt2Bias = theIctPrt2Bias;
   theScienceCalibrationRecord.setIctPrt2Bias(theIctPrt2Bias);
}
inline void ScienceDataProcessor::setIctPrt1Bias(Float64 theIctPrt1Bias)
{
   AlgorithmParameter.ictPrt1Bias = theIctPrt1Bias;
   theScienceCalibrationRecord.setIctPrt1Bias(theIctPrt1Bias);
}
inline void ScienceDataProcessor::setBeamsplitterTempBench(
            Float64 theBeamsplitterTempBench) 
{
   TestEnvironmentParameter.beamsplitterTempBench = 
      theBeamsplitterTempBench;
   theScienceCalibrationRecord.setBeamsplitterTempBench(
      theBeamsplitterTempBench);
}
inline void ScienceDataProcessor::setBeamsplitterTempChamber(
            Float64 theBeamsplitterTempChamber) 
{
   TestEnvironmentParameter.beamsplitterTempChamber = 
      theBeamsplitterTempChamber;
   theScienceCalibrationRecord.setBeamsplitterTempChamber(
      theBeamsplitterTempChamber);
}
inline void ScienceDataProcessor::setDsTempBench(Float64 theDsTempBench) 
{
   TestEnvironmentParameter.dsTempBench = theDsTempBench;
   theScienceCalibrationRecord.setDsTempBench(theDsTempBench);
   theTargetTempCalibrationRecord.setDsTempBench(theDsTempBench);
}
inline void ScienceDataProcessor::setDsTempChamber(Float64 theDsTempChamber) 
{
   TestEnvironmentParameter.dsTempChamber = theDsTempChamber;
   theScienceCalibrationRecord.setDsTempChamber(theDsTempChamber);
   theTargetTempCalibrationRecord.setDsTempChamber(theDsTempChamber);
}
inline void ScienceDataProcessor::setIctTempBench(Float64 theIctTempBench) 
{
   TestEnvironmentParameter.ictTempBench = theIctTempBench;
   theScienceCalibrationRecord.setIctTempBench(theIctTempBench);
}
inline void ScienceDataProcessor::setIctTempChamber(Float64 theIctTempChamber) 
{
   TestEnvironmentParameter.ictTempChamber = theIctTempChamber;
   theScienceCalibrationRecord.setIctTempChamber(theIctTempChamber);
}
inline void ScienceDataProcessor::setMeanDsEmissivityBench(
            Float64 theMeanDsEmissivityBench) 
{
   TestEnvironmentParameter.meanDsEmissivityBench = theMeanDsEmissivityBench;
}
inline void ScienceDataProcessor::setMeanDsEmissivityChamber(
            Float64 theMeanDsEmissivityChamber) 
{
   TestEnvironmentParameter.meanDsEmissivityChamber = 
      theMeanDsEmissivityChamber;
}
inline void ScienceDataProcessor::setOmaTempBench(Float64 theOmaTempBench) 
{
   TestEnvironmentParameter.omaTempBench = theOmaTempBench;
   theScienceCalibrationRecord.setOmaTempBench(theOmaTempBench);
}
inline void ScienceDataProcessor::setOmaTempChamber(Float64 theOmaTempChamber) 
{
   TestEnvironmentParameter.omaTempChamber = theOmaTempChamber;
   theScienceCalibrationRecord.setOmaTempChamber(theOmaTempChamber);
}
inline void ScienceDataProcessor::setScanBaffleTempBench(
            Float64 theScanBaffleTempBench) 
{
   TestEnvironmentParameter.scanBaffleTempBench = theScanBaffleTempBench;
   theScienceCalibrationRecord.setScanBaffleTempBench(theScanBaffleTempBench);
}
inline void ScienceDataProcessor::setScanBaffleTempChamber(
            Float64 theScanBaffleTempChamber) 
{
   TestEnvironmentParameter.scanBaffleTempChamber = 
      theScanBaffleTempChamber;
   theScienceCalibrationRecord.setScanBaffleTempChamber(
      theScanBaffleTempChamber);
}
inline void ScienceDataProcessor::setInstrumentTemperatureOrigin(
            InstrumentTemperatureOriginSource theInstrumentTemperatureOrigin) 
{
   CalibrationParameter.instrumentTemperatureOrigin = 
      theInstrumentTemperatureOrigin;
   theScienceCalibrationRecord.setInstrumentTemperatureOrigin(
                               theInstrumentTemperatureOrigin);
}
inline void ScienceDataProcessor::setInstrumentLocation(
            InstrumentLocation theInstrumentLocation)
{
   GeneralParameter.instrumentLocation = theInstrumentLocation;
   theScienceCalibrationRecord.setInstrumentLocation(
            (InstrumentLocation)theInstrumentLocation);
   theTargetTempCalibrationRecord.setInstrumentLocation(theInstrumentLocation);
}
inline void ScienceDataProcessor::setHotReferenceFieldOfRegard(
            FieldOfRegard newFOR)
{
   InstrumentCharacteristic.hotReferenceIdentifer = newFOR;
   processingPeriod.setHotReferenceFieldOfRegard(newFOR);
}
inline void ScienceDataProcessor::setColdReferenceFieldOfRegard(
            FieldOfRegard newFOR)
{
   InstrumentCharacteristic.coldReferenceIdentifer = newFOR;
   processingPeriod.setColdReferenceFieldOfRegard(newFOR);
}
inline void ScienceDataProcessor::enableNEdN(bool state)
{ 
   GeneralParameter.requiresNEdN = state;
   processingPeriod.setMaintainReferenceSceneVariance(state);
}
inline void ScienceDataProcessor::setBoxCarWidth(Int32 theWidth)
{
   GeneralParameter.boxCarWidth = theWidth;
   processingPeriod.setBoxCarWidth(theWidth);
}
inline void ScienceDataProcessor::setPostCalibrationA1(
            Band::Wavelength theBand, Int32 thePostCalibrationA1) 
{
   AlgorithmParameter.postCalibrationA1[theBand] = thePostCalibrationA1;
}
inline void ScienceDataProcessor::setPostCalibrationA2(
            Band::Wavelength theBand, Float64 thePostCalibrationA2) 
{
   AlgorithmParameter.postCalibrationA2[theBand] = thePostCalibrationA2;
}
inline void ScienceDataProcessor::setPostCalibrationA3(
            Band::Wavelength theBand, Int32 thePostCalibrationA3) 
{
   AlgorithmParameter.postCalibrationA3[theBand] = thePostCalibrationA3;
}
inline void ScienceDataProcessor::setPostCalibrationA4(
            Band::Wavelength theBand, Float64 thePostCalibrationA4) 
{
   AlgorithmParameter.postCalibrationA4[theBand] = thePostCalibrationA4;
}
inline void ScienceDataProcessor::setPostCalibrationK(
            Band::Wavelength theBand, Int32 thePostCalibrationK) 
{
   AlgorithmParameter.postCalibrationK[theBand] = thePostCalibrationK;
}
inline void ScienceDataProcessor::setPostCalibrationK0(
            Band::Wavelength theBand, Int32 thePostCalibrationK0) 
{
   AlgorithmParameter.postCalibrationK0[theBand] = thePostCalibrationK0;
}
inline void ScienceDataProcessor::setPostCalibrationK1(
            Band::Wavelength theBand, Int32 thePostCalibrationK1)
{
   AlgorithmParameter.postCalibrationK1[theBand] = thePostCalibrationK1;
}
inline void ScienceDataProcessor::setFceParamAmpThreshRejectLimit(
            Spectra::FCE_LIMIT type,
            Band::Wavelength theBand, 
            Float64 theFceParamAmpThreshRejectLimit) 
{
   AlgorithmParameter.fceParamAmpThreshRejectLimit[theBand][type] = 
      theFceParamAmpThreshRejectLimit;
   Spectra::configureFCEParamAmpThreshRejectLimit(
            type, theFceParamAmpThreshRejectLimit, theBand);
}
inline void ScienceDataProcessor::setFceParamDimensionThresholdLimit(
            Band::Wavelength theBand, 
            Float64 theFceParamDimensionThresholdLimit)
{
   AlgorithmParameter.fceParamDimensionThresholdLimit[theBand] = 
      theFceParamDimensionThresholdLimit;
   Spectra::configureFCEParamDimensionThresholdLimit(
            theFceParamDimensionThresholdLimit, 
            theBand);
}
inline void ScienceDataProcessor::setFceParamFractionalFceThresholdLimit(
            Band::Wavelength theBand, 
            Float64 theFceParamFractionalFceThresholdLimit) 
{
   AlgorithmParameter.fceParamFractionalFceThresholdLimit[theBand] = 
      theFceParamFractionalFceThresholdLimit;
   Spectra::configureFCEParamFractionalFceThresholdLimit(
            theFceParamFractionalFceThresholdLimit, 
            theBand);
}
inline void ScienceDataProcessor::setFceParamGoodLinearFittingThreshLimit(
            Band::Wavelength theBand, 
            Float64 theFceParamGoodLinearFittingThreshLimit) 
{
   AlgorithmParameter.fceParamGoodLinearFittingThreshLimit[theBand] = 
      theFceParamGoodLinearFittingThreshLimit;
   Spectra::configureFCEParamGoodLinearFittingThreshLimit(
            theFceParamGoodLinearFittingThreshLimit, 
            theBand);
}
inline void ScienceDataProcessor::setFceParamMaxIndex(
            Band::Wavelength theBand, 
            Int32 theFceParamMaxIndex) 
{
   AlgorithmParameter.fceParamMaxIndex[theBand] = theFceParamMaxIndex;
   Spectra::configureFCEParamMaxIndex(theFceParamMaxIndex, theBand);
}
inline void ScienceDataProcessor::setFceParamMaxFceThreshLimit(
            Band::Wavelength theBand, 
            Float64 theFceParamMaxFceThreshLimit)
{
   AlgorithmParameter.fceParamMaxFceThreshLimit[theBand] = 
      theFceParamMaxFceThreshLimit;
   Spectra::configureFCEParamMaxFceThreshLimit(
            theFceParamMaxFceThreshLimit, theBand);
}
inline void ScienceDataProcessor::setFceParamMinIndex(
            Band::Wavelength theBand, 
            Int32 theFceParamMinIndex) 
{
   AlgorithmParameter.fceParamMinIndex[theBand] = theFceParamMinIndex;
   Spectra::configureFCEParamMinIndex(theFceParamMinIndex, theBand);
}
inline void ScienceDataProcessor::setFceParamDefaultDetectorBand(
            Int32 theFceParamDefaultDetectorBand)
{
   AlgorithmParameter.fceParamDefaultDetectorBand = 
      theFceParamDefaultDetectorBand;
   Spectra::configureFCEParamDefaultDetectorBand(
            theFceParamDefaultDetectorBand);
}
inline void ScienceDataProcessor::setFceParamDefaultDetectorFOV(
            Int32 theFceParamDefaultDetectorFOV)
{
   AlgorithmParameter.fceParamDefaultDetectorFOV = 
      theFceParamDefaultDetectorFOV;
   Spectra::configureFCEParamDefaultDetectorFOV(theFceParamDefaultDetectorFOV);
}

inline void  ScienceDataProcessor::setFCE_esImagThreshold(
                Float64 esImagThreshold)
{
   AlgorithmParameter.esImagThreshold = esImagThreshold;
   Spectra::configureFCEParamEsImagThreshold(esImagThreshold);
}
inline void ScienceDataProcessor::setFCE_esFceNumErrorThreshold(
               UInt32 esFceNumErrorThreshold)
{
   AlgorithmParameter.esFceNumErrorThreshold = esFceNumErrorThreshold;
   Spectra::configureFCEParamEsFceNumErrorThreshold(esFceNumErrorThreshold);
}
inline void ScienceDataProcessor::setStartWavenumber(UInt32 startWavenumber)
{
   AlgorithmParameter.startWavenumber = startWavenumber;
   Spectra::configureFCEParamStartWavenumber(startWavenumber);
}
inline void ScienceDataProcessor::setEndWavenumber(UInt32 endWavenumber)
{
   AlgorithmParameter.endWavenumber = endWavenumber;
   Spectra::configureFCEParamEndWavenumber(endWavenumber);
}

inline void ScienceDataProcessor::setApplyPolarizationCorrections(
            Int32 theApplyPolarizationCorrections)
{
   CorrectionParameter.applyPolarizationCorrections = 
      theApplyPolarizationCorrections;
}
inline void 
   ScienceDataProcessor::setApplyPostCalibrationFilterMatrixCorrection(
      Int32 theApplyPostCalibrationFilterMatrixCorrection)
{
   CorrectionParameter.applyPostCalibrationFilterMatrixCorrection = 
      theApplyPostCalibrationFilterMatrixCorrection;
}
inline void ScienceDataProcessor::setApplyIlsFovEffectsCorrection(
            Int32 theApplyIlsFovEffectsCorrection)
{
   CorrectionParameter.applyIlsFovEffectsCorrection = 
      theApplyIlsFovEffectsCorrection;
}
inline void ScienceDataProcessor::setApplyIlsResidualEffectCorrection(
            Int32 theApplyIlsResidualEffectCorrection)
{
   CorrectionParameter.applyIlsResidualEffectCorrection = 
      theApplyIlsResidualEffectCorrection;
}
inline void ScienceDataProcessor::setApplyResamplingMatrix(
            Int32 theApplyResamplingMatrix)
{
   CorrectionParameter.applyResamplingMatrix = theApplyResamplingMatrix;
}
inline void ScienceDataProcessor::setDisableLaserMonitoring(
            Int32 theDisableLaserMonitoring)
{
   CorrectionParameter.disableLaserMonitoring = 
      theDisableLaserMonitoring;
}
inline void ScienceDataProcessor::setPerformFringeCountErrorHandling(
            Int32 thePerformFringeCountErrorHandling)
{
   CorrectionParameter.performFringeCountErrorHandling = 
      thePerformFringeCountErrorHandling;
}
inline void ScienceDataProcessor::setPerformPolarizationCorrection(
            Int32 thePerformPolarizationCorrection)
{
   CorrectionParameter.performPolarizationCorrection = 
      thePerformPolarizationCorrection;
}
inline void ScienceDataProcessor::setPerformSpectralAndSpatialCorrection(
            Int32 thePerformSpectralAndSpatialCorrection)
{
   CorrectionParameter.performSpectralAndSpatialCorrection = 
      thePerformSpectralAndSpatialCorrection;
}
inline void ScienceDataProcessor::setUseSavedMatrices(Int32 theUseSavedMatrices)
{
   CorrectionParameter.useSavedMatrices = theUseSavedMatrices;
}
inline void ScienceDataProcessor::setUsePostFilterOrCosineFilter(
               Int32 theUsePostFilterOrCosineFilter)
{
   CorrectionParameter.usePostFilterOrCosineFilter = 
                          theUsePostFilterOrCosineFilter;
}
inline void ScienceDataProcessor::setApodizationType(
            ApodizationType theApodizationType) 
{
   CorrectionParameter.apodizationType = theApodizationType;
}
inline void ScienceDataProcessor::setCalibrationOrder(
               CalibrationOrder theCalibrationOrder) 
{
   CorrectionParameter.calibrationOrder = theCalibrationOrder;
}
inline void ScienceDataProcessor::setSincFlag(CalibrationOrder theCalibrationOrder)
{                                                      
 if( theCalibrationOrder == RadCalFirst_ATBD_versionA )                     
 {                                                     
   CorrectionParameter.sincFlag  = 1;      
 }                                                     
 else                                                  
 {                                                     
   CorrectionParameter.sincFlag  = 0;      
 }                                                     
}                                                      
inline void ScienceDataProcessor::setLaserDiodeWavelengthOrigin(
            LaserWavelengthSource theLaserDiodeWavelengthOrigin) 
{
   CorrectionParameter.laserDiodeWavelengthOrigin = 
      theLaserDiodeWavelengthOrigin;
   theEngineeringCalibrationRecord.setLaserDiodeWavelengthOrigin(
      theLaserDiodeWavelengthOrigin);
}
inline void ScienceDataProcessor::setImpulseNoiseCountThreshold(
            UInt32 theImpulseNoiseCountThreshold) 
{
   CorrectionParameter.impulseNoiseCountThreshold = 
      theImpulseNoiseCountThreshold;
   if(Spectra::IMPULSE_NOISE_COUNT_THRESHOLD != theImpulseNoiseCountThreshold)
      {
         std::string logText;
         char tmpLogText[200];
         sprintf(tmpLogText,
            "NOTE:  Impulse Noise Count Threshold changed to %d",
            theImpulseNoiseCountThreshold);
         logText += tmpLogText;
         EventLogEngine::append(logText.c_str());
         Spectra::configureImpulseNoiseCountThreshold(
            theImpulseNoiseCountThreshold);
      }
}
inline void ScienceDataProcessor::setEdrNumberOfPoints(
            Band::Wavelength theBand, 
            UInt32 theEdrNumberOfPoints)
{
   CorrectionParameter.edrNumberOfPoints[theBand] = theEdrNumberOfPoints;
}
inline void ScienceDataProcessor::setEdrDeltaSigma(
            Band::Wavelength theBand, 
            Float64 theEdrDeltaSigma)
{
   CorrectionParameter.edrDeltaSigma[theBand] = theEdrDeltaSigma;
   InstrumentCharacteristic.deltaSigma[theBand] = theEdrDeltaSigma;
}
inline void ScienceDataProcessor::setEdrMinimumWavenumber(
            Band::Wavelength theBand, 
            Float64 theEdrMinimumWavenumber)
{
   CorrectionParameter.edrMinimumWavenumber[theBand] = theEdrMinimumWavenumber;
   processingPeriod.setEdrMinimumWavenumber(theBand, theEdrMinimumWavenumber);
   InstrumentCharacteristic.minWaveNumber[theBand] = theEdrMinimumWavenumber;
   calculateDesiredBandCenter();
}
inline void ScienceDataProcessor::setEdrMaximumWavenumber(
            Band::Wavelength theBand, 
            Float64 theEdrMaximumWavenumber)
{
   CorrectionParameter.edrMaximumWavenumber[theBand] = theEdrMaximumWavenumber;
   processingPeriod.setEdrMaximumWavenumber(theBand, theEdrMaximumWavenumber);
   InstrumentCharacteristic.maxWaveNumber[theBand] = theEdrMaximumWavenumber;
   calculateDesiredBandCenter();
}
inline void ScienceDataProcessor::setDataPointsDecimatedInterferogram(
            Band::Wavelength theBand, 
            UInt32 theDataPointsDecimatedInterferogram)
{
   //only update if value changes
   if (theDataPointsDecimatedInterferogram != 
          InstrumentCharacteristic.dataPointsDecimatedInterferogram[theBand])
      {
         InstrumentCharacteristic.dataPointsDecimatedInterferogram[theBand] = 
            theDataPointsDecimatedInterferogram;
         calculateWavenumberBinSpacing(theBand);
      }
}
inline void ScienceDataProcessor::setDecimationFactor(
            Band::Wavelength theBand, 
            UInt32 theDecimationFactor)
{
   //only update if value changes
   if (theDecimationFactor != 
          InstrumentCharacteristic.decimationFactor[theBand])
      {
         InstrumentCharacteristic.decimationFactor[theBand] = 
            theDecimationFactor;
         calculateWavenumberBinSpacing(theBand);
      }
}
inline void 
   ScienceDataProcessor::setDataPointsUndecimatedInterferogram(
      Band::Wavelength theBand, 
      UInt32 theDataPointsUndecimatedInterferogram)
{
   InstrumentCharacteristic.dataPointsUndecimatedInterferogram[theBand] = 
      theDataPointsUndecimatedInterferogram;
}
inline void 
   ScienceDataProcessor::setCalibrationTargetDataValidityDuration(
      Float64 theCalibrationTargetDataValidityDuration)
{
   CalibrationParameter.calibrationTargetDataValidityDuration = 
      theCalibrationTargetDataValidityDuration;
   validCalTargetDuration.startTime = SystemTimeUtilities::setTime(
      0,0,0,0.0f,0.0f,theCalibrationTargetDataValidityDuration*2.0f,0);
   validCalTargetDuration.stopTime  = SystemTimeUtilities::setTime(
      0,0,0,0.0f,0.0f,theCalibrationTargetDataValidityDuration*2.0f,0);
}
inline void 
   ScienceDataProcessor::setCalibrationTargetDataValidityDurationTolerance(
      Float64 theCalibrationTargetDataValidityDurationTolerance)
{
   CalibrationParameter.calibrationTargetDataValidityDurationTolerance = 
      theCalibrationTargetDataValidityDurationTolerance;
   validCalTargetTolerence = 
      SystemTimeUtilities::setTime(
         0,0,0,0.0f,0.0f,theCalibrationTargetDataValidityDurationTolerance,0);
   validCalTargetDuration.startTime = 
      SystemTimeUtilities::subtract(
         validCalTargetDuration.startTime, validCalTargetTolerence);
   validCalTargetDuration.stopTime  = 
      SystemTimeUtilities::add(
         validCalTargetDuration.stopTime, validCalTargetTolerence);
}
inline void ScienceDataProcessor::setElapsedTimeForValidScienceTlmData(
            Float64 theElapsedTimeForValidScienceTlmData)
{
   CalibrationParameter.elapsedTimeForValidScienceTlmData = 
      theElapsedTimeForValidScienceTlmData;
}
inline void 
   ScienceDataProcessor::setElapsedTimeForValidSpaceTargetTemperature(
      Float64 theElapsedTimeForValidSpaceTargetTemperature)
{
   CalibrationParameter.elapsedTimeForValidSpaceTargetTemperature = 
      theElapsedTimeForValidSpaceTargetTemperature;
   validColdTargetDuration.startTime = 
      SystemTimeUtilities::setTime(0,0,0,0.0f,0.0f,0.0f,0);
   validColdTargetDuration.stopTime  = 
      SystemTimeUtilities::setTime(0,0,0,0.0f,0.0f,0.0f,0);
}
inline void 
   ScienceDataProcessor::setSpaceTargetTemperatureTimeDifferenceTolerance(
      Float64 theSpaceTargetTemperatureTimeDifferenceTolerance)
{
   CalibrationParameter.spaceTargetTemperatureTimeDifferenceTolerance = 
      theSpaceTargetTemperatureTimeDifferenceTolerance;
   validColdTargetTolerence = 
      SystemTimeUtilities::setTime(
         0,0,0,0.0f,0.0f,theSpaceTargetTemperatureTimeDifferenceTolerance,0);
   validColdTargetDuration.stopTime  = 
      SystemTimeUtilities::add(validColdTargetDuration.stopTime, 
                               validColdTargetTolerence);
}
inline void ScienceDataProcessor::setScienceTlmTimeDifferenceTolerance(
            Float64 theScienceTlmTimeDifferenceTolerance)
{
   CalibrationParameter.scienceTlmTimeDifferenceTolerance = 
      theScienceTlmTimeDifferenceTolerance;
}
inline void ScienceDataProcessor::setDisableTimeStampBasedMovingWindow(
            Int32 theDisableTimeStampBasedMovingWindow)
{
   CalibrationParameter.disableTimeStampBasedMovingWindow = 
      theDisableTimeStampBasedMovingWindow;
   processingPeriod.enableWindowTimeMonitoring(
      theDisableTimeStampBasedMovingWindow == false);
}
inline void ScienceDataProcessor::sendIlsInfoToEngCalRec(
            IlsOriginSource theSource, 
            ILS_Parameters theParams)
{
   theEngineeringCalibrationRecord.setIlsOriginSource(theSource);
   if(theSource == ILS_CONFIG)
   {
      theEngineeringCalibrationRecord.setIlsParameters(theParams);
   }
}


inline void ScienceDataProcessor::setFirFilterResponse(
            Band::Wavelength band, 
            const BOOST::vector<std::complex<Float64> >& response)
{
   InstrumentCharacteristic.
      LinearityCorrectionParameter.FIRresponse[band] = response;
}

inline void ScienceDataProcessor::setLinearityCorrectionParameter_a2(
            Band::Wavelength band, 
            const BOOST::vector<Float64> a2Values)
{
   for(Int32 fovIndex = 0; fovIndex < MAX_FOV; fovIndex++)
   {
      InstrumentCharacteristic.
         LinearityCorrectionParameter.a2[fovIndex][band] = a2Values[fovIndex];
   }
}

inline void ScienceDataProcessor::setLinearityCorrectionParameter_Vinst(
            Band::Wavelength band, 
            const BOOST::vector<Float64> vInst)
{
   for(Int32 fovIndex = 0; fovIndex < MAX_FOV; fovIndex++)
   {
      InstrumentCharacteristic.
         LinearityCorrectionParameter.Vinst[fovIndex][band] = vInst[fovIndex];
   }
}

inline void ScienceDataProcessor::setLinearityCorrectionParameter_ModEff(
            Band::Wavelength band, 
            const BOOST::vector<Float64> modEff)
{
   for(Int32 fovIndex = 0; fovIndex < MAX_FOV; fovIndex++)
   {
       InstrumentCharacteristic.
          LinearityCorrectionParameter.
             ModEff[fovIndex][band] = modEff[fovIndex];
   }
}

inline void ScienceDataProcessor::setPerformLinearityCorrectionControl(
            Band::Wavelength theBand, 
            bool value)
{
   InstrumentCharacteristic.performLinearityCorrectionControl[theBand] = value;
   processingPeriod.setLinearityErrorCorrectionMode(theBand, value);
}

inline void ScienceDataProcessor::setLinearityModeSelectControl(
            Band::Wavelength theBand, 
            LinearityCorrectionControlOriginSource theSource)
{
   InstrumentCharacteristic.
      LinearityCorrectionParameter.
         linearityCorrectionControlParam[theBand] = theSource;
}

inline void ScienceDataProcessor::setLaserDiodeWavelengthOffsetMW(
            Float64 offset)
{
   InstrumentCharacteristic.defaultLaserWavelengthOffsetMW = offset;
}

inline void ScienceDataProcessor::setLaserDiodeWavelengthOffsetSW(
            Float64 offset)
{
   InstrumentCharacteristic.defaultLaserWavelengthOffsetSW = offset;
}

inline void ScienceDataProcessor::setEffGainSetting(
            Band::Wavelength theBand, 
            BOOST::vector<Int32> effectiveGain)
{
   for(Int32 fovIndex = 0; fovIndex < MAX_FOV; fovIndex++)
   {
      InstrumentCharacteristic.LinearityCorrectionParameter.GainSetting
         [fovIndex][theBand] = effectiveGain[fovIndex];
   }
}

inline void ScienceDataProcessor::setEffectiveGainMap(
            Band::Wavelength theBand,
            GAIN_MAP effectiveGainMap)
{
   InstrumentCharacteristic.LinearityCorrectionParameter.EffectiveGainMap
      [theBand] = effectiveGainMap;
}

inline void ScienceDataProcessor::setEngCalInstrumentCharacteristics(
   const CrIS_InstrumentCharacteristics& newValue)
{
   //save values
   theEngineeringCalibrationRecord.setInstrumentCharacteristics(newValue);

   for(Int32 everyBand = 0; everyBand < Band::TOTAL_BANDS; everyBand++)
   {
      //specify LinErr source
      if(newValue.LinearityCorrectionParameter.linearityCorrectionControlParam
         [everyBand] == LINEARITY_TLM)
      {
         //use values from Engineering Cal telemetry 
         processingPeriod.setLinearityErrorParameters(
            (Band::Wavelength) everyBand, 
             theEngineeringCalibrationRecord.
                getEngCalRec_LinearityErrorParameters().getParameters());
      }
      else
      {
         //use values from Config File settings 
         processingPeriod.setLinearityErrorParameters(
            (Band::Wavelength) everyBand, 
            &InstrumentCharacteristic.LinearityCorrectionParameter);

         //refresh Effective Gain from Settings
         for(Int32 everyFOV = 0; everyFOV < MAX_FOV; everyFOV++)
         {
            InstrumentCharacteristic.LinearityCorrectionParameter.EffGain
               [everyFOV][everyBand] =
            InstrumentCharacteristic.
               LinearityCorrectionParameter.
                  EffectiveGainMap[everyBand].find(
                                InstrumentCharacteristic.
                                LinearityCorrectionParameter.
                                GainSetting[everyFOV][everyBand])->second;
         }
      }
   }
}

#endif /* _INC_SCIENCEDATAPROCESSOR_3F86B3F60212_INCLUDED */
