/* BEGIN JPSS HEADER */
/*****************************************************************************
*
* Peraton, Inc.  All rights reserved.
*
* GOVERNMENT RIGHTS:
* Per contract 1332KP23CNEEJ0001, CAR 1352.227-70 applies giving full
* copyrights to data, including reports and other copyrighted materials
* to the United States Government.
*
* THIRD PARTY RIGHTS:
* Note: Third party entities may own copyrights in this work.
*
* EXPORT CONTROL:
* Unless otherwise specified beneath this header, this item does not contain
* Technology or Technical Data controlled under either the U.S. International
* Traffic in Arms Regulations or the U.S. Export Administration Regulations.
*
* CLASSIFICATION: B
* Refer to Software Standards and Practices Manual (SSPM) for coding and
* documentation standards.
*
*****************************************************************************/
/* END JPSS HEADER */

/*****************************************************************************
*
*  NAME: CrIS_SDR.cpp
*
*  DESCRIPTION:   See description preceeding declaration below
*
*****************************************************************************/

/***********************************************************************
*  HISTORY:
*
* DATE        PR#      AUTHOR            Build    DESCRIPTION
* ---------   ---      ------            -----    -----------
* 25Apr2005            Bryan Henderson     1.3    Initial Version
*                      Dan Elliott
*                      Tim Hahn
* 01Mar2006            Dan Elliott         1.4    Fringe Count Error
*                      Tim Hahn                   Follow On Drop
* 19Jun2007            Jeff DeLotelle      1.5    Updated copyright and govern-
*                                                 ment rights info in prologue.
* 29Jun2007            Dan Elliott         1.5    Modification based upon TM:
*                      Bryan Henderson            NP-EMD-2007.510.0027.Rev.A
* 30Oct2007   15919    Roger Ferrel        1.5x1  ECR-122 added spacecraft data
*                                                  to Geolocation.
* 05Dec2007   16138    Bryan Henderson     1.5    Updated to add debug
*                                                 message to staisfy
*                                                 pass/fail for packet
*                                                 parsing
* 14Dec2007   15906    Kelly Boswell       1.5x1  ECR A-103, EDRPR 1.8 CP3
*                                                 Updates moved roll and pitch
*                                                 to temp struct (removed yaw)
* 07Jan2008   15748    Bryan Henderson    1.5.x.1 ISTN_CRIS_SDR_NGST_1.1
*                                                 drop implementation
* 21Feb2008   15748    Kelly Boswell      1.5.x.1 ISTN_CRIS_SDR_NGST_1.1
*                                                 drop implementation -
*                                                 format updates for DFCB
*                                                 compliance
*                                                 ref. NP-EMD-2007.510.0027
*                                                 RevB
* 19MAR2008   17044    K. Boswell         1.5.x.1 Fixed dereferencing null
*                                                 algorithm data pointer from
*                                                 temporary variable; removed
*                                                 PRINT_MEM_INFO statements.
* 12JUN2008   17888    B. Bohannan                logger refactor APIs
* 12JUN2008   17910    K. Boswell          1.5x1  Made
*                                                 theScienceDataProcessor_
*                                                 and sdrHeapDataPtr
*                                                 into smart pointers.
* 25Oct2008            L. Wang           ALG01342 Implem with ITT update 2.14
* 20Jan2009   19301    K. Boswell        1.5x1    ISTN_CRIS_SDR_UNIX_NGST_2.1
*                                                 drop implementation
*                                                 per TM NP-EMD.2008.510.0055
* 21May2009            L. Wang           ALG1443  Update for nonlinearity
*                                                 correction
* 03Feb2010   21718    K. Boswell        SensChar ISTN_CRIS_SDR_UNIX_NGST_2.2
*                                                 drop implementation
*                                                 per TM NP-EMD.2009.510.0046
* 10MAR2010   21464    K. Boswell        SensChar  Changes in support of
*                                                  NP-EMD.2009.510.0046
*                                                  CrIS_Scan Baffle Temperature
*                                                  Correction
* 19MAY2010   23135    K. Boswell        SensChar ISTN_CRIS_SDR_UNIX_NGST_2.2.1
*                                                  drop implementation
*                                                  per TM
*                                                  NP-EMD.2010.510.0006
*                                                  and
*                                                 ISTN_CRIS_SDR_UNIX_NGST_2.2.2
*                                                  per TM
*                                                  NP-EMD.2010.510.0006 Rev A
*                                                  and ECR A-275
* 18OCT2010   24814    W. Dare           1.5.4    Assigned 0 to unassigned ptrs
* 07MAR2012   29775    T. Gardner        Mx7A     Corrected logic in invalid
*                                                 spectral calibration QF
*                                                 setting
* 09MAR2012   29795    T. Gardner          mx6    Fix memory leak
* 12MAR2012   28309    S. Manoj          Mx7      ADR 4389 CrIS SDR:
*                                                 ESZPDMagnitude data type
*                                                 field inconsistencies
* 13JUL2012   31147    S. Manoj           Mx6.3   Code upadate according to
*                                                 474-CCR-12-0441 (ADR 4661
*                                                 and part of ADR 4680), which
*                                                 are related to SDR overall
*                                                 flags that are working
*                                                 incorrectly.
* 31JUL2012   31562    T. Gardner        Mx6.3    Set flag in QF3 and QF4 if
*                                                 imaginary radiance is out of
*                                                 range
* 19NOV2012   32693    T. Gardner         Mx7     Update extraction record on
*                                                  startup with a CMO
*                                                 rather than using default
* 03MAR2013   33615    T. Gardner         Mx7.1   Do not check fill data during
*                                                  radiance threshold check
* 04MAY2013   34609    T. Gardner         Mx8.0   Correct handling of missing
*                                                  science cal packets
* 04JUN2012   30126    J. Schwartz       Blk1.5   Messaging/TaskData/TK updates
* 15MAY2013   33696    J. Schwartz       Blk1.5   Util Time Updates
* 02JUL2013   33702    S. Wilderdyke/    Blk2.0   Changes to work with
*                      S. Manoj                   INF Util Time Updates
* 25JUL2013   35029    T. Gardner         Mx8.1   Check for fill ES RDR Impulse
*                                                  Noise and set IRC
*                                                 to invalid in this case
* 07AUG2013   35180    T. Gardner         Mx8.1   Set SDR Quality to NA for
*                                                 last scan of short granule
* 17FEB2014   37308    R. Neely           Blk2.0  Check imaginary radiance
*                                                 regardless of SDR Quality(
*                                                 DR7365)
*                                                 (Clone of PCR035907)
* 25JUN2014            Y. Chen/Y.Han      Mx8.5   Add calibrationOrder variables
*                                                 in PCT
* 10AUG2014            Y. Chen            Mx8.5   Set flag for CMO and EngPkt
*                                                 loading, and firstEngPkt
*                                                 Add foldIndexOffset and
*                                                 engPktOutputTimeStep in PCT
* 14OCT2014  042555    SEClark         Blk2.0.0.4 CrIS SSPM Compliance.
* 03JAN2015   41268    J. Eberle         Blk2.0   Header Changes
* 11MAY2015   48586    T. Gardner        Blk2.0   Integrate TR/FR updates into
*                                                 block 2.0
* 15May2015            L. Wang        V2.0_beta   Fringe count error correction
*                                                 implementation
* 01JUL2015            L. Wang        V2.0_beta   Updates for FCE handling
* 02JUL2015            Y. Chen        V2.0_beta   DR7982 change maxLunarRadiance
*                                                 to an array and check all
*                                                 bands for lunar intrusion
* 20AUG2015            Y. Chen        V2.0_beta   Add usePostFilterOrCosineFilte
*                                                 variable in PCT
* 28AUG2015   49994    Tim Fisher        Blk2.0   Remove impulseNoiseCountThresh
* 28MAR2016            Y. Chen           Blk2.0   Intergrate TR/FR updates into
*                                                 block 2.0
* 06May2015            Y. Chen           Blk2.0   Adding variable
*                                                 dataPointsTruncatedInterferogram
* 27JUL2017            Y. Chen           Blk2.0   ADR 8361, only output EngPkt
*                                                 using checksum field from
*                                                 EngPkt APID 1290
* 06SEP2017            G. Pachl          Blk2.1   CCRs-17-3543, 3541.
* 30SEP2017            J. Gibbs          Blk2.1   removed unused parameters
* 31AUG2017   63167    J. Gibbs          Blk2.1   fixed compile warnings
* 10Oct2017            Y. Chen           Blk2.0   Code upadate according to
*                                                 ADR 8489 (CCR 3654) to improve the
*                                                 performance when missing DS Cal
*                                                 Packets and ADR 8490 to set the
*                                                 Overall quality to invalid when
*                                                 radiances are NaN
* 10DEC2017   64624    G. Pachl          Blk2.1   CCRs-17-3656.
* 17JUL2018   66227    G. Pachl          Blk2.1   Merge CCR 3922 Lunar intrusion
* 17APR2019            Y. Chen           Blk2.1   CCR 4469 Implemented Polarization
*                                                 Correction
* 13JUN2019   68367    Pachl             Blk2.1   Baselined CCR 4469
* 23OCT2019   68993    D. Tislin         Blk2.2   Fixed compile warnings.
***********************************************************************/

// TODO:  Resolve ProCmnTimeUtil vs InfUtil_TimApi
#include <InfUtil_TimApi.h>
#include <ProCmnTimeUtil.h>

#include <cmath>   // std::isnan()
#include <InfUtil_TimException.h>
#include <string>
#include <ProCmnMessage.h>
#include <EventLogEngine.h>
#include <SweepDirection.h>
#include <Band.h>
#include <Octets.h>
#include <ScienceDataProcessor.h>
#include <CorrectionMatrix.h>
#include <TelemetryServer.h>
#include <TelemetryProcessor.h>
#include <TelemetryStateConverter.h>
#include <CcsdsPrimaryHeader.h>
#include <CcsdsSecondaryHeader.h>
#include <UncalibratedSpectra.h>
#include <InstrumentCharacteristic.h>
#include <CrIS_SDR.h>
#include <CrisSdrCfgParms.h>
#include <TestEnvironmentParameter.h>
#include <CalibrationParameter.h>
#include <CorrectionParameter.h>
#include <GeneralParameter.h>
#include <AlgorithmParameter.h>
#include <ProCmnDefs.h>
#include <ProSdrCrisStruct.h>
#include <sys/stat.h>
#include <limits>
#include <ProCmnLogger.h>
#include <ProCmnProcessData.h>
#include <ProSdrCrisPktGenerator.h>
#include <SpectralResDefs.h>
#include <ProSdrCrisException.h>

/******** Function Prototypes ********/
bool refreshConfigurationData(
        CalibrationParameter&,
        CorrectionParameter&,
        InstrumentCharacteristic&,
        AlgorithmParameter&,
        TestEnvironmentParameter&,
        GeneralParameter&,
        std::auto_ptr<ScienceDataProcessor>&);

void checkRadianceValues(
        UInt32 scan,
        CrisSdrAlgDataType* algDataPtr);

// FCE update
void checkSDR_Status(
        UInt32 scan,
        CrisSdrAlgDataType* algDataPtr,
        std::auto_ptr<ScienceDataProcessor>& theScienceDataProcessor,
        AlgorithmParameter algParams);

Int32 crisMain(
        CrisSdrAlgDataType* algDataPtr,
        std::auto_ptr<ScienceDataProcessor>& theScienceDataProcessor,
        Int16 backInWindow,
        Int32 *stopStatus)
{
    CalibrationParameter calParams;
    CorrectionParameter corParams;
    InstrumentCharacteristic insParams;
    AlgorithmParameter algParams;
    TestEnvironmentParameter testEnvParams;
    GeneralParameter genParams;

    calParams.setConfigParam(algDataPtr);
    corParams.setConfigParam(algDataPtr);
    insParams.setConfigParam(algDataPtr);
    algParams.setConfigParam(algDataPtr);
    testEnvParams.setConfigParam(algDataPtr);
    genParams.setConfigParam(algDataPtr);

    char logMessage[EVENT_LOG_LENGTH];
    sprintf(logMessage,"Calibration Approach: %d ",
                corParams.getCalibrationOrder() );
    (void)EventLogEngine::append(logMessage);


    // Set the min and max North Pole crossing times of the processed
    // granule.
    try
    {
        InfUtil_TimApi* timeUtil = InfUtil_TimApi::getInstance();
        if (timeUtil == 0)
        {
            EventLogEngine::append(
                    "Failed to retrieve an instance of the INF time utility");
            return PRO_FAIL;
        }

        // Convert IET times to SYSTEMTIME.
        InfUtil_TimeGdt time;
        timeUtil->iet2Gdt(algDataPtr->ietGranStartNpCrossing, &time);
        SYSTEMTIME minNpCrossing;
        SystemTimeUtilities::setTime(time, minNpCrossing);
        theScienceDataProcessor->setMinNpCrossingTime(minNpCrossing);

        timeUtil->iet2Gdt(algDataPtr->ietGranEndNpCrossing, &time);
        SYSTEMTIME maxNpCrossing;
        SystemTimeUtilities::setTime(time, maxNpCrossing);
        theScienceDataProcessor->setMaxNpCrossingTime(maxNpCrossing);
    }
    catch (const InfUtil_TimException& ex)
    {
        std::ostringstream exStr;
        exStr << "InfUtil_TimException caught " << ex << "; " << __FILE__
            << __LINE__;
        EventLogEngine::append(exStr.str());
        return PRO_FAIL;
    }

    // Clear out message throttles.
    theScienceDataProcessor->clearMsgThrottles();

    /***Update ScienceDataProcessor with user-supplied options***/
    if(!refreshConfigurationData(calParams,corParams,insParams,algParams,
        testEnvParams,genParams,theScienceDataProcessor))
    {
        (void)EventLogEngine::append(
             "Options not accepted by ScienceDataProcessor.");
        return PRO_FAIL;
    }

    // The following debug message, statisfies the P/F criteria for:
    // The Processing SI shall unpack and byte-align the AP data within
    // the RDRs needed for CrIS SDR processing.
    DEBUG_MED(ProCmnLogger::getLogger(),
              "ProSdrCris - crisMain() packet unpacking beginning");

    /***parse RDRs and send them off to be processed***/
    static UInt32 primaryHeaderSize = (UInt32)TelemetryServer::instance()->
       theTelemetryProcessor->sizeOfPrimaryHeader;
    static UInt32 secondaryHeaderSize = (UInt32)TelemetryServer::instance()->
       theTelemetryProcessor->sizeOfSecondayHeader;
    static CCSDSFormat* theTelemetryFormat =
       (CCSDSFormat*)CommonMessageFormat::instance("CCSDSFormat");

    UInt32 frameApid = 0;
    UInt32 scanIdx = 0;

    /******** save as much of the sliding window as possible ********/
    //we only want to loop over the number of scans in a granule
    //plus the number of scans needed to refill the window
    if(abs(backInWindow) >= CRIS_WINDOW_SIZE/2)//if this is too far back in
                                               //time to make it worthwhile...
    {
        // may be the first run, but message still holds true
        ProCmnMessage::getInstance()->updateHealth(
                "Load Correction Matrix Started");
        theScienceDataProcessor->loadCMOs();
        ProCmnMessage::getInstance()->updateHealth(
                "Load Correction Matrix Ended");
    }
    else//if last tasking was recent enough to make saving the
        //window worthwhile...
    {
        //backInWindow is negative and cannot be less than  negative of window
        //size + scans per granule
        //backInWindow is negative and therefore will set the counter to
        //backInWindow less than
        //number of scans per granule plus number of scans per window which is
        //the max number of scans to process
        //for any given tasking
        scanIdx = (CRIS_SCAN_PER_GRAN+CRIS_WINDOW_SIZE) + backInWindow;
        ProCmnMessage::getInstance()->updateHealth(
                "Load Correction Matrix Started");
        theScienceDataProcessor->loadCMOs();
        ProCmnMessage::getInstance()->updateHealth(
                "Load Correction Matrix Ended");
    }
    // load ILS CMO if it exists
    ProCmnMessage::getInstance()->updateHealth(
            "Load Correction Matrix Started");
    if( !theScienceDataProcessor->get_IlsCorrection_Loaded() )
    {
      if(theScienceDataProcessor->loadCMOs())
      {
         theScienceDataProcessor->set_IlsCorrection_Loaded(true);
      }
    }
    ProCmnMessage::getInstance()->updateHealth(
            "Load Correction Matrix Ended");

    // No Eng Packet in the RDR data, need to load SA and EngPkt and calculate
    // the correction matrix
    if ((algDataPtr->CMOPtr_in != 0) && (algDataPtr->CMOPtr_in != DM_BADADDRESS) &&
            (algDataPtr->EngPtr_in != 0) && (algDataPtr->EngPtr_in != DM_BADADDRESS) &&
            (!theScienceDataProcessor->get_firstEngPkt()))
    {
        ProCmnMessage::getInstance()->updateHealth(
                "Load EngPkt Started");
        if( !theScienceDataProcessor->get_EngPkt_Loaded() )
        {
          if(theScienceDataProcessor->loadEngPkt())
          {
             theScienceDataProcessor->set_EngPkt_Loaded(true);
          }
        }
        ProCmnMessage::getInstance()->updateHealth(
                "Load EngPkt ended");
        if(!theScienceDataProcessor->get_firstEngPkt())
        {
          ProCmnMessage::getInstance()->updateHealth(
                "Build Correction Matrix Started");
          theScienceDataProcessor->refreshCorrectionMatrix(true);
          theScienceDataProcessor->set_firstEngPkt(true);
          ProCmnMessage::getInstance()->updateHealth(
                "Build Correction Matrix Ended");
        }
    }

    // If we have just started up we may be running with the default extraction
    // record.  If so, and we have data from the CMO, we should initialize with
    // the last extraction record data, otherwise we would use the default data
    // until we encounter an engineering packet.
    if ((TelemetryProcessor::instance()->theVideoData.usingDefaultExtRcd()) &&
         (theScienceDataProcessor->getEngineeringCalibrationRecord().
            getEngCalRec_ExtractionRecord().getCollectedSamples(0) !=  0))
    {
       if (ProCmnLogger::getLogger().isDebugHighEnabled())
       {
           std::ostringstream msg;
           msg << "Loading extraction record";
           DEBUG_HIGH(ProCmnLogger::getLogger(), msg.str());
       }

       for (UInt32 band = 0; band < TOTAL_WAVELENGTHS; band++)
       {
          VideoData::EXTRACTION_RECORD record = theScienceDataProcessor->
             getEngineeringCalibrationRecord().getEngCalRec_ExtractionRecord().
                getExtractionRecord(band);
          TelemetryProcessor::instance()->theVideoData.
             setExtractionRecord(record, band);
       }

       TelemetryProcessor::instance()->
                              theVideoData.setUsingDefaultExtRcd(false);
    }

    UInt32 totalNumNeonCal[CRIS_SCAN_PER_GRAN] = {0};
    UInt32 numInvalidNeonCal[CRIS_SCAN_PER_GRAN] = {0};
    UInt32 maxScanIdx = CRIS_SCAN_PER_GRAN + CRIS_WINDOW_SIZE;

    // Adjust for short granule. Skip processing the last scan, because it will
    // be processed in the next granule.
    if (algDataPtr->isShortGranule)
    {
        maxScanIdx -= 1;
    }

    for( ; scanIdx < maxScanIdx; scanIdx++)
    {
        if(*stopStatus == PROCESS_STOP)
        {
            return PRO_STOP;
        }

        theScienceDataProcessor->setScanIdx(scanIdx);
        std::ostringstream message;
        message << "Processing CrIS Scan = " << scanIdx;
        // Update health status
        ProCmnMessage::getInstance()->updateHealth(message.str());

        for (UInt32 pktIdx = 0; pktIdx < CRIS_MAX_PKTS_SCAN; pktIdx++)
        {
            ProCmnRDRItemPacketStruct* pktInfo =
                &algDataPtr->pktPtr->scans[scanIdx].arrayStruct.pktPtr[pktIdx];
            if (pktInfo->packetPtr == 0)
            {
                // Suppress log message for 4-min engineering packets.
                if (pktIdx > 0)
                {
                    // Missing scene packets should be represented by
                    // placeholders. This should only happen for missing
                    // eng & science telemetry packets.
                    if (ProCmnLogger::getLogger().isDebugHighEnabled())
                    {
                        std::ostringstream msg;
                        msg << "Encountered a missing packet pointer at "
                            << "scan index " << scanIdx
                            << " and packet index " << pktIdx;
                        DEBUG_HIGH(ProCmnLogger::getLogger(), msg.str());
                    }
                }
                continue;
            }

            // Check if the packet is a placeholder for missing data.
            bool missingPacket = ProSdrCrisPktGenerator::getInstance().
                                      isFill(*pktInfo);

            // Copy the packet into the Octets object for processing
            Octets* theData = new Octets(pktInfo->packetSize,
                                         pktInfo->packetPtr );

            UInt32 startingPosition = 0;
            UInt32 startOfPacket;

            bool validData = true;
            UInt32 detector, band, scene;
            //for each packet in the file
            UInt32 dataSize = theData->size();
            while((startingPosition < dataSize) && validData)
            {
                startOfPacket = startingPosition;
                if(theTelemetryFormat->unpack(*theData, startingPosition) ==
                   success())//success() from MissivePart.h
                {
                    //get packet's data
                    UInt32 packetSize = theTelemetryFormat->theFrameSize_ +
                       primaryHeaderSize + 1;
                    frameApid = theTelemetryFormat->theFrameType_;
                    //identify RDR
                    TelemetryProcessor::instance()->rdrLookup(frameApid,
                                                              &detector, &band,
                                                              &scene);
                    //forward interesting RDRs
                    if((band != VideoData::Unknown) && (detector > 0))
                    {
                        //this is an  interferogram packet
                        //check if parameters indicate this FOV should be processed
                        if(genParams.getSelectedDetector(
                           (SceneElement)(detector-1),(Band::Wavelength)band))
                        {
                            TelemetryServer::instance()->dispatchData(theData->
                               data() + startOfPacket,(Int32)packetSize);
                            FieldOfRegard currentFieldOfRegard;
                            if(TelemetryProcessor::instance()->
                               getCurrentVideoSource() ==
                                  TelemetryProcessor::NORMAL_BURST)
                            {
                                currentFieldOfRegard =
                                   TelemetryStateConverter::getScene(
                                                    (SceneElement)(detector-1),
                                                        (Band::Wavelength)band,
                                  (TelemetryProcessor::SceneLocationType)scene);
                            }
                            else
                            {
                                currentFieldOfRegard =
                                   TelemetryStateConverter::getDiagScene(
                                      (Band::Wavelength)band);
                            }
                            //determine if a particular FOR should be processed.
                            if(genParams.getSelectedFOR(
                                  currentFieldOfRegard < TOTAL_SCENES ?
                                     currentFieldOfRegard : NonStandardDwell))
                            {
                                //cleans up unsynchronized windows if a packet
                                //is lost during streaming of data
                                if(calParams.getCalibratedWindowSize() != 0 &&
                                   frameApid == theScienceDataProcessor->
                                    getScienceCalibrationRecordIdentifier())
                                {
                                    theScienceDataProcessor->syncWindows();
                                }

                                //make single spectra
                                Spectra* newSpectra = 0;
                                if(newSpectra = theScienceDataProcessor->
                                   processRawDataRecord(
                                            frameApid,
                                            missingPacket,
                                            theTelemetryFormat,
                                            algDataPtr->cfgParmsPtr,
                                            algDataPtr))//test for NULL
                                {
                                    //Add newSpectra to outgoing SDR
                                    if(pktInfo->packetTime >=
                                            (algDataPtr->ietScanPrevStart +
                                            (CRIS_WINDOW_SIZE*EIGHT_SECS)) &&
                                       pktInfo->packetTime <
                                        (algDataPtr->ietScanPrevStart +
                                         ((CRIS_WINDOW_SIZE+ProCmnProcessData::
                                          getInstance()->
                                           getActScan())*EIGHT_SECS)))
                                    {
                                        UInt32 scanIdxTmp = scanIdx -
                                           CRIS_WINDOW_SIZE;
                                        newSpectra->updateCrisSdrGeolocation(
                                           algDataPtr, scanIdxTmp,
                                           theScienceDataProcessor->
                                            getEngineeringCalibrationRecord().
                                             getEngCalRec_PitchRollYawInfo().
                                              getForTimeStampBias());
                                        newSpectra->updateCrisSdrData(
                                           algDataPtr->sdrPtr,
                                           &algDataPtr->headerData, scanIdxTmp,
                                           theScienceDataProcessor);

                                        numInvalidNeonCal[scanIdxTmp] +=
                                            newSpectra->getSDR_InvalidNeonCal();
                                        totalNumNeonCal[scanIdxTmp]++;
                                    }

                                    if(!SpectraManager::
                                       isReferenceScene(*newSpectra)||
                                           theScienceDataProcessor->
                                            getCorrectionParameter().
                                             performFringeCountErrorHandling ==
                                              false)
                                    {
                                        delete newSpectra;
                                        newSpectra = 0;
                                    }
                                }
                            }
                            else
                            {
                                //skip this FOR
                            }
                        }
                        else
                        {
                            //skip this FOV
                        }
                    }
                    else
                    {
                        //not an interferogram packet
                        Int32 recordID = TelemetryProcessor::instance()->
                           findRecord(frameApid);
                        if(recordID !=  TelemetryProcessor::NOT_FOUND)
                        {
                            if(TelemetryProcessor::instance()->
                               getRecord(recordID).getFrameGroup().
                                  find("Housekeeping") == std::string::npos)
                            {
                                TelemetryServer::instance()->
                                   dispatchData(theData->data() +
                                      startOfPacket, (Int32)packetSize);
                                //make single spectra
                                Spectra* newSpectra = 0;

                                CrisSdrAlgDataType* algDataPtrTmp = 0;
                                UInt32 recordID_tmp = theScienceDataProcessor->
                                   getScienceCalibrationRecordIdentifier();
                                Int32 scanIdxTmp = scanIdx - CRIS_WINDOW_SIZE;
                                if(frameApid == recordID_tmp)
                                {
                                   if((scanIdxTmp >= 0) &&
                                      (scanIdxTmp < CRIS_SCAN_PER_GRAN))
                                   {
                                      algDataPtrTmp = algDataPtr;
                                   }
                                }
                                try
                                {

                                if(newSpectra = theScienceDataProcessor->
                                   processRawDataRecord(
                                                    frameApid,
                                                    missingPacket,
                                                    theTelemetryFormat,
                                                    algDataPtr->cfgParmsPtr,
                                                    algDataPtrTmp,
                                                    scanIdxTmp))//test for NULL
                                   {
                                       delete newSpectra;
                                       newSpectra = 0;
                                   }
                                }
                                // stop execution if error updating CMO
                                catch (const ProSdrCrisException& ilschanged)
                                {

                                  std::ostringstream exStr;
                                  exStr << ilschanged.getErrorString();
                                  EventLogEngine::append(exStr.str());
                                  return PRO_FAIL;
                                }
                            }
                            else
                            {
                                //general housekeeping packet not needed for
                                //processing
                            }
                        }
                        else
                        {
                            (void)EventLogEngine::append("APID unknown!");
                        }
                    }
                    //move to next packet
                    if((CCSDSFormat*)(theTelemetryFormat)->
                        getPrimaryHeader()->containsSecondaryHeader())
                    {
                        startingPosition +=
                           (theTelemetryFormat->theFrameSize_ + 1) -
                               secondaryHeaderSize;
                    }
                    else
                    {
                        startingPosition +=
                           (theTelemetryFormat->theFrameSize_ + 1);
                    }
                }
                else
                {
                    //bad header
                    validData = false;
                    (void)EventLogEngine::append("Bad header detected!");
                }
            }
            delete theData;
            theData = 0;
        }

        if(theScienceDataProcessor->getScienceCalibrationRecord().
           isSciCalMissing() && scanIdx > CRIS_WINDOW_SIZE)
        {
            // When the function isSciMissing() returns a boolean value true,
            // it indicates a missing sciCal packet in the previous scan
            theScienceDataProcessor->
              setScienceCalibrationRecord_ICT_Status(scanIdx-CRIS_WINDOW_SIZE-1,
                 algDataPtr);
        }

    }//finished with all raw RDRs.

    // Take care of possibly missing science calibration packet in the last
    // scan (scan # 34)
    theScienceDataProcessor->checkICT_Status((
       (!algDataPtr->isShortGranule) ?
            CRIS_SCAN_PER_GRAN-1 :
               CRIS_SCAN_PER_GRAN-2), algDataPtr);

    for(UInt32 scanIdxCnt = 0;
        scanIdxCnt < ProCmnProcessData::getInstance()->getActScan(); scanIdxCnt++)
    {
        // Check radiance values for divide by zero
#ifndef PSEUDO_TEST_DATA
        checkRadianceValues(scanIdxCnt, algDataPtr);
#endif

        Float64 pctInvalidCal =
            static_cast  <Float64>(numInvalidNeonCal[scanIdxCnt]) /
            static_cast  <Float64>(totalNumNeonCal[scanIdxCnt]);
        if (pctInvalidCal > algDataPtr->cfgParmsPtr->
            invalidNeonCalibrationPercentageThreshold)
        {
            algDataPtr->sdrPtr->QF1[scanIdxCnt] |= SUSPECT_NEON_CALIBRATION;
        }

        // FCE update
        checkSDR_Status(scanIdxCnt,algDataPtr,theScienceDataProcessor,algParams);
    }

    // Set SDR quality to NA for the non-existent scan of a short granule
    if (algDataPtr->isShortGranule)
    {
        std::fill_n(&algDataPtr->sdrPtr->QF3[CRIS_SCAN_PER_GRAN-1][0][0][0],
           EV_FOR_PER_SCAN * CRIS_MAX_FOV * Band::TOTAL_BANDS, SDR_QUALITY_NA);
    }

    return (PRO_SUCCESS);
}

bool refreshConfigurationData(
        CalibrationParameter& calParams,
        CorrectionParameter& corParams,
        InstrumentCharacteristic& insParams,
        AlgorithmParameter& algParams,
        TestEnvironmentParameter& testEnvParams,
        GeneralParameter& genParams,
        std::auto_ptr<ScienceDataProcessor>& theScienceDataProcessor)
{
    Spectra::enableScaling(true);

    //setting Calibration Parameters in Sci Data Proc from
    //CalibrationParameterTab
    theScienceDataProcessor->setAllowCalibrationTargetDataMissing(
       calParams.getAllowCalibrationTargetDataMissing() );
    theScienceDataProcessor->setAllowEngineeringDataPacketsMissing(
       calParams.getAllowEngineeringDataPacketsMissing() );
    theScienceDataProcessor->setAllowSpaceTargetTemperatureDataMissing(
       calParams.getAllowSpaceTargetTemperatureDataMissing() );
    theScienceDataProcessor->setDisableTimeStampBasedMovingWindow(
       calParams.getDisableTimeStampBasedMovingWindow() );
    theScienceDataProcessor->setPerformRadiometricCalibration(
       calParams.getPerformRadiometricCalibration() );
    theScienceDataProcessor->setSkipIctDsPhaseSynchronization(
       calParams.getSkipIctDsPhaseSynchronization() );
    theScienceDataProcessor->setUseDeepSpaceRadiance(
       calParams.getUseDeepSpaceRadiance() );
    theScienceDataProcessor->setUseIctEnvironmentalCorrectionModel(
       calParams.getUseIctEnvironmentalCorrectionModel() );
    theScienceDataProcessor->setUseWavenumberDependentDsEmissivity(
       calParams.getUseWavenumberDependentDsEmissivity() );
    theScienceDataProcessor->setUseWavenumberDependentIctEmissivity(
       calParams.getUseWavenumberDependentIctEmissivity() );
    theScienceDataProcessor->setCalibrationTargetDataValidityDuration(
       calParams.getCalibrationTargetDataValidityDuration() );
    theScienceDataProcessor->setCalibrationTargetDataValidityDurationTolerance(
       calParams.getCalibrationTargetDataValidityDurationTolerance() );
    theScienceDataProcessor->setElapsedTimeForValidScienceTlmData(
       calParams.getElapsedTimeForValidScienceTlmData() );
    theScienceDataProcessor->setElapsedTimeForValidSpaceTargetTemperature(
       calParams.getElapsedTimeForValidSpaceTargetTemperature() );
    theScienceDataProcessor->setSpaceTargetTemperatureTimeDifferenceTolerance(
       calParams.getSpaceTargetTemperatureTimeDifferenceTolerance() );
    theScienceDataProcessor->setScienceTlmTimeDifferenceTolerance(
       calParams.getScienceTlmTimeDifferenceTolerance() );
    theScienceDataProcessor->setMovingAverageWindowSize(
       calParams.getMovingAverageWindowSize() );
    theScienceDataProcessor->setAllowScienceTlmDataMissing(
       calParams.getAllowScienceTlmDataMissing() );
    theScienceDataProcessor->
       setMaximumNumberOfFceTriesDuringIctDsSynchronization(
          calParams.getMaximumNumberOfFceTriesDuringIctDsSynchronization() );
    theScienceDataProcessor->setMaximumNumberOfIctDsSynchronizationTries(
       calParams.getMaximumNumberOfIctDsSynchronizationTries() );
    theScienceDataProcessor->setDsTemperatureOrigin(
       (DsTemperatureOriginSource)(calParams.getDsTemperatureOrigin()) );
    theScienceDataProcessor->setIctEmissivityOrigin(
       (IctEmissivityOriginSource)(calParams.getIctEmissivityOrigin()) );
    theScienceDataProcessor->setInstrumentTemperatureOrigin(
       (InstrumentTemperatureOriginSource)(
          calParams.getInstrumentTemperatureOrigin()) );
    theScienceDataProcessor->enableLunarIntrusionTracking(
       calParams.getMonitorLunarIntrusion(), Band::LONGWAVE,
       calParams.getMaxLunarRadianceRatio(Band::LONGWAVE),
       calParams.getMinFreqMicroWindow(Band::LONGWAVE),
       calParams.getMaxFreqMicroWindow(Band::LONGWAVE));
    theScienceDataProcessor->enableLunarIntrusionTracking(
       calParams.getMonitorLunarIntrusion(), Band::MIDWAVE,
       calParams.getMaxLunarRadianceRatio(Band::MIDWAVE),
       calParams.getMinFreqMicroWindow(Band::MIDWAVE),
       calParams.getMaxFreqMicroWindow(Band::MIDWAVE));
    theScienceDataProcessor->enableLunarIntrusionTracking(
       calParams.getMonitorLunarIntrusion(), Band::SHORTWAVE,
       calParams.getMaxLunarRadianceRatio(Band::SHORTWAVE),
       calParams.getMinFreqMicroWindow(Band::SHORTWAVE),
       calParams.getMaxFreqMicroWindow(Band::SHORTWAVE));
    theScienceDataProcessor->setReferenceWindowSize(
       calParams.getReferenceWindowSize());
    theScienceDataProcessor->setTargetTempWindowSize(
       calParams.getTargetTempWindowSize());
    theScienceDataProcessor->setCalibratedWindowSize(
       calParams.getCalibratedWindowSize());
    theScienceDataProcessor->setValidWindowSize(
       calParams.getValidWindowSize());
    theScienceDataProcessor->setSuppressBaffleProfile(
       calParams.getSuppressSsmBaffleProfile());

    //setting Correction Parameters in Sci Data Proc from corParamseterTab
    theScienceDataProcessor->setApplyPolarizationCorrections(
       corParams.getApplyPolarizationCorrections() );
    theScienceDataProcessor->setApplyPostCalibrationFilterMatrixCorrection(
       corParams.getApplyPostCalibrationFilterMatrixCorrection() );
    theScienceDataProcessor->setApplyIlsFovEffectsCorrection(
       corParams.getApplyIlsFovEffectsCorrection() );
    theScienceDataProcessor->setApplyIlsResidualEffectCorrection(
       corParams.getApplyIlsResidualEffectCorrection() );
    theScienceDataProcessor->setApplyResamplingMatrix(
       corParams.getApplyResamplingMatrix() );
    theScienceDataProcessor->setDisableLaserMonitoring(
       corParams.getDisableLaserMonitoring() );
    theScienceDataProcessor->setPerformFringeCountErrorHandling(
       corParams.getPerformFringeCountErrorHandling() );
    theScienceDataProcessor->setPerformPolarizationCorrection(
       corParams.getPerformPolarizationCorrection() );
    theScienceDataProcessor->setPerformSpectralAndSpatialCorrection(
       corParams.getPerformSpectralAndSpatialCorrection() );
    theScienceDataProcessor->setUseSavedMatrices(
       corParams.getUseSavedMatrices() );
    theScienceDataProcessor->setUsePostFilterOrCosineFilter(
       corParams.getUsePostFilterOrCosineFilter() );
    theScienceDataProcessor->setApodizationType(
       (ApodizationType) corParams.getApodizationType() );
    theScienceDataProcessor->setLaserDiodeWavelengthOrigin(
       corParams.getLaserDiodeWavelengthOrigin() );
    theScienceDataProcessor->setImpulseNoiseCountThreshold(
       corParams.getImpulseNoiseCountThreshold() );
    theScienceDataProcessor->setUserClippingSelection(
       corParams.getUserSelectedClipping() == true);

    theScienceDataProcessor->setEdrNumberOfPoints(
       Band::LONGWAVE, corParams.getEdrLwNumberOfPoints() );
    theScienceDataProcessor->setEdrNumberOfPoints(
       Band::MIDWAVE, corParams.getEdrMwNumberOfPoints() );
    theScienceDataProcessor->setEdrNumberOfPoints(
       Band::SHORTWAVE, corParams.getEdrSwNumberOfPoints() );
    theScienceDataProcessor->setEdrDeltaSigma(
       Band::LONGWAVE, corParams.getEdrLwDeltaSigma() );
    theScienceDataProcessor->setEdrDeltaSigma(
       Band::MIDWAVE, corParams.getEdrMwDeltaSigma() );
    theScienceDataProcessor->setEdrDeltaSigma(
       Band::SHORTWAVE, corParams.getEdrSwDeltaSigma() );
    theScienceDataProcessor->setEdrMaximumWavenumber(
       Band::LONGWAVE, corParams.getEdrLwMaximumWavenumber() );
    theScienceDataProcessor->setEdrMaximumWavenumber(
       Band::MIDWAVE, corParams.getEdrMwMaximumWavenumber() );
    theScienceDataProcessor->setEdrMaximumWavenumber(
       Band::SHORTWAVE, corParams.getEdrSwMaximumWavenumber() );
    theScienceDataProcessor->setEdrMinimumWavenumber(
       Band::LONGWAVE, corParams.getEdrLwMinimumWavenumber() );
    theScienceDataProcessor->setEdrMinimumWavenumber(
       Band::MIDWAVE, corParams.getEdrMwMinimumWavenumber() );
    theScienceDataProcessor->setEdrMinimumWavenumber(
       Band::SHORTWAVE, corParams.getEdrSwMinimumWavenumber() );

    //setting Instrument Parameters in Sci Data Proc from
    //theInstrumentCharacteristicTab
    theScienceDataProcessor->setBenchMeanIctEmissivity(
       Band::LONGWAVE, insParams.getLwBenchMeanIctEmissivity() );
    theScienceDataProcessor->setBenchMeanIctEmissivity(
       Band::MIDWAVE, insParams.getMwBenchMeanIctEmissivity() );
    theScienceDataProcessor->setBenchMeanIctEmissivity(
       Band::SHORTWAVE,  insParams.getSwBenchMeanIctEmissivity() );
    theScienceDataProcessor->setChamberMeanIctEmissivity(
       Band::LONGWAVE, insParams.getLwChamberMeanIctEmissivity() );
    theScienceDataProcessor->setChamberMeanIctEmissivity(
       Band::MIDWAVE, insParams.getMwChamberMeanIctEmissivity() );
    theScienceDataProcessor->setChamberMeanIctEmissivity(
       Band::SHORTWAVE,  insParams.getSwChamberMeanIctEmissivity() );
    theScienceDataProcessor->setHotReferenceFieldOfRegard(
       (FieldOfRegard)insParams.getICT_FieldOfRegard() );
    theScienceDataProcessor->setColdReferenceFieldOfRegard(
       (FieldOfRegard)insParams.getDS_FieldOfRegard() );
    theScienceDataProcessor->setDefaultLaserWavelength(
       insParams.getDefaultLaserWavelength() );
    theScienceDataProcessor->setCalibrationOrder(
       (CalibrationOrder) corParams.getCalibrationOrder() );
    theScienceDataProcessor->setSincFlag(
       (CalibrationOrder) corParams.getCalibrationOrder() );
    theScienceDataProcessor->setSAExpansionFactor(
       Band::LONGWAVE, corParams.getSAExpansionFactor(Band::LONGWAVE));
    theScienceDataProcessor->setSAExpansionFactor(
       Band::MIDWAVE, corParams.getSAExpansionFactor(Band::MIDWAVE));
    theScienceDataProcessor->setSAExpansionFactor(
       Band::SHORTWAVE, corParams.getSAExpansionFactor(Band::SHORTWAVE));
    theScienceDataProcessor->setNumberFOR( insParams.getNumberFOR() );
    theScienceDataProcessor->setNumberFOV( insParams.getNumberFOV() );
    theScienceDataProcessor->setNumberSpectralBands(
       insParams.getNumberSpectralBands() );
    theScienceDataProcessor->setNumberSamplesPerLaserWavelength(
       insParams.getNumberSamplesPerLaserWavelength() );
    theScienceDataProcessor->setSpaceTargetTemperatureDriftLimit(
       insParams.getSpaceTargetTemperatureDriftLimit() );
    theScienceDataProcessor->setReverseSweepDirectionIdentifier(
       insParams.getReverseSweepDirectionIdentifier() );
    theScienceDataProcessor->setForwardSweepDirectionIdentifier(
       insParams.getForwardSweepDirectionIdentifier() );
    theScienceDataProcessor->assignCalibrationRecordIdentifiers(
       insParams.getSciencePacketFrameType(),
       insParams.getEngineeringPacketFrameType());
    theScienceDataProcessor->setFirAccumulatorStartBit(
       Band::LONGWAVE, insParams.getAccumulatorStartBit(Band::LONGWAVE));
    theScienceDataProcessor->setFirAccumulatorStartBit(
       Band::MIDWAVE, insParams.getAccumulatorStartBit(Band::MIDWAVE));
    theScienceDataProcessor->setFirAccumulatorStartBit(
       Band::SHORTWAVE, insParams.getAccumulatorStartBit(Band::SHORTWAVE));
    theScienceDataProcessor->setFirFilterScale(
       Band::LONGWAVE, insParams.getFirFilterScaleFactor(Band::LONGWAVE));
    theScienceDataProcessor->setFirFilterScale(
       Band::MIDWAVE, insParams.getFirFilterScaleFactor(Band::MIDWAVE));
    theScienceDataProcessor->setFirFilterScale(
       Band::SHORTWAVE, insParams.getFirFilterScaleFactor(Band::SHORTWAVE));

    theScienceDataProcessor->setPassBandStart(
       Band::LONGWAVE, insParams.getPassBandStart(Band::LONGWAVE));
    theScienceDataProcessor->setPassBandStart(
       Band::MIDWAVE, insParams.getPassBandStart(Band::MIDWAVE));
    theScienceDataProcessor->setPassBandStart(
       Band::SHORTWAVE, insParams.getPassBandStart(Band::SHORTWAVE));

    theScienceDataProcessor->setPassBandStop(
       Band::LONGWAVE, insParams.getPassBandStop(Band::LONGWAVE));
    theScienceDataProcessor->setPassBandStop(
       Band::MIDWAVE, insParams.getPassBandStop(Band::MIDWAVE));
    theScienceDataProcessor->setPassBandStop(
       Band::SHORTWAVE, insParams.getPassBandStop(Band::SHORTWAVE));

    theScienceDataProcessor->setFirFilterResponse(
       Band::LONGWAVE, insParams.getFirFilterResponse(Band::LONGWAVE));
    theScienceDataProcessor->setFirFilterResponse(
       Band::MIDWAVE, insParams.getFirFilterResponse(Band::MIDWAVE));
    theScienceDataProcessor->setFirFilterResponse(
       Band::SHORTWAVE, insParams.getFirFilterResponse(Band::SHORTWAVE));

    theScienceDataProcessor->setLinearityCorrectionParameter_a2(
       Band::LONGWAVE, insParams.getLinErr_a2(Band::LONGWAVE));
    theScienceDataProcessor->setLinearityCorrectionParameter_a2(
       Band::MIDWAVE, insParams.getLinErr_a2(Band::MIDWAVE));
    theScienceDataProcessor->setLinearityCorrectionParameter_a2(
       Band::SHORTWAVE, insParams.getLinErr_a2(Band::SHORTWAVE));

    theScienceDataProcessor->setLinearityCorrectionParameter_Vinst(
       Band::LONGWAVE, insParams.getLinErr_Vinst(Band::LONGWAVE));
    theScienceDataProcessor->setLinearityCorrectionParameter_Vinst(
       Band::MIDWAVE, insParams.getLinErr_Vinst(Band::MIDWAVE));
    theScienceDataProcessor->setLinearityCorrectionParameter_Vinst(
       Band::SHORTWAVE, insParams.getLinErr_Vinst(Band::SHORTWAVE));

    theScienceDataProcessor->setLinearityCorrectionParameter_ModEff(
       Band::LONGWAVE, insParams.getLinErr_ModEff(Band::LONGWAVE));
    theScienceDataProcessor->setLinearityCorrectionParameter_ModEff(
       Band::MIDWAVE, insParams.getLinErr_ModEff(Band::MIDWAVE));
    theScienceDataProcessor->setLinearityCorrectionParameter_ModEff(
       Band::SHORTWAVE, insParams.getLinErr_ModEff(Band::SHORTWAVE));

    theScienceDataProcessor->setPerformLinearityCorrectionControl(
       Band::LONGWAVE, insParams.getPerformLWLinearityCorrectionControl());
    theScienceDataProcessor->setPerformLinearityCorrectionControl(
       Band::MIDWAVE, insParams.getPerformMWLinearityCorrectionControl());
    theScienceDataProcessor->setPerformLinearityCorrectionControl(
       Band::SHORTWAVE, insParams.getPerformSWLinearityCorrectionControl());

    theScienceDataProcessor->setLinearityModeSelectControl(
       Band::LONGWAVE, insParams.
          getLinearityCorrectionControlOrigin(Band::LONGWAVE));
    theScienceDataProcessor->setLinearityModeSelectControl(
       Band::MIDWAVE, insParams.
          getLinearityCorrectionControlOrigin(Band::MIDWAVE));
    theScienceDataProcessor->setLinearityModeSelectControl(
       Band::SHORTWAVE, insParams.
          getLinearityCorrectionControlOrigin(Band::SHORTWAVE));

    theScienceDataProcessor->setEffGainSetting(
       Band::LONGWAVE, insParams.getGainSetting(Band::LONGWAVE));
    theScienceDataProcessor->setEffGainSetting(
       Band::MIDWAVE, insParams.getGainSetting(Band::MIDWAVE));
    theScienceDataProcessor->setEffGainSetting(
       Band::SHORTWAVE, insParams.getGainSetting(Band::SHORTWAVE));

    theScienceDataProcessor->setEffectiveGainMap(
       Band::LONGWAVE, insParams.getEffectiveGainMap(Band::LONGWAVE));
    theScienceDataProcessor->setEffectiveGainMap(
       Band::MIDWAVE, insParams.getEffectiveGainMap(Band::MIDWAVE));
    theScienceDataProcessor->setEffectiveGainMap(
       Band::SHORTWAVE, insParams.getEffectiveGainMap(Band::SHORTWAVE));

    theScienceDataProcessor->setLaserDiodeWavelengthOffsetMW(
       insParams.getDefaultLaserWavelengthOffsetMW());
    theScienceDataProcessor->setLaserDiodeWavelengthOffsetSW(
       insParams.getDefaultLaserWavelengthOffsetSW());

    theScienceDataProcessor->setFoldIndexOffset(
       Band::LONGWAVE, insParams.getFoldIndexOffset(Band::LONGWAVE));
    theScienceDataProcessor->setFoldIndexOffset(
       Band::MIDWAVE, insParams.getFoldIndexOffset(Band::MIDWAVE));
    theScienceDataProcessor->setFoldIndexOffset(
       Band::SHORTWAVE, insParams.getFoldIndexOffset(Band::SHORTWAVE));


    theScienceDataProcessor->calculateWavenumberBinSpacing(Band::LONGWAVE);
    theScienceDataProcessor->calculateWavenumberBinSpacing(Band::MIDWAVE);
    theScienceDataProcessor->calculateWavenumberBinSpacing(Band::SHORTWAVE);

    theScienceDataProcessor->setEngCalInstrumentCharacteristics(
       theScienceDataProcessor->getInstrumentCharacteristic());

    char logMessage[EVENT_LOG_LENGTH];

    sprintf(logMessage,"ATTN::4 min Engineering packet ID set to: 0x%X",
       theScienceDataProcessor->getEngineeringCalibrationRecordIdentifier());
    (void)EventLogEngine::append(logMessage);

    sprintf(logMessage,"ATTN::8 sec Science packet ID set to: 0x%X",
       theScienceDataProcessor->getScienceCalibrationRecordIdentifier());
    (void)EventLogEngine::append(logMessage);

    //snapshot desired sizes
    UInt32 correctionMatrixDimension[Band::TOTAL_BANDS];

    theScienceDataProcessor->setDecimationFactor(
       Band::LONGWAVE, insParams.getLwDecimationFactor() );
    correctionMatrixDimension[Band::LONGWAVE] =
       insParams.getLwDataPointsDecimatedInterferogram();
    theScienceDataProcessor->setDataPointsDecimatedInterferogram(
       Band::LONGWAVE, correctionMatrixDimension[Band::LONGWAVE]);
    theScienceDataProcessor->setDataPointsUndecimatedInterferogram(
       Band::LONGWAVE, insParams.getLwDataPointsUndecimatedInterferogram() );

    theScienceDataProcessor->setDecimationFactor(
       Band::MIDWAVE, insParams.getMwDecimationFactor() );
    correctionMatrixDimension[Band::MIDWAVE] =
       insParams.getMwDataPointsDecimatedInterferogram();
    theScienceDataProcessor->setDataPointsDecimatedInterferogram(
       Band::MIDWAVE, correctionMatrixDimension[Band::MIDWAVE]);
    theScienceDataProcessor->setDataPointsUndecimatedInterferogram(
       Band::MIDWAVE, insParams.getMwDataPointsUndecimatedInterferogram() );

    theScienceDataProcessor->setDecimationFactor(
       Band::SHORTWAVE, insParams.getSwDecimationFactor() );
    correctionMatrixDimension[Band::SHORTWAVE] =
       insParams.getSwDataPointsDecimatedInterferogram();
    theScienceDataProcessor->setDataPointsDecimatedInterferogram(
       Band::SHORTWAVE, correctionMatrixDimension[Band::SHORTWAVE]);
    theScienceDataProcessor->setDataPointsUndecimatedInterferogram(
       Band::SHORTWAVE, insParams.getSwDataPointsUndecimatedInterferogram() );

    theScienceDataProcessor->setDataPointsTruncatedInterferogram(
       Band::LONGWAVE, insParams.getDataPointsTruncatedInterferogram(Band::LONGWAVE));
    theScienceDataProcessor->setDataPointsTruncatedInterferogram(
       Band::MIDWAVE, insParams.getDataPointsTruncatedInterferogram(Band::MIDWAVE));
    theScienceDataProcessor->setDataPointsTruncatedInterferogram(
       Band::SHORTWAVE, insParams.getDataPointsTruncatedInterferogram(Band::SHORTWAVE));

    for (int c = 0; c < SURF_EMISS_COEFF_CMAX; ++c)
    {
        SurfaceEmissivityCoeffIdxEnum idx = (SurfaceEmissivityCoeffIdxEnum) c;
        theScienceDataProcessor->setSurfaceEmissivityCoeff(
                idx,
                insParams.getSurfaceEmissivityCoeff(idx));
    }

    theScienceDataProcessor->setEngCalInstrumentCharacteristics(
       theScienceDataProcessor->getInstrumentCharacteristic());

    //resize correction tables
    for(Int32 theFOV = 0; theFOV < MAX_FOV; theFOV++)
    {
        for(Int32 theBand = 0; theBand < Band::TOTAL_BANDS; theBand++)
        {
            theScienceDataProcessor->setTableSize(
               (SceneElement)theFOV, (Band::Wavelength)theBand,
                  correctionMatrixDimension[theBand]);
        }
    }

    //configure band specfic metrology laser wavelength
    Spectra::configureLaserWavelengthOffSet(
       Band::LONGWAVE, 1.0);   //no shift, other bands are relative to longwave
    Spectra::configureLaserWavelengthOffSet(
       Band::MIDWAVE, (1.0 + (
          insParams.getDefaultLaserWavelengthOffsetMW() * 0.000001)));
    Spectra::configureLaserWavelengthOffSet(
       Band::SHORTWAVE, (1.0 + (
          insParams.getDefaultLaserWavelengthOffsetSW() * 0.000001)));

    if(corParams.getLaserDiodeWavelengthOrigin() == LASER_CONFIG)
    {
        Spectra::configureLaserWavelength(
           insParams.getDefaultLaserWavelength());
        Spectra::buildResamplingWavelength(
           insParams.getNumberSamplesPerLaserWavelength());

        char logMessageText[EVENT_LOG_LENGTH];
        sprintf(logMessageText,"ATTN::Metrology Wavelength has been set to %f",
           Spectra::LASER_WAVELENGTH);
        (void)EventLogEngine::append(logMessageText);
    }
    //setting Test Environment Parameters in Sci Data Proc from
    //theTestEnvironmentParameterTab
    theScienceDataProcessor->setLWDsEffectiveEmissivity(
       testEnvParams.getDsEffectiveEmissivity(Band::LONGWAVE) );
    theScienceDataProcessor->setMWDsEffectiveEmissivity(
       testEnvParams.getDsEffectiveEmissivity(Band::MIDWAVE) );
    theScienceDataProcessor->setSWDsEffectiveEmissivity(
       testEnvParams.getDsEffectiveEmissivity(Band::SHORTWAVE) );
    theScienceDataProcessor->setScanBaffleTempOffset(
       testEnvParams.getScanBaffleTempOffset() );
    theScienceDataProcessor->setBeamsplitterTempBench(
       testEnvParams.getBeamsplitterTempBench() );
    theScienceDataProcessor->setBeamsplitterTempChamber(
       testEnvParams.getBeamsplitterTempChamber() );
    theScienceDataProcessor->setDsTempBench(
       testEnvParams.getDsTempBench() );
    theScienceDataProcessor->setDsTempChamber(
       testEnvParams.getDsTempChamber() );
    theScienceDataProcessor->setIctTempBench(
       testEnvParams.getIctTempBench() );
    theScienceDataProcessor->setIctTempChamber(
       testEnvParams.getIctTempChamber() );
    theScienceDataProcessor->setMeanDsEmissivityBench(
       testEnvParams.getMeanDsEmissivityBench() );
    theScienceDataProcessor->setMeanDsEmissivityChamber(
       testEnvParams.getMeanDsEmissivityChamber() );
    theScienceDataProcessor->setOmaTempBench(
       testEnvParams.getOmaTempBench() );
    theScienceDataProcessor->setOmaTempChamber(
       testEnvParams.getOmaTempChamber() );
    theScienceDataProcessor->setOmaEmissivity(
       Band::LONGWAVE, testEnvParams.getOmaFrameEmissivity(Band::LONGWAVE));
    theScienceDataProcessor->setOmaEmissivity(
       Band::MIDWAVE, testEnvParams.getOmaFrameEmissivity(Band::MIDWAVE));
    theScienceDataProcessor->setOmaEmissivity(
       Band::SHORTWAVE, testEnvParams.getOmaFrameEmissivity(Band::SHORTWAVE));
    theScienceDataProcessor->setScanBaffleTempBench(
       testEnvParams.getScanBaffleTempBench() );
    theScienceDataProcessor->setScanBaffleTempChamber(
       testEnvParams.getScanBaffleTempChamber() );
    theScienceDataProcessor->setScanBaffleEmissivity(
       Band::LONGWAVE, testEnvParams.getScanBaffleEmissivity(Band::LONGWAVE));
    theScienceDataProcessor->setScanBaffleEmissivity(
       Band::MIDWAVE, testEnvParams.getScanBaffleEmissivity(Band::MIDWAVE));
    theScienceDataProcessor->setScanBaffleEmissivity(
       Band::SHORTWAVE, testEnvParams.getScanBaffleEmissivity(Band::SHORTWAVE));
    theScienceDataProcessor->setDurationOfOrbit(
       testEnvParams.getDurationOfOrbit() );
    if(calParams.getInstrumentTemperatureOrigin() == INSTRUMENT_CONFIG)
    {
       theScienceDataProcessor->buildBaffleTemperatureOffset(
          testEnvParams.getScanBaffleTempOffset(),
          testEnvParams.getDurationOfOrbit());
    }
    theScienceDataProcessor->setColdBeamsplitterViewFactor(
       testEnvParams.getColdBeamsplitterViewFactor());
    theScienceDataProcessor->setWarmBeamsplitterViewFactor(
       testEnvParams.getWarmBeamsplitterViewFactor());
    theScienceDataProcessor->setScanBaffleViewFactor(
       testEnvParams.getScanBaffleViewFactor());
    theScienceDataProcessor->setOmaViewFactor(
       testEnvParams.getOmaFrameViewFactor());
    theScienceDataProcessor->setIctBaffleViewFactor(
       testEnvParams.getIctBaffleViewFactor());
    theScienceDataProcessor->setIctBaffleEmissivity(
       Band::LONGWAVE, testEnvParams.getIctBaffleEmissivity(Band::LONGWAVE));
    theScienceDataProcessor->setIctBaffleEmissivity(
       Band::MIDWAVE, testEnvParams.getIctBaffleEmissivity(Band::MIDWAVE));
    theScienceDataProcessor->setIctBaffleEmissivity(
       Band::SHORTWAVE, testEnvParams.getIctBaffleEmissivity(Band::SHORTWAVE));
    theScienceDataProcessor->setSpaceViewFactor(
       testEnvParams.getEarthTargetViewFactor());
    theScienceDataProcessor->setEarthTargetTempBench(
       testEnvParams.getEarthTargetTempBench());
    theScienceDataProcessor->setEarthTargetTempChamber(
       testEnvParams.getEarthTargetTempChamber());
    theScienceDataProcessor->setEarthTargetEmissivity(
       Band::LONGWAVE, testEnvParams.getEarthTargetEmissivity(Band::LONGWAVE));
    theScienceDataProcessor->setEarthTargetEmissivity(
       Band::MIDWAVE, testEnvParams.getEarthTargetEmissivity(Band::MIDWAVE));
    theScienceDataProcessor->setEarthTargetEmissivity(
       Band::SHORTWAVE,
       testEnvParams.getEarthTargetEmissivity(Band::SHORTWAVE));
    theScienceDataProcessor->setEarthTargetOverride(
       testEnvParams.getOverrideEarthTargetTemp());

    //setting Algorithm Parameters in Sci Data Proc from
    //theAlgorithmParameterTab
    theScienceDataProcessor->setIctPrt1Bias( algParams.getIctPrt1Bias() );
    theScienceDataProcessor->setIctPrt2Bias( algParams.getIctPrt2Bias() );
    theScienceDataProcessor->setFceParamDefaultDetectorBand(
       algParams.getFceParamDefaultDetectorBand() );
    theScienceDataProcessor->setFceParamDefaultDetectorFOV(
       algParams.getFceParamDefaultDetectorFOV() );
    theScienceDataProcessor->setPolarizationCorrectionFitOrder(
       algParams.getPolarizationCorrectionFitOrder() );
    theScienceDataProcessor->setMaximumFractionRejections(
       algParams.getMaximumFractionRejections() );
    theScienceDataProcessor->setComputedWavelengthRejectionThreshold(
       algParams.getComputedWavelengthRejectionThreshold() );
    theScienceDataProcessor->setLaserWavelengthDriftTolerance(
       algParams.getLaserWavelengthDriftTolerance() );
    theScienceDataProcessor->setNumberOpdOverscanSamples(
       algParams.getNumberOpdOverscanSamples() );
    Spectra::configureNumberOPDOverscanSamples(
       algParams.getNumberOpdOverscanSamples() );
    theScienceDataProcessor->setPostCalibrationA1(
       Band::LONGWAVE, algParams.getPostCalibrationLwA1() );
    theScienceDataProcessor->setPostCalibrationA1(
       Band::MIDWAVE, algParams.getPostCalibrationMwA1() );
    theScienceDataProcessor->setPostCalibrationA1(
       Band::SHORTWAVE, algParams.getPostCalibrationSwA1() );
    theScienceDataProcessor->setPostCalibrationA2(
       Band::SHORTWAVE, algParams.getPostCalibrationSwA2() );
    theScienceDataProcessor->setPostCalibrationA2(
       Band::MIDWAVE, algParams.getPostCalibrationMwA2() );
    theScienceDataProcessor->setPostCalibrationA2(
       Band::LONGWAVE, algParams.getPostCalibrationLwA2() );
    theScienceDataProcessor->setPostCalibrationA3(
       Band::LONGWAVE, algParams.getPostCalibrationLwA3() );
    theScienceDataProcessor->setPostCalibrationA3(
       Band::MIDWAVE, algParams.getPostCalibrationMwA3() );
    theScienceDataProcessor->setPostCalibrationA3(
       Band::SHORTWAVE, algParams.getPostCalibrationSwA3() );
    theScienceDataProcessor->setPostCalibrationA4(
       Band::SHORTWAVE, algParams.getPostCalibrationSwA4() );
    theScienceDataProcessor->setPostCalibrationA4(
       Band::MIDWAVE, algParams.getPostCalibrationMwA4() );
    theScienceDataProcessor->setPostCalibrationA4(
       Band::LONGWAVE, algParams.getPostCalibrationLwA4() );
    theScienceDataProcessor->setPostCalibrationK (
       Band::LONGWAVE, algParams.getPostCalibrationLwK()  );
    theScienceDataProcessor->setPostCalibrationK0(
       Band::LONGWAVE, algParams.getPostCalibrationLwK0() );
    theScienceDataProcessor->setPostCalibrationK1(
       Band::LONGWAVE, algParams.getPostCalibrationLwK1() );
    theScienceDataProcessor->setPostCalibrationK (
       Band::MIDWAVE, algParams.getPostCalibrationMwK()  );
    theScienceDataProcessor->setPostCalibrationK0(
       Band::MIDWAVE, algParams.getPostCalibrationMwK0() );
    theScienceDataProcessor->setPostCalibrationK1(
       Band::MIDWAVE, algParams.getPostCalibrationMwK1() );
    theScienceDataProcessor->setPostCalibrationK (
       Band::SHORTWAVE, algParams.getPostCalibrationSwK()  );
    theScienceDataProcessor->setPostCalibrationK0(
       Band::SHORTWAVE, algParams.getPostCalibrationSwK0() );
    theScienceDataProcessor->setPostCalibrationK1(
       Band::SHORTWAVE, algParams.getPostCalibrationSwK1() );

    // Add Spectra::CAL_THRESHOLD to its arguments
    theScienceDataProcessor->setFceParamAmpThreshRejectLimit(
       Spectra::CAL_THRESHOLD, Band::LONGWAVE,
       algParams.getFceParamLwAmpThreshRejectLimit() );
    theScienceDataProcessor->setFceParamAmpThreshRejectLimit(
       Spectra::CAL_THRESHOLD, Band::MIDWAVE,
       algParams.getFceParamMwAmpThreshRejectLimit() );
    theScienceDataProcessor->setFceParamAmpThreshRejectLimit(
       Spectra::CAL_THRESHOLD, Band::SHORTWAVE,
       algParams.getFceParamSwAmpThreshRejectLimit() );
    // Add Spectra::REF_THRESHOLD
    theScienceDataProcessor->setFceParamAmpThreshRejectLimit(
            Spectra::REF_THRESHOLD,
            Band::LONGWAVE,
            algParams.getFceParamLwRefAmpThreshRejectLimit() );
    theScienceDataProcessor->setFceParamAmpThreshRejectLimit(
            Spectra::REF_THRESHOLD,
            Band::MIDWAVE,
            algParams.getFceParamMwRefAmpThreshRejectLimit() );
    theScienceDataProcessor->setFceParamAmpThreshRejectLimit(
            Spectra::REF_THRESHOLD,
            Band::SHORTWAVE,
            algParams.getFceParamSwRefAmpThreshRejectLimit() );

    theScienceDataProcessor->setFceParamDimensionThresholdLimit(
       Band::LONGWAVE, algParams.getFceParamLwDimensionThresholdLimit() );
    theScienceDataProcessor->setFceParamDimensionThresholdLimit(
       Band::MIDWAVE, algParams.getFceParamMwDimensionThresholdLimit() );
    theScienceDataProcessor->setFceParamDimensionThresholdLimit(
       Band::SHORTWAVE, algParams.getFceParamSwDimensionThresholdLimit() );
    theScienceDataProcessor->setFceParamFractionalFceThresholdLimit(
       Band::LONGWAVE, algParams.getFceParamLwFractionalFceThresholdLimit() );
    theScienceDataProcessor->setFceParamFractionalFceThresholdLimit(
       Band::MIDWAVE, algParams.getFceParamMwFractionalFceThresholdLimit() );
    theScienceDataProcessor->setFceParamFractionalFceThresholdLimit(
       Band::SHORTWAVE, algParams.getFceParamSwFractionalFceThresholdLimit() );
    theScienceDataProcessor->setFceParamGoodLinearFittingThreshLimit(
       Band::MIDWAVE, algParams.getFceParamMwGoodLinearFittingThreshLimit() );
    theScienceDataProcessor->setFceParamGoodLinearFittingThreshLimit(
       Band::LONGWAVE, algParams.getFceParamLwGoodLinearFittingThreshLimit() );
    theScienceDataProcessor->setFceParamGoodLinearFittingThreshLimit(
       Band::SHORTWAVE, algParams.getFceParamSwGoodLinearFittingThreshLimit() );
    theScienceDataProcessor->setFceParamMaxIndex(
       Band::MIDWAVE, algParams.getFceParamMwMaxIndex() );
    theScienceDataProcessor->setFceParamMaxIndex(
       Band::LONGWAVE, algParams.getFceParamLwMaxIndex() );
    theScienceDataProcessor->setFceParamMaxIndex(
       Band::SHORTWAVE, algParams.getFceParamSwMaxIndex() );
    theScienceDataProcessor->setFceParamMaxFceThreshLimit(
       Band::LONGWAVE, algParams.getFceParamLwMaxFceThreshLimit() );
    theScienceDataProcessor->setFceParamMaxFceThreshLimit(
       Band::MIDWAVE, algParams.getFceParamMwMaxFceThreshLimit() );
    theScienceDataProcessor->setFceParamMaxFceThreshLimit(
       Band::SHORTWAVE, algParams.getFceParamSwMaxFceThreshLimit() );
    theScienceDataProcessor->setFceParamMinIndex(
       Band::LONGWAVE, algParams.getFceParamLwMinIndex() );
    theScienceDataProcessor->setFceParamMinIndex(
       Band::MIDWAVE, algParams.getFceParamMwMinIndex() );
    theScienceDataProcessor->setFceParamMinIndex(
       Band::SHORTWAVE, algParams.getFceParamSwMinIndex() );
    // FCE update
    theScienceDataProcessor->setFCE_esImagThreshold(
       algParams.getEsImagThreshold() );
    theScienceDataProcessor->setFCE_esFceNumErrorThreshold(
       algParams.getEsFceNumErrorThreshold() );
    theScienceDataProcessor->setStartWavenumber(
       algParams.getStartWavenumber() );
    theScienceDataProcessor->setEndWavenumber(
       algParams.getEndWavenumber() );

    // polarization parameters
    theScienceDataProcessor->setPolarizationAlpha(
       Band::LONGWAVE, corParams.getPolarizationAlpha(Band::LONGWAVE) );
    theScienceDataProcessor->setPolarizationAlpha(
       Band::MIDWAVE, corParams.getPolarizationAlpha(Band::MIDWAVE) );
    theScienceDataProcessor->setPolarizationAlpha(
       Band::SHORTWAVE, corParams.getPolarizationAlpha(Band::SHORTWAVE) );
    for(Int32 theFOV = 0; theFOV < MAX_FOV; theFOV++)
    {
       theScienceDataProcessor->setDsPolarizationDelta(
         (SceneElement)theFOV, corParams.getDsPolarizationDelta((SceneElement)theFOV));
       theScienceDataProcessor->setIctPolarizationDelta(
         (SceneElement)theFOV, corParams.getIctPolarizationDelta((SceneElement)theFOV));
       theScienceDataProcessor->setESPolarizationDelta(
         (SceneElement)theFOV, corParams.getEsPolarizationDelta((SceneElement)theFOV));
       theScienceDataProcessor->setLWPolarizationReflTrans(
         (SceneElement)theFOV,
         corParams.getPolarizationReflTrans(Band::LONGWAVE, (SceneElement)theFOV));
       theScienceDataProcessor->setMWPolarizationReflTrans(
         (SceneElement)theFOV,
         corParams.getPolarizationReflTrans(Band::MIDWAVE, (SceneElement)theFOV));
       theScienceDataProcessor->setSWPolarizationReflTrans(
         (SceneElement)theFOV,
         corParams.getPolarizationReflTrans(Band::SHORTWAVE, (SceneElement)theFOV));
    }

    //setting the General Parameters in Sci Data Proc from
    //theGeneralParameterTab
    theScienceDataProcessor->setInstrumentLocation(
       (InstrumentLocation)genParams.getInstrumentLocation());
    theScienceDataProcessor->
       declareWindowSize(calParams.getReferenceWindowSize(),
                     calParams.getTargetTempWindowSize(),
                     calParams.getCalibratedWindowSize());
    theScienceDataProcessor->
       setSpectralCalibrationMethod(genParams.getSpectralCalMethod());
    theScienceDataProcessor->enableNEdN(genParams.getRequestedNEdN());

    theScienceDataProcessor->setAppShiftFactorFlag(
            insParams.getAppShiftFactorFlag());
    for (UInt16 p = 0; p < NUM_SHIFT_FACTOR_PTS; ++p)
    {
        theScienceDataProcessor->setShiftFactor(p,
                insParams.getShiftFactor(p));
    }

    return true;
}

void checkRadianceValues(UInt32 scan, CrisSdrAlgDataType* algDataPtr)
{

    for(UInt32 ev = 0; ev < EV_FOR_PER_SCAN; ev++)
    {
        for(UInt32 fov = 0; fov < CRIS_MAX_FOV; fov++)
        {
            //Check Longwave Values, do not include fill values.
            //Float values run from the test value to 1.0 less than it.
            //Also check for NAN number.
            if((algDataPtr->sdrPtr->QF3[scan][ev][fov][Band::LONGWAVE] &
                    SDR_QUALITY) == SDR_QUALITY_GOOD)
            {
                for(UInt32 point = 0; point < LWNUMBEROFPOINTS; point++)
                {
                    Float32 chkESRealLW = algDataPtr->sdrPtr->ESRealLW[scan][ev][fov][point];
                    if(((chkESRealLW > FLOAT32_FILL_TEST) && (chkESRealLW < LW_MIN_RADIANCE)) ||
                        (chkESRealLW < (FLOAT32_FILL_TEST - 1.0)) ||
                        (std::isnan(chkESRealLW)))
                    {
                        //Set invalid data flag
                        algDataPtr->sdrPtr->
                           QF3[scan][ev][fov][Band::LONGWAVE] |=
                              SDR_QUALITY_INVALID;

                        char buffer[EVENT_LOG_LENGTH];
                        sprintf(buffer,
                          "Longwave Radiance Value Outside of Range. (%f < %d)",
                           chkESRealLW,
                           LW_MIN_RADIANCE);
                        (void)EventLogEngine::append(buffer);
                        break;
                    }
                    else if(chkESRealLW > LW_MAX_RADIANCE)
                    {
                        //Set invalid data flag
                        algDataPtr->sdrPtr->
                           QF3[scan][ev][fov][Band::LONGWAVE] |=
                              SDR_QUALITY_INVALID;

                        char buffer[EVENT_LOG_LENGTH];
                        sprintf(buffer,
                          "Longwave Radiance Value Outside of Range. (%f > %d)",
                           algDataPtr->sdrPtr->ESRealLW[scan][ev][fov][point],
                           LW_MAX_RADIANCE);
                        (void)EventLogEngine::append(buffer);
                        break;
                    }
                }
            }
            //Check Midwave Values, do not include fill values
            //Float values run from the test value to 1.0 less than it.
            //Also check for NAN number.
            if((algDataPtr->sdrPtr->QF3[scan][ev][fov][Band::MIDWAVE] &
                    SDR_QUALITY) == SDR_QUALITY_GOOD)
            {
               for(UInt32 point = 0; point < MW_NUMBER_OF_POINTS; point++)
               {
                  Float32 chkESRealMW = algDataPtr->sdrPtr->ESRealMW[scan][ev][fov][point];
                  if(((chkESRealMW > FLOAT32_FILL_TEST) && (chkESRealMW < MW_MIN_RADIANCE)) ||
                      (chkESRealMW < (FLOAT32_FILL_TEST - 1.0)) ||
                      (std::isnan(chkESRealMW)))
                  {
                     //Set invalid data flag
                     algDataPtr->sdrPtr->QF3[scan][ev][fov][Band::MIDWAVE]
                        |= SDR_QUALITY_INVALID;

                     char buffer[EVENT_LOG_LENGTH];
                     sprintf(buffer,
                           "Midwave Radiance Value Outside of Range. (%f < %d)",
                           chkESRealMW,
                           MW_MIN_RADIANCE);
                     (void)EventLogEngine::append(buffer);
                     break;
                  }
                  else if(chkESRealMW > MW_MAX_RADIANCE)
                  {
                     //Set invalid data flag
                     algDataPtr->sdrPtr->QF3[scan][ev][fov][Band::MIDWAVE]
                        |= SDR_QUALITY_INVALID;
                     char buffer[EVENT_LOG_LENGTH];
                     sprintf(buffer,
                             "%s (%f > %d)",
                             "Midwave Radiance Value Outside of Range.",
                             chkESRealMW,
                             MW_MAX_RADIANCE);
                     (void)EventLogEngine::append(buffer);
                     break;
                  }
               }
            }
            //Check Shortwave Values, do not include fill values
            //Float values run from the test value to 1.0 less than it.
            //Also check for NAN number.
            if((algDataPtr->sdrPtr->QF3[scan][ev][fov][Band::SHORTWAVE] &
                    SDR_QUALITY) == SDR_QUALITY_GOOD)
            {
               for(UInt32 point = 0; point < SW_NUMBER_OF_POINTS; point++)
               {
                  Float32 chkESRealSW = algDataPtr->sdrPtr->ESRealSW[scan][ev][fov][point];
                  if(((chkESRealSW > FLOAT32_FILL_TEST) && (chkESRealSW < SW_MIN_RADIANCE)) ||
                     (chkESRealSW < (FLOAT32_FILL_TEST - 1.0)) ||
                     (std::isnan(chkESRealSW)))
                  {
                     //Set invalid data flag
                     algDataPtr->sdrPtr->
                        QF3[scan][ev][fov][Band::SHORTWAVE]
                        |= SDR_QUALITY_INVALID;

                     char buffer[EVENT_LOG_LENGTH];
                     sprintf(buffer,
                             "%s (%f < %d)",
                             "Shortwave Radiance Value Outside of Range.",
                             chkESRealSW,
                             SW_MIN_RADIANCE);
                     (void)EventLogEngine::append(buffer);
                     break;
                  }
                  else if(chkESRealSW > SW_MAX_RADIANCE)
                  {
                     //Set invalid data flag
                     algDataPtr->sdrPtr->
                        QF3[scan][ev][fov][Band::SHORTWAVE]
                        |= SDR_QUALITY_INVALID;

                     char buffer[EVENT_LOG_LENGTH];
                     sprintf(buffer,
                             "%s (%f > %d)",
                             "Shortwave Radiance Value Outside of Range.",
                             chkESRealSW,
                             SW_MAX_RADIANCE);
                     (void)EventLogEngine::append(buffer);
                     break;
                  }
               }
            }

            CrisSdrCoefficients* cfgPtr = algDataPtr->cfgParmsPtr;

            // DR4478 -- Check Longwave imaginary radiance values, do not
            // include fill values
            // Float values run from the test value to 1.0 less than it.
            if ((algDataPtr->sdrPtr->
               QF4[scan][ev][fov][Band::LONGWAVE] & IMAG_RAD_INVALID) !=
                  IMAG_RAD_INVALID)
            {
                // Set initial min/max so first iteration will set it to a
                // new value
                Float32 minRad = std::numeric_limits<Float32>::max();
                Float32 maxRad = std::numeric_limits<Float32>::min();

                for(UInt32 point = cfgPtr->lwImagRadCheckStart;
                    point < cfgPtr->lwImagRadCheckEnd; point++)
                {
                    if (algDataPtr->sdrPtr->
                       ESImaginaryLW[scan][ev][fov][point] > maxRad)
                    {
                        maxRad = algDataPtr->sdrPtr->
                           ESImaginaryLW[scan][ev][fov][point];
                    }
                    if (((algDataPtr->sdrPtr->
                           ESImaginaryLW[scan][ev][fov][point] >
                              FLOAT32_FILL_TEST) ||
                        (algDataPtr->sdrPtr->
                           ESImaginaryLW[scan][ev][fov][point] <
                              (FLOAT32_FILL_TEST - 1.0))) &&
                        (algDataPtr->sdrPtr->
                           ESImaginaryLW[scan][ev][fov][point] < minRad))
                    {
                        minRad = algDataPtr->sdrPtr->
                           ESImaginaryLW[scan][ev][fov][point];
                    }
                }

                if(maxRad > cfgPtr->lwImagRadUpperThresh ||
                   minRad < cfgPtr->lwImagRadLowerThresh)
                {
                    //Set invalid data flag
                    algDataPtr->sdrPtr->QF4[scan][ev][fov][Band::LONGWAVE] |=
                        IMAG_RAD_INVALID;

                    char buffer[EVENT_LOG_LENGTH];
                    sprintf(buffer,
                          "%s (%f  %f  %f  %f)",
                          "Longwave Imaginary Radiance Value Outside of Range.",
                           maxRad, cfgPtr->lwImagRadUpperThresh,
                           minRad, cfgPtr->lwImagRadLowerThresh);
                    (void)EventLogEngine::append(buffer);
                }
            }

            // DR4478 -- Check Midwave imaginary radiance values, do not include
            // fill values
            // Float values run from the test value to 1.0 less than it.
            if ((algDataPtr->sdrPtr->
               QF4[scan][ev][fov][Band::MIDWAVE] & IMAG_RAD_INVALID) !=
                  IMAG_RAD_INVALID)
            {
                // Set initial min/max so first iteration will set it to a new
                // value
                Float32 minRad = std::numeric_limits<Float32>::max();
                Float32 maxRad = std::numeric_limits<Float32>::min();
                for(UInt32 point = cfgPtr->mwImagRadCheckStart;
                    point < cfgPtr->mwImagRadCheckEnd; point++)
                {
                    if (algDataPtr->sdrPtr->
                       ESImaginaryMW[scan][ev][fov][point] > maxRad)
                    {
                        maxRad = algDataPtr->sdrPtr->
                           ESImaginaryMW[scan][ev][fov][point];
                    }
                    if (((algDataPtr->sdrPtr->
                       ESImaginaryMW[scan][ev][fov][point] >
                          FLOAT32_FILL_TEST) ||
                             (algDataPtr->sdrPtr->
                                ESImaginaryMW[scan][ev][fov][point] <
                                  (FLOAT32_FILL_TEST - 1.0))) &&
                         (algDataPtr->sdrPtr->
                            ESImaginaryMW[scan][ev][fov][point] < minRad))
                    {
                        minRad = algDataPtr->sdrPtr->
                           ESImaginaryMW[scan][ev][fov][point];
                    }
                }

                if(maxRad > cfgPtr->mwImagRadUpperThresh ||
                   minRad < cfgPtr->mwImagRadLowerThresh)
                {
                    //Set invalid data flag
                    algDataPtr->sdrPtr->QF4[scan][ev][fov][Band::MIDWAVE] |=
                        IMAG_RAD_INVALID;

                    char buffer[EVENT_LOG_LENGTH];
                    sprintf(buffer,
                           "%s (%f  %f  %f  %f)",
                           "Midwave Imaginary Radiance Value Outside of Range.",
                            maxRad, cfgPtr->mwImagRadUpperThresh,
                            minRad, cfgPtr->mwImagRadLowerThresh);
                    (void)EventLogEngine::append(buffer);
                }
            }

            // DR4478 -- Check Shortwave imaginary radiance values, do not
            // include fill values
            // Float values run from the test value to 1.0 less than it.
            if ((algDataPtr->sdrPtr->QF4[scan][ev][fov][Band::SHORTWAVE] &
                IMAG_RAD_INVALID) != IMAG_RAD_INVALID)
            {
                // Set initial min/max so first iteration will set it to a new
                // value
                Float32 minRad = std::numeric_limits<Float32>::max();
                Float32 maxRad = std::numeric_limits<Float32>::min();
                for(UInt32 point = cfgPtr->swImagRadCheckStart;
                    point < cfgPtr->swImagRadCheckEnd; point++)
                {
                    if (algDataPtr->sdrPtr->
                       ESImaginarySW[scan][ev][fov][point] > maxRad)
                    {
                        maxRad = algDataPtr->sdrPtr->
                           ESImaginarySW[scan][ev][fov][point];
                    }
                    if (((algDataPtr->sdrPtr->
                            ESImaginarySW[scan][ev][fov][point] >
                               FLOAT32_FILL_TEST) ||
                         (algDataPtr->sdrPtr->
                            ESImaginarySW[scan][ev][fov][point] <
                               (FLOAT32_FILL_TEST - 1.0))) &&
                         (algDataPtr->sdrPtr->
                            ESImaginarySW[scan][ev][fov][point] < minRad))
                    {
                        minRad = algDataPtr->sdrPtr->
                           ESImaginarySW[scan][ev][fov][point];
                    }
                }

                if(maxRad > cfgPtr->swImagRadUpperThresh ||
                   minRad < cfgPtr->swImagRadLowerThresh)
                {
                    //Set invalid data flag
                    algDataPtr->sdrPtr->QF4[scan][ev][fov][Band::SHORTWAVE] |=
                        IMAG_RAD_INVALID;

                    char buffer[EVENT_LOG_LENGTH];
                    sprintf(buffer,
                            "%s %s (%f  %f  %f  %f)",
                            "Shortwave Imaginary Radiance Value",
                            "Outside of Range.",
                            maxRad, cfgPtr->swImagRadUpperThresh,
                            minRad, cfgPtr->swImagRadLowerThresh);
                    (void)EventLogEngine::append(buffer);
                }
            }
        }
    }
}

// FCE update
void checkSDR_Status(
        UInt32 scan,
        CrisSdrAlgDataType* algDataPtr,
        std::auto_ptr<ScienceDataProcessor>& theScienceDataProcessor,
        AlgorithmParameter algParams)
{
    // Add Deep Space Symmetry for output
    for(UInt32 fov = 0; fov < CRIS_MAX_FOV; fov++)
    {
        for(UInt32 band = 0; band < CRIS_TOTAL_BANDS; band++)
        {
            Float64 DS_symmetry = 0.;

            // If either DS cal packet is missing, then one of the vectors
            // (forward or reverse) will be empty.  Check size and index
            // to avoid indexing outside the array (boost documentation
            // is not clear on the effect of indexing out-of-bounds).
            UInt32 forIgmSize = theScienceDataProcessor->
               get_forDS_reals(scan, fov, band).size();
            UInt32 revIgmSize = theScienceDataProcessor->
               get_revDS_reals(scan, fov, band).size();

            // FCE update
            UInt32 igmSize = (forIgmSize > revIgmSize) ?
               revIgmSize: forIgmSize;
            UInt32 num_opd_overscan_samples =
               algParams.getNumberOpdOverscanSamples();

            // make sure igmSize large than 2*num_opd_overscan_samples
            // to protect against the zero igmSize
            // under missing Deep Space packet situation
            if ( igmSize > 2 * num_opd_overscan_samples )
            {
            for(UInt32 i = num_opd_overscan_samples;
                 i < igmSize - num_opd_overscan_samples; ++i)
            {
               std::complex<Float64> tmp1 = std::complex<Float64>(0.0, 0.0);
               std::complex<Float64> tmp2 = std::complex<Float64>(0.0, 0.0);
               if (i < forIgmSize)
               {
                  tmp1 = std::complex <Float64>(
                     theScienceDataProcessor->get_forDS_reals(
                        scan, fov, band)[i],
                     theScienceDataProcessor->get_forDS_imags(
                        scan, fov, band)[i]);
               }

               if (i < revIgmSize)
               {
                  tmp2 = std::complex <Float64>(
                     theScienceDataProcessor->get_revDS_reals(
                        scan, fov, band)[i],
                     theScienceDataProcessor->get_revDS_imags(
                        scan, fov, band)[i]);
               }

               std::complex<Float64> tmpDS = tmp1 - tmp2;
               DS_symmetry += sqrt(tmpDS.real() * tmpDS.real() +
                                   tmpDS.imag() * tmpDS.imag());
            }

               // FCE update
               algDataPtr->sdrPtr->DS_Symmetry[scan][fov][band] = DS_symmetry /
                                    (igmSize  - 2 * num_opd_overscan_samples);
            }
            else
            {
               algDataPtr->sdrPtr->DS_Symmetry[scan][fov][band] = NA_FLOAT64_FILL;
            }
        }
    }

    if(algDataPtr->sdrHeapDataPtr->ICT_TemperatureSensor[scan][REVERSE] >= 0. &&
       algDataPtr->sdrHeapDataPtr->ICT_TemperatureSensor[scan][FORWARD] >= 0.)
    {
        algDataPtr->sdrPtr->ICT_TemperatureConsistency[scan] =
           algDataPtr->sdrHeapDataPtr->ICT_TemperatureSensor[scan][REVERSE] -
              algDataPtr->sdrHeapDataPtr->ICT_TemperatureSensor[scan][FORWARD];
    }


    for(UInt32 ev = 0; ev < EV_FOR_PER_SCAN; ev++)
    {
        for(UInt32 fov = 0; fov < CRIS_MAX_FOV; fov++)
        {
           for(UInt32 band = 0; band < CRIS_TOTAL_BANDS; band++)
           {
              if(((algDataPtr->sdrPtr->QF3[scan][ev][fov][band] &
                   FRINGE_COUNT_ERROR_CORRECT_FAIL) ==
                  FRINGE_COUNT_ERROR_CORRECT_FAIL) ||
                 (((algDataPtr->sdrPtr->QF1[scan] &
                    SUSPECT_NEON_CALIBRATION) == SUSPECT_NEON_CALIBRATION)
                 &&
                  ((algDataPtr->sdrPtr->QF1[scan] &
                     LAMBDA_MONITORED_QUALITY) == LAMBDA_MONITORED_QUALITY)))
              {
                algDataPtr->sdrPtr->QF3[scan][ev][fov][band] |=
                   INVALID_SPECTRAL_CALIBRATION;
              }
              else if(((algDataPtr->sdrPtr->QF4[scan][ev][fov][band] &
                       FRINGE_COUNT_ERROR_DETECTED) ==
                      FRINGE_COUNT_ERROR_DETECTED)
                      ||
                      (((algDataPtr->sdrPtr->QF1[scan] &
                         SUSPECT_NEON_CALIBRATION) ==
                        SUSPECT_NEON_CALIBRATION)
                       &&
                       (!((algDataPtr->sdrPtr->QF1[scan] &
                         LAMBDA_MONITORED_QUALITY) ==
                        LAMBDA_MONITORED_QUALITY)))
                       ||
                       (!((algDataPtr->sdrPtr->QF1[scan] &
                          SUSPECT_NEON_CALIBRATION) ==
                         SUSPECT_NEON_CALIBRATION)
                        &&
                        ((algDataPtr->sdrPtr->QF1[scan] &
                          LAMBDA_MONITORED_QUALITY) ==
                         LAMBDA_MONITORED_QUALITY)))
              {
                 algDataPtr->sdrPtr->QF3[scan][ev][fov][band] |=
                    DEGRADED_SPECTRAL_CALIBRATION;
              }
              else
              {
                 algDataPtr->sdrPtr->QF3[scan][ev][fov][band] |=
                    GOOD_SPECTRAL_CALIBRATION;
              }

              UInt16 tmp1 = algDataPtr->sdrPtr->
                 DS_WindowSize[scan][REVERSE][fov][band];
              UInt16 tmp2 = algDataPtr->sdrPtr->
                 DS_WindowSize[scan][FORWARD][fov][band];
              UInt16 DSWindowSize = (tmp1 < tmp2) ? tmp1 : tmp2;

              tmp1 = algDataPtr->sdrPtr->
                 ICT_WindowSize[scan][REVERSE][fov][band];
              tmp2 = algDataPtr->sdrPtr->
                 ICT_WindowSize[scan][FORWARD][fov][band];
              UInt16 ICTWindowSize = (tmp1 < tmp2) ? tmp1 : tmp2;

              // DR4680  replace CRIS_WINDOW_SIZE by CRIS_WINDOW_SIZE / 2
              if(DSWindowSize == 0 || ICTWindowSize == 0 ||
                 (algDataPtr->sdrPtr->
                    ESRdrImpulseNoise[scan][ev][fov][band] > UINT8_FILL_TEST))
              {
                 algDataPtr->sdrPtr->QF3[scan][ev][fov][band] |=
                    INVALID_RADIOMETRIC_CALIBRATION;
              }
              else if( DSWindowSize < CRIS_WINDOW_SIZE/2 ||
                       ICTWindowSize < CRIS_WINDOW_SIZE/2 ||
                       ((algDataPtr->sdrPtr->QF1[scan] &
                         EXCESS_THERMAL_DRIFT) == EXCESS_THERMAL_DRIFT) ||
                       ((algDataPtr->sdrPtr->QF1[scan] &
                         INVALID_INSTRUMENT_TEMPERATURES) ==
                        INVALID_INSTRUMENT_TEMPERATURES) ||
                       algDataPtr->sdrPtr->ICT_TemperatureStability[scan][REVERSE] >
                          algDataPtr->cfgParmsPtr->ictTempStabilityThreshold ||
                       algDataPtr->sdrPtr->ICT_TemperatureStability[scan][FORWARD] >
                          algDataPtr->cfgParmsPtr->ictTempStabilityThreshold ||
                       algDataPtr->sdrPtr->ICT_TemperatureConsistency[scan] >
                          algDataPtr->cfgParmsPtr->ictTempConsistencyThreshold ||
                       algDataPtr->sdrPtr->numberOfValidPRTTemps[scan][REVERSE] <
                          algDataPtr->cfgParmsPtr->numOfValidPRTTempThreshold ||
                       algDataPtr->sdrPtr->numberOfValidPRTTemps[scan][FORWARD] <
                          algDataPtr->cfgParmsPtr->numOfValidPRTTempThreshold ||
                       algDataPtr->sdrPtr->ESRdrImpulseNoise[scan][ev][fov][band] >
                          algDataPtr->cfgParmsPtr->impulseNoiseCountThreshold)
              {
                 algDataPtr->sdrPtr->QF3[scan][ev][fov][band] |=
                    DEGRADED_RADIOMETRIC_CALIBRATION;
              }
              else
              {
                 algDataPtr->sdrPtr->QF3[scan][ev][fov][band] |=
                    GOOD_RADIOMETRIC_CALIBRATION;
              }

              if(((algDataPtr->sdrPtr->QF4[scan][ev][fov][band] &
                   BIT_TRIM_FAILED) == BIT_TRIM_FAILED)  ||
                  ((algDataPtr->sdrPtr->QF4[scan][ev][fov][band] &
                    FRINGE_COUNT_ERROR_DETECTED) ==
                   FRINGE_COUNT_ERROR_DETECTED) ||
                  ((algDataPtr->sdrPtr->QF4[scan][ev][fov][band] &
                    INVALID_RDR_DATA) == INVALID_RDR_DATA) ||
                  ((algDataPtr->sdrPtr->QF3[scan][ev][fov][band] &
                    INVALID_RADIOMETRIC_CALIBRATION) ==
                   INVALID_RADIOMETRIC_CALIBRATION) ||
                  ((algDataPtr->sdrPtr->QF3[scan][ev][fov][band] &
                    INVALID_SPECTRAL_CALIBRATION) ==
                   INVALID_SPECTRAL_CALIBRATION) ||
                  ((algDataPtr->sdrPtr->QF4[scan][ev][fov][band] &
                    IMAG_RAD_INVALID) == IMAG_RAD_INVALID))
              {
                algDataPtr->sdrPtr->
                   QF3[scan][ev][fov][band] |= SDR_QUALITY_INVALID;
              }
              // Check if the pixel is missing.  The field being checked is
              // irrelevant--it could be any that are set to fill when the
              // spectra is missing.
              else if (algDataPtr->sdrPtr->
                 ESZPDAmplitude[scan][ev][fov][band] == MISS_INT16_FILL)
              {
                 algDataPtr->sdrPtr->
                    QF3[scan][ev][fov][band] |= SDR_QUALITY_INVALID;
              }
              else if(((algDataPtr->sdrPtr->QF3[scan][ev][fov][band] &
                        INVALID_GEOLOCATION) == INVALID_GEOLOCATION) ||
                      ((algDataPtr->sdrPtr->QF3[scan][ev][fov][band] &
                         DEGRADED_SPECTRAL_CALIBRATION) ==
                            DEGRADED_SPECTRAL_CALIBRATION) ||
                      ((algDataPtr->sdrPtr->QF3[scan][ev][fov][band] &
                         DEGRADED_RADIOMETRIC_CALIBRATION) ==
                            DEGRADED_RADIOMETRIC_CALIBRATION))
              {
                 //DR4661 -- add if-statement to avoid overall flag having
                 // value 3
                 if ((algDataPtr->sdrPtr->
                    QF3[scan][ev][fov][band] & SDR_QUALITY) !=
                       SDR_QUALITY_INVALID)
                 {
                    algDataPtr->sdrPtr->
                       QF3[scan][ev][fov][band] |= SDR_QUALITY_DEGRADED;
                 }
              }
              else
              {
                 algDataPtr->sdrPtr->QF3[scan][ev][fov][band] |=
                    SDR_QUALITY_GOOD;
              }
           }
        }
    }
}
